﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "cdacm_PrivateIncludes.h"

/**
 * @file    cdacm_RS232.h
 * @brief   RS232 Abstraction
 *
 * @details This file defines the classes for commmunicating with different supported
 *          USB-to-Serial devices.
 */

namespace nn {
namespace cdacm {
namespace driver {

enum FtdiBrequest
{
    FtdiBrequest_Reset = 0,
    FtdiBrequest_ModemCtrl,
    FtdiBrequest_FlowCtrl,
    FtdiBrequest_BaudRate,
    FtdiBrequest_Data,
    FtdiBrequest_EventChar = 6,
    FtdiBrequest_Latency = 9
};

const uint8_t ProlificUartStateIndex         = 8;
const uint8_t ProlificUartStateMsrMask       = 0x8b;
const uint8_t ProlificUartStateTransientMask = 0x74;
const uint8_t ProlificUartDcd                = 0x01;
const uint8_t ProlificUartDsr                = 0x02;
const uint8_t ProlificUartBreakError         = 0x04;
const uint8_t ProlificUartRing               = 0x08;
const uint8_t ProlificUartFrameError         = 0x10;
const uint8_t ProlificUartParityError        = 0x20;
const uint8_t ProlificUartOverrunError       = 0x40;
const uint8_t ProlificUartCts                = 0x80;

enum ProlificBrequest
{
    ProlificBrequest_ReadWrite = 0x01,
    ProlificBrequest_SetLine = 0x20,
    ProlificBrequest_GetLine,
    ProlificBrequest_SetControl
};


class RS232
{
public:
    RS232()
        : m_pInterface(nullptr)
    {
    }

    virtual ~RS232()
    {
    }

    virtual Result Initialize(nn::usb::HostInterface *pInterface, struct CdAcmParameters * parameters) NN_NOEXCEPT;
    virtual Result Finalize();

    virtual Result Read(size_t *pBytesTransferred, void *buffer, uint32_t length) NN_NOEXCEPT = 0;
    Result Write(size_t *pBytesTransferred, const void *buffer, uint32_t length) NN_NOEXCEPT;
    Result WriteAsync(uint32_t                 *pOutXferId,
                      nn::os::SystemEventType **pCompletionEventPtr,
                      const void               *buffer,
                      uint32_t                  length) NN_NOEXCEPT;
    Result GetWriteAsyncResult(uint32_t *pOutCount, nn::usb::XferReport *pOutReport, uint32_t count) NN_NOEXCEPT;

protected:
    Result ClearBulkIn() NN_NOEXCEPT;
    Result ClearBulkOut() NN_NOEXCEPT;

protected:
    const uint32_t               MaxXferSize = nn::usb::HsLimitMaxUrbTransferSize;

    nn::usb::HostInterface      *m_pInterface;
    nn::usb::HostEndpoint        m_BulkIn;
    nn::usb::HostEndpoint        m_BulkOut;

    nn::usb::InterfaceProfile    m_IfProfile;
};

class FtdiRS232 : public RS232
{
public:
    virtual ~FtdiRS232()
    {
    }
protected:
    virtual Result Initialize(nn::usb::HostInterface *pInterface, struct CdAcmParameters * parameters) NN_NOEXCEPT;
    virtual Result Finalize();
    virtual Result Read(size_t *pBytesTransferred, void *buffer, uint32_t length) NN_NOEXCEPT;
private:
    static Result FtdiEncodeBaudrate(uint32_t speed, uint16_t *pwValue, uint16_t *pwIndex) NN_NOEXCEPT;
};

class ProlificRS232 : public RS232
{
public:
    virtual ~ProlificRS232()
    {
    }
protected:
    nn::usb::HostEndpoint        m_IntIn;

    virtual Result Initialize(nn::usb::HostInterface *pInterface, struct CdAcmParameters * parameters) NN_NOEXCEPT;
    virtual Result Finalize() NN_NOEXCEPT;
    virtual Result Read(size_t *pBytesTransferred, void *pInBuffer, uint32_t length) NN_NOEXCEPT;
private:
    void EncodeConfig(uint8_t * pBuffer, CdAcmParameters * parameters) NN_NOEXCEPT;
    bool IsDirectBaudRateSupported(uint32_t baudRate) NN_NOEXCEPT;
    NN_USB_DMA_ALIGN uint8_t m_buffer[4096];
};

} // end of namespace driver
} // end of namespace cdacm
} // end of namespace nn
