﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdacm_RS232.h"

namespace nn {
namespace cdacm {

namespace driver {

const uint32_t FtdiReadOffset        = 2;
const uint32_t FtdiReadSleepTimeInMs = 1000;

Result
RS232::Initialize(nn::usb::HostInterface *pInterface,
                  CdAcmParameters * parameters) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_UNUSED(parameters);

    m_pInterface = pInterface;

    NN_CDACM_DO(m_pInterface->SetInterface(0));
    if (result.IsFailure())
    {
        goto Exit;
    }

    NN_CDACM_DO(m_pInterface->GetInterface(&m_IfProfile));
    if (result.IsFailure())
    {
        goto Exit;
    }

Exit:
    return result;
}

/*
 * Finalize
 *
 * ResultSuccess
 *   Object cleaned up.
 *
 */
Result
RS232::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    std::memset(&m_IfProfile, 0, sizeof(m_IfProfile));

    m_pInterface = nullptr;

    return result;
}

/*
 * Read
 *
 * ResultSuccess
 *   Data read successfully from Bulk IN endpoint.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultUsbTransactionError
 *   USB transaction failed.
 */
Result
FtdiRS232::Read(size_t *pBytesTransferred, void *buffer, uint32_t length) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    bool fCont = true;

    do
    {
        *pBytesTransferred = 0;
        NN_CDACM_DO(m_BulkIn.PostBuffer(pBytesTransferred, buffer, length));
        if (nn::usb::ResultStalled::Includes(result))
        {
            NN_CDACM_WARN("IN Endpoint Stalled!  Clearing stall and reissuing transfer\n");
            NN_CDACM_DO(ClearBulkIn());
            NN_CDACM_DO(m_BulkIn.PostBuffer(pBytesTransferred, buffer, length));
        }
        fCont = (result.IsSuccess() && *pBytesTransferred <= FtdiReadOffset);
        if (fCont)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FtdiReadSleepTimeInMs));
        }
    } while (fCont);

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        NN_CDACM_WARN("USB layer error %d:%d\n",
            result.GetModule(), result.GetDescription());
        result = ResultUsbTransactionError();
    }
    else
    {
        nn::cdacm::ShiftBuffer(buffer, *pBytesTransferred, FtdiReadOffset);
        *pBytesTransferred -= FtdiReadOffset;
    }

    return result;
}

/*
 * Write
 *
 * ResultSuccess
 *   Data sent to Bulk OUT endpoint successfully.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultUsbTransactionError
 *   USB transaction failed.
 */
Result
RS232::Write(size_t *pBytesTransferred, const void *buffer, uint32_t length) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_DO(m_BulkOut.PostBuffer(pBytesTransferred, const_cast<void*>(buffer), length));
    if (nn::usb::ResultStalled::Includes(result))
    {
        NN_CDACM_WARN("Write Stalled!  Clearing stall and reissuing write\n");
        NN_CDACM_DO(ClearBulkOut());
        NN_CDACM_DO(m_BulkOut.PostBuffer(pBytesTransferred, const_cast<void*>(buffer), length));
    }

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        NN_CDACM_WARN("Write Failed!  Interface not available\n");
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        NN_CDACM_WARN("Write Error %d:%d\n",
                      result.GetModule(), result.GetDescription());
        result = ResultUsbTransactionError();
    }

    return result;
}

/*
* WriteAsync
*
* ResultSuccess
*   Buffer submitted to be sent to Bulk OUT endpoint successfully.
*
*/
Result
RS232::WriteAsync(
    uint32_t                 *pOutXferId,
    nn::os::SystemEventType **pCompletionEventPtr,
    const void               *pOutBuffer,
    uint32_t                  length) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_DO(m_BulkOut.PostBufferAsync(pOutXferId, const_cast<void*>(pOutBuffer), length, NULL));

    if (result.IsSuccess())
    {
        *pCompletionEventPtr = m_BulkOut.GetCompletionEvent();
    }

    return result;
}

Result
RS232::GetWriteAsyncResult(
    uint32_t            *pOutCount,
    nn::usb::XferReport *pOutReport,
    uint32_t             count) NN_NOEXCEPT
{
    return m_BulkOut.GetXferReport(pOutCount, pOutReport, count);
}

/*
 * ClearBulkIn
 *
 * ResultSuccess
 *   Bulk IN endpoint cleared of its HALT condition.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultUsbTransactionError
 *   USB transaction failed.
 */
Result
RS232::ClearBulkIn() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_DO(m_BulkIn.ClearEndpointHalt());
    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        result = ResultUsbTransactionError();
    }

    return result;
}

/*
 * ClearBulkOut
 *
 * ResultSuccess
 *   Bulk OUT endpoint cleared of its HALT condition.
 *
 * ResultDeviceNotAvailable
 *   Device is detached, nothing we can do.
 *
 * ResultUsbTransactionError
 *   USB transaction failed.
 */
Result
RS232::ClearBulkOut() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_DO(m_BulkOut.ClearEndpointHalt());
    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        result = ResultRS232OutOfSync();
    }

    return result;
}

/*
 * Initialize
 *
 * ResultSuccess
 *   FTDI RS232 device initialized
 *
 * ResultInvalidInterface
 *   Bulk IN and OUT endpoints both not found.
 *
 * Other
 *   Vendor commands or initialization of bulk endponts failed.
 *
 */
Result
FtdiRS232::Initialize(nn::usb::HostInterface *pInterface, CdAcmParameters * parameters) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    size_t bytesTransferred = 0;
    uint16_t wValue = 0;
    uint16_t wIndex = 0;
    nn::usb::UsbEndpointDescriptor *pBulkInDesc = nullptr;
    nn::usb::UsbEndpointDescriptor *pBulkOutDesc = nullptr;

    result = RS232::Initialize(pInterface, parameters);
    if (result.IsFailure())
    {
        goto Exit;
    }

    for (auto& desc : m_IfProfile.epInDesc)
    {
        if (nn::usb::UsbEndpointIsBulk(&desc))
        {
            pBulkInDesc = &desc;
            break;
        }
    }

    for (auto& desc : m_IfProfile.epOutDesc)
    {
        if (nn::usb::UsbEndpointIsBulk(&desc))
        {
            pBulkOutDesc = &desc;
            break;
        }
    }

    if (pBulkInDesc == nullptr || pBulkOutDesc == nullptr)
    {
        NN_CDACM_WARN("One of the bulk endpoints is found to be null (IN: %p, OUT %p)\n",
            pBulkInDesc, pBulkOutDesc);
        result = ResultInvalidInterface();
        goto Exit;
    }

    NN_CDACM_DO(m_BulkIn.Initialize(m_pInterface, pBulkInDesc, 1, MaxXferSize));
    if (result.IsFailure())
    {
        goto Exit;
    }

    NN_CDACM_DO(m_BulkOut.Initialize(m_pInterface, pBulkOutDesc, 1, MaxXferSize));
    if (result.IsFailure())
    {
        m_BulkIn.Finalize();
        goto Exit;
    }

    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_Reset,
                                             0, 0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_Latency,
                                             parameters->latency,
                                             0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_FlowCtrl,
                                             parameters->flowCtrl,
                                             0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    NN_CDACM_DO(FtdiEncodeBaudrate(parameters->baudRate, &wValue, &wIndex));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_BaudRate,
                                             wValue,
                                             wIndex,
                                             0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_ModemCtrl,
                                             parameters->modemControl,
                                             0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_Data,
                                             parameters->dataCharacteristics,
                                             0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_FlowCtrl,
                                             parameters->flowCtrl,
                                             0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    bytesTransferred = 0;
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
                                             NULL,
                                             BmRequestType(Vendor, Device, HostToDevice),
                                             FtdiBrequest_EventChar,
                                             parameters->eventCharacter,
                                             0, 0));
    if (result.IsFailure())
    {
        goto Cleanup;
    }
    return ResultSuccess();

Cleanup:
    m_BulkIn.Finalize();
    m_BulkOut.Finalize();
Exit:
    return result;
} // NOLINT(impl/function_size)

 /*
 * Finalize
 *
 * ResultSuccess
 *   Object cleaned up.
 *
 * Other
 *   Finalizing bulk endpoints failed.
 */
Result
FtdiRS232::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_DO(RS232::Finalize());
    NN_CDACM_DO(m_BulkOut.Finalize());
    NN_CDACM_DO(m_BulkIn.Finalize());

    return result;
}

/*
 * FtdiEncodeBaudrate
 *
 * ResultSuccess
 *   Baud rate computed.
 *
 * ResultInvalidArgs
 *   Requested speed not reachable.
 */
Result
FtdiRS232::FtdiEncodeBaudrate(uint32_t speed,
    uint16_t *pwValue,
    uint16_t *pwIndex) NN_NOEXCEPT
{
    static const uint8_t encoded_fraction[8] = { 0, 3, 2, 4, 1, 5, 6, 7 };
    uint32_t clk, divisor, fastclk_flag, frac, hwspeed;

    clk = 3000000;
    fastclk_flag = 0;

    /*
    * Make sure the requested speed is reachable with the available clock
    * and a 14-bit divisor.
    */
    if (speed < (clk >> 14) || speed > clk)
    {
        return ResultInvalidArgs();
    }

    /*
    * Calculate the divisor, initially yielding a fixed point number with a
    * 4-bit (1/16ths) fraction, then round it to the nearest fraction the
    * hardware can handle.  When the integral part of the divisor is
    * greater than one, the fractional part is in 1/8ths of the base clock.
    * The FT8U232AM chips can handle only 0.125, 0.250, and 0.5 fractions.
    * Later chips can handle all 1/8th fractions.
    *
    * If the integral part of the divisor is 1, a special rule applies: the
    * fractional part can only be .0 or .5 (this is a limitation of the
    * hardware).  We handle this by truncating the fraction rather than
    * rounding, because this only applies to the two fastest speeds the
    * chip can achieve and rounding doesn't matter, either you've asked for
    * that exact speed or you've asked for something the chip can't do.
    *
    * For the FT8U232AM chips, use a roundoff table to adjust the result
    * to the nearest 1/8th fraction that is supported by the hardware,
    * leaving a fixed-point number with a 3-bit fraction which exactly
    * represents the math the hardware divider will do.  For later-series
    * chips that support all 8 fractional divisors, just round 16ths to
    * 8ths by adding 1 and dividing by 2.
    */
    divisor = (clk << 4) / speed;
    if ((divisor & 0xf) == 1)
    {
        divisor &= 0xfffffff8;
    }
    divisor += 1;  /* Rounds odd 16ths up to next 8th. */
    divisor >>= 1;

    /*
    * Ensure the resulting hardware speed will be within operational
    * tolerance (within 3% of nominal).
    */
    hwspeed = (clk << 3) / divisor;
    if ((hwspeed < (speed * 100) / 103) || (hwspeed >(speed * 100) / 97))
    {
        return ResultInvalidArgs();
    }

    /*
    * Re-pack the divisor into hardware format. The lower 14-bits hold the
    * integral part, while the upper bits specify the fraction by indexing
    * a table of fractions within the hardware which is laid out as:
    *    {0.0, 0.5, 0.25, 0.125, 0.325, 0.625, 0.725, 0.875}
    * The A-series chips only have the first four table entries; the
    * roundoff table logic above ensures that the fractional part for those
    * chips will be one of the first four values.
    *
    * When the divisor is 1 a special encoding applies:  1.0 is encoded as
    * 0.0, and 1.5 is encoded as 1.0.  The rounding logic above has already
    * ensured that the fraction is either .0 or .5 if the integral is 1.
    */
    frac = divisor & 0x07;
    divisor >>= 3;
    if (divisor == 1)
    {
        if (frac == 0)
        {
            divisor = 0;  /* 1.0 becomes 0.0 */
        }
        else
        {
            frac = 0;     /* 1.5 becomes 1.0 */
        }
    }
    divisor |= (encoded_fraction[frac] << 14) | fastclk_flag;

    *pwValue = (uint16_t)divisor;
    *pwIndex = (uint16_t)(divisor >> 16);

    return ResultSuccess();
}

/*
* Initialize
*
* ResultSuccess
*   Prolific RS232 dveice initialized
*
* ResultInvalidInterface
*   Bulk IN, OUT, and/or IN endpoints both not found.
*
* Other
*   Vendor commands or initialization of bulk endponts failed.
*
*/
Result
ProlificRS232::Initialize(nn::usb::HostInterface *pInterface, CdAcmParameters * parameters) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    nn::usb::UsbEndpointDescriptor *pBulkInDesc = nullptr;
    nn::usb::UsbEndpointDescriptor *pBulkOutDesc = nullptr;
    nn::usb::UsbEndpointDescriptor *pIntInDesc = nullptr;
    size_t bytesTransferred = 0;

    result = RS232::Initialize(pInterface, parameters);
    if (result.IsFailure())
    {
        goto Exit;
    }

    for (auto& desc : m_IfProfile.epInDesc)
    {
        if (nn::usb::UsbEndpointIsBulk(&desc))
        {
            pBulkInDesc = &desc;
            break;
        }
    }

    for (auto& desc : m_IfProfile.epOutDesc)
    {
        if (nn::usb::UsbEndpointIsBulk(&desc))
        {
            pBulkOutDesc = &desc;
            break;
        }
    }

    for (auto& desc : m_IfProfile.epInDesc)
    {
        if (nn::usb::UsbEndpointIsInt(&desc))
        {
            pIntInDesc = &desc;
            break;
        }
    }

    if (pBulkInDesc == nullptr || pBulkOutDesc == nullptr || pIntInDesc == nullptr)
    {
        NN_CDACM_WARN("One+ of the bulk endpoints is null (IN: %p, OUT %p, Int In: %p)\n",
            pBulkInDesc, pBulkOutDesc, pIntInDesc);
        result = ResultInvalidInterface();
        goto Exit;
    }

    NN_CDACM_DO(m_BulkIn.Initialize(m_pInterface, pBulkInDesc, 1, MaxXferSize));
    if (result.IsFailure())
    {
        goto Exit;
    }

    NN_CDACM_DO(m_BulkOut.Initialize(m_pInterface, pBulkOutDesc, 1, MaxXferSize));
    if (result.IsFailure())
    {
        m_BulkIn.Finalize();
        goto Exit;
    }

    NN_CDACM_DO(m_IntIn.Initialize(m_pInterface, pIntInDesc, 1, MaxXferSize));
    if (result.IsFailure())
    {
        m_BulkIn.Finalize();
        m_BulkOut.Finalize();
        goto Exit;
    }

    //
    // The following are a series of mostly undocumented commands
    // being issued to the Prolific part on FreeBSD.  The magic
    // numbers of 0x8484, 0x8383, 0x0404, 0x44 are all undocumented.
    //

    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Vendor, Device, DeviceToHost),
        ProlificBrequest_ReadWrite,
        0x8484, 0, 1));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        NULL,
        BmRequestType(Vendor, Device, HostToDevice),
        ProlificBrequest_ReadWrite,
        0x0404, 0, 0));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Vendor, Device, DeviceToHost),
        ProlificBrequest_ReadWrite,
        0x8484, 0, 1));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Vendor, Device, DeviceToHost),
        ProlificBrequest_ReadWrite,
        0x8383, 0, 1));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Vendor, Device, DeviceToHost),
        ProlificBrequest_ReadWrite,
        0x8484, 0, 1));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        NULL,
        BmRequestType(Vendor, Device, HostToDevice),
        ProlificBrequest_ReadWrite,
        0x0404, 1, 0));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Vendor, Device, DeviceToHost),
        ProlificBrequest_ReadWrite,
        0x8484, 0, 1));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Vendor, Device, DeviceToHost),
        ProlificBrequest_ReadWrite,
        0x8383, 0, 1));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        NULL,
        BmRequestType(Vendor, Device, HostToDevice),
        ProlificBrequest_ReadWrite,
        0, 1, 0));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        NULL,
        BmRequestType(Vendor, Device, HostToDevice),
        ProlificBrequest_ReadWrite,
        1, 0, 0));
    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        NULL,
        BmRequestType(Vendor, Device, HostToDevice),
        ProlificBrequest_ReadWrite,
        2, 0x44, 0));

    EncodeConfig(m_buffer, parameters);

    NN_CDACM_DO(m_pInterface->ControlRequest(&bytesTransferred,
        static_cast<void*>(m_buffer),
        BmRequestType(Class, Interface, HostToDevice),
        ProlificBrequest_SetLine,
        0, 0, 7));
    if (result.IsFailure())
    {
        goto Cleanup;
    }

    goto Exit;

Cleanup:
    m_BulkIn.Finalize();
    m_BulkOut.Finalize();
    m_IntIn.Finalize();
Exit:
    return result;
} // NOLINT(impl/function_size)


/*
* Read
*
* ResultSuccess
*   Data read successfully from Bulk IN endpoint.
*
* ResultDeviceNotAvailable
*   Device is detached, nothing we can do.
*
* ResultUsbTransactionError
*   USB transaction failed.
*/
Result
ProlificRS232::Read(size_t *pBytesTransferred, void *buffer, uint32_t length) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    *pBytesTransferred = 0;
    NN_CDACM_DO(m_BulkIn.PostBuffer(pBytesTransferred, buffer, length));
    if (nn::usb::ResultStalled::Includes(result))
    {
        NN_CDACM_WARN("IN Endpoint Stalled!  Clearing stall and reissuing transfer\n");
        NN_CDACM_DO(ClearBulkIn());
        NN_CDACM_DO(m_BulkIn.PostBuffer(pBytesTransferred, buffer, length));
    }

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultDeviceNotAvailable();
    }
    else if (result.IsFailure())
    {
        NN_CDACM_WARN("USB layer error %d:%d\n",
            result.GetModule(), result.GetDescription());
        result = ResultUsbTransactionError();
    }


    return result;
}

Result
ProlificRS232::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_DO(RS232::Finalize());
    NN_CDACM_DO(m_BulkOut.Finalize());
    NN_CDACM_DO(m_BulkIn.Finalize());
    NN_CDACM_DO(m_IntIn.Finalize());

    return result;
}

bool
ProlificRS232::IsDirectBaudRateSupported(uint32_t baudRate) NN_NOEXCEPT
{
    uint32_t directSupportedBaudRates[] = { 1200, 2400, 4800, 9600, 19200, 38400, 57600, 1152000 };

    for (unsigned int i = 0; i < sizeof(directSupportedBaudRates) / sizeof(uint32_t); i++)
    {
        if (baudRate == directSupportedBaudRates[i])
        {
            return true;
        }
    }
    return false;
}

void
ProlificRS232::EncodeConfig(uint8_t * pBuffer, CdAcmParameters * parameters) NN_NOEXCEPT
{

    if (IsDirectBaudRateSupported(parameters->baudRate))
    {
        pBuffer[0] = parameters->baudRate  & 0x000000ffUL;
        pBuffer[1] = (parameters->baudRate & 0x0000ff00UL) >> 8;
        pBuffer[2] = (parameters->baudRate & 0x00ff0000UL) >> 16;
        pBuffer[3] = (parameters->baudRate & 0xff000000UL) >> 24;
    }
    else
    {
        unsigned int baseline, mantissa, exponent;
        /*
        *   baudrate = 12M * 32 / (mantissa * 4^exponent)
        * where
        *   mantissa = buf[8:0]
        *   exponent = buf[11:9]
        */
        baseline = 12000000 * 32;
        mantissa = baseline / parameters->baudRate;
        if (mantissa == 0)
            mantissa = 1;    /* Avoid dividing by zero if baud > 32*12M. */
        exponent = 0;
        while (mantissa >= 512) {
            if (exponent < 7) {
                mantissa >>= 2; /* divide by 4 */
                exponent++;
            }
            else {
                /* Exponent is maxed. Trim mantissa and leave. */
                mantissa = 511;
                break;
            }
        }

        pBuffer[3] = 0x80;
        pBuffer[2] = 0;
        pBuffer[1] = exponent << 1 | mantissa >> 8;
        pBuffer[0] = mantissa & 0xff;
    }

    // Number of bits
    pBuffer[6] = parameters->dataCharacteristics & 0xFF;

    // Stop bits
    switch ((parameters->dataCharacteristics & 0x700) >> 8)
    {
    case 0:          pBuffer[4] = 0; break;
    case 1:          pBuffer[4] = 1; break;
    default: case 2: pBuffer[4] = 2; break;
    }

    switch ((parameters->dataCharacteristics & 0x3800) >> 11)
    {
    default: case 0: pBuffer[5] = 0; break; // no parity
    case 1:          pBuffer[5] = 1; break; // odd parity
    case 2:          pBuffer[5] = 2; break; // even parity
    case 3:          pBuffer[5] = 3; break; // mark parity
    case 4:          pBuffer[5] = 4; break; // space parity
    }

}

} // end of namespace driver
} // end of namespace cdacm
} // end of namespace nn
