﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    cdacm_device.h
 * @brief   Device Abstraction
 *
 * @details This file defines the device class for commmunicating with different supported
 *          USB-to-Serial devices.
 */

#include "cdacm_PrivateIncludes.h"
#include "cdacm_RS232.h"

namespace nn {
namespace cdacm {
namespace driver {

class Device
{
private:
    union RS232Impl {
        FtdiRS232 Ftdi;
        ProlificRS232 Prolific;
    };

    typedef ::nn::util::TypedStorage<RS232Impl, sizeof(RS232Impl), NN_ALIGNOF(RS232Impl)> RS232Storage;
public:
    Device();
    ~Device();

    Result Initialize(nn::usb::Host    *pUsbHost,
        nn::usb::InterfaceQueryOutput  *pProfile,
        nn::os::SystemEventType       **pDetachEventPtr) NN_NOEXCEPT;

    Result Finalize() NN_NOEXCEPT;

    Result Detach() NN_NOEXCEPT;

    Result Read(size_t *pBytesXferred, void *buffer, uint32_t length) NN_NOEXCEPT;
    Result Write(size_t *pBytesXferred, const void *buffer, uint32_t length) NN_NOEXCEPT;
    Result WriteAsync(uint32_t                 *pOutXferId,
                      nn::os::SystemEventType **pCompletionEventPtr,
                      const void               *pOutBuffer,
                      uint32_t                  length) NN_NOEXCEPT;
    Result GetWriteAsyncResult(uint32_t            *pOutCount,
                               nn::usb::XferReport *pOutReport,
                               uint32_t             count) NN_NOEXCEPT;

    void       SetHandle(UnitHandle handle) NN_NOEXCEPT;
    UnitHandle GetHandle() const NN_NOEXCEPT;
    void       GetProfile(UnitProfile *pOutProfile) const NN_NOEXCEPT;

    Result InitRS232(CdAcmParameters *parameters) NN_NOEXCEPT;

    enum DeviceState GetDeviceState() NN_NOEXCEPT;

private:

    nn::usb::Host                         *m_pUsbHost;
    nn::usb::HostInterface                 m_UsbInterface;
    nn::usb::InterfaceQueryOutput          m_Profile;

    enum DeviceState                       m_DeviceState;
    RS232                                 *m_pRS232;

    nn::os::Mutex                          m_Mutex;

    UnitHandle                             m_Handle;

    RS232Storage                           m_RS232Memory;
};

} // driver
} // cdacm
} // nn
