﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>

#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator
#include <nn/btm/btm_IBtm.sfdl.h>
#include <nn/btm/btm_ServiceName.h>

namespace nn { namespace btm {

namespace {
bool isInitialized = false;
nn::sf::SharedPointer<IBtmSystem> g_BtmSystem;
nn::sf::SharedPointer<IBtmSystemCore> g_BtmSystemCore;
struct  BtmSystemByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BtmSystemByHipcTag> BtmSystemAllocator;
nn::os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
}   // namespace

// BtmSystemAllocator を静的コンストラクタで初期化するためのヘルパー
class BtmSystemAllocatorInitializer
{
public:

    BtmSystemAllocatorInitializer() NN_NOEXCEPT
    {
        BtmSystemAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BtmSystemAllocatorInitializer;


void InitializeBtmSystemInterface() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_BtmSystem);

    nn::sf::SharedPointer<IBtmSystem> ret;
    auto result = nn::sf::CreateHipcProxyByName<IBtmSystem, BtmSystemAllocator::Policy>(&ret, BtmSystemServiceName);
    NN_ABORT_UNLESS(result.IsSuccess());

    //[Todo]ポインタを直に渡してしまっても良さそう
    g_BtmSystem = ret;
    NN_UNUSED(result);
}

void FinalizeBtmSystemInterface() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_BtmSystem);
    g_BtmSystem = nullptr;
}

void EnterCriticalSection() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);

    if(!isInitialized)
    {
        nn::sf::SharedPointer<nn::btm::IBtmSystemCore> core;

        InitializeBtmSystemInterface();
        g_BtmSystem->GetCoreImpl(&core);
        NN_SDK_ASSERT(core);
        FinalizeBtmSystemInterface();

        g_BtmSystemCore = core;
        isInitialized = true;
    }
}

void ExitCriticalSection() NN_NOEXCEPT
{
    UnlockMutex(&g_Mutex);
}

}}// namespace nn::btm

namespace nn { namespace btm { namespace system {

void AcquireRadioEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    EnterCriticalSection();
    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmSystemCore->AcquireRadioEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
    ExitCriticalSection();
}

void AcquireGamepadPairingEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    EnterCriticalSection();
    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmSystemCore->AcquireGamepadPairingEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
    ExitCriticalSection();
}

bool IsGamepadPairingStarted() NN_NOEXCEPT
{
    EnterCriticalSection();
    bool isGamepadPairingStarted;
    isGamepadPairingStarted = g_BtmSystemCore->IsGamepadPairingStartedImpl();
    ExitCriticalSection();
    return isGamepadPairingStarted;
}

Result StartGamepadPairing() NN_NOEXCEPT
{
    EnterCriticalSection();
    auto result = g_BtmSystemCore->StartGamepadPairingImpl();
    ExitCriticalSection();
    return result;
}

void CancelGamepadPairing() NN_NOEXCEPT
{
    EnterCriticalSection();
    g_BtmSystemCore->CancelGamepadPairingImpl();
    ExitCriticalSection();
}

void ClearGamepadPairingDatabase() NN_NOEXCEPT
{
    EnterCriticalSection();
    g_BtmSystemCore->ClearGamepadPairingDatabaseImpl();
    ExitCriticalSection();
}

int GetPairedGamepadCount() NN_NOEXCEPT
{
    uint8_t pairedGamepadCount;
    EnterCriticalSection();
    g_BtmSystemCore->GetPairedGamepadCountImpl(&pairedGamepadCount);
    ExitCriticalSection();
    return pairedGamepadCount;
}

void EnableRadio(bool enable) NN_NOEXCEPT
{
    EnterCriticalSection();
    if(enable)
    {
        g_BtmSystemCore->EnableRadioImpl();
    }
    else
    {
        g_BtmSystemCore->DisableRadioImpl();
    }
    ExitCriticalSection();
}

bool IsRadioEnabled() NN_NOEXCEPT
{
    bool isRadioOn;
    EnterCriticalSection();
    g_BtmSystemCore->GetRadioOnOffImpl(&isRadioOn);
    ExitCriticalSection();
    return isRadioOn;
}
}}} // namespace nn::btm::system
