﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/bpc/bpc_Result.h>
#include <nn/bpc/bpc_RtcTypes.h>
#include <nn/bpc/bpc_WakeupConfigTypes.h>
#include <nn/bpc/driver/bpc_Rtc.h>
#include "detail/bpc_ExternalRtc.h"
#include "detail/bpc_PowerReason.h"
#include "detail/bpc_WakeupTimer.h"

namespace nn { namespace bpc { namespace driver {

nn::Result GetRtcTime(int64_t *pOutRtcTimeInSeconds) NN_NOEXCEPT
{
    return detail::GetRtcTime(pOutRtcTimeInSeconds);
}

nn::Result SetRtcTime(int64_t rtcTimeInSeconds) NN_NOEXCEPT
{
    // TORIAEZU: デバイスの有効範囲を超えた数字でもラップアラウンドするだけで許容
    // NN_SDK_ASSERT_RANGE(rtcTimeInSeconds, 0, MaxRtcTimeInSeconds);
    return detail::SetRtcTime(rtcTimeInSeconds);
}

nn::Result GetRtcResetDetected(bool* pOutRtcResetDetected) NN_NOEXCEPT
{
    return detail::GetRtcResetDetected(pOutRtcResetDetected);
}

nn::Result ClearRtcResetDetected() NN_NOEXCEPT
{
    return detail::ClearRtcResetDetected();
}

nn::Result CreateWakeupTimer(int *pOutTimerHandle, nn::TimeSpan interval, WakeupTimerType type) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutTimerHandle);
    NN_SDK_ASSERT_RANGE(interval.GetSeconds(), MinWakeupIntervalInSeconds, MaxRtcTimeInSeconds);
    NN_SDK_ASSERT_NOT_EQUAL(type, WakeupTimerType_None);

    int64_t currentTime;
    NN_RESULT_DO(GetRtcTime(&currentTime));

    auto* pTimer = detail::AllocateWakeupTimer();
    NN_RESULT_THROW_UNLESS(pTimer != nullptr, ResultMaxTimers());

    pTimer->SetTargetRtcTime(currentTime + interval.GetSeconds());
    pTimer->SetType(type);
    auto& arbiter = detail::GetWakeupTimerArbiter();
    NN_ABORT_UNLESS_RESULT_SUCCESS(arbiter.Register(pTimer));
    *pOutTimerHandle = pTimer->GetHandle();
    NN_RESULT_SUCCESS;
}

nn::Result CancelWakeupTimer(int timerHandle) NN_NOEXCEPT
{
    auto* pTimer = detail::GetWakeupTimer(timerHandle);
    NN_RESULT_THROW_UNLESS(pTimer != nullptr, ResultNotRegistered());
    auto& arbiter = detail::GetWakeupTimerArbiter();
    (void)arbiter.Unregister(pTimer);
    detail::FreeWakeupTimer(pTimer);
    NN_RESULT_SUCCESS;
}

nn::Result EnableWakeupTimerOnDevice(bool *pOutIsEnabled, int* pOutTimerHandle, WakeupTimerType* pOutTimerType) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutIsEnabled);
    auto& arbiter = detail::GetWakeupTimerArbiter();
    const detail::WakeupTimer* pEnabledTimer;
    NN_RESULT_DO(arbiter.UpdateDevice(pOutIsEnabled, &pEnabledTimer));
    if ( *pOutIsEnabled && pEnabledTimer )
    {
        if ( pOutTimerHandle )
        {
            *pOutTimerHandle = pEnabledTimer->GetHandle();
        }
        if ( pOutTimerType )
        {
            *pOutTimerType = pEnabledTimer->GetType();
        }
    }
    NN_RESULT_SUCCESS;
}

WakeupTimerType GetLastEnabledWakeupTimerType() NN_NOEXCEPT
{
    auto& arbiter = detail::GetWakeupTimerArbiter();
    return arbiter.GetLastEnabledWakeupTimerType();
}

nn::Result CleanAllWakeupTimers(WakeupTimerType type) NN_NOEXCEPT
{
    auto& arbiter = detail::GetWakeupTimerArbiter();
    return arbiter.CleanAllWakeupTimers(type);
}

nn::Result SetUpRtcResetOnShutdown() NN_NOEXCEPT
{
    detail::SetUpRtcResetOnShutdown();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::bpc::driver
