﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bpc/driver/bpc.h>
#include "detail/bpc_BoardPowerController.h"
#include "detail/bpc_ExternalRtc.h"
#include "detail/bpc_IrqHandler.h"
#include "detail/bpc_PowerReason.h"
#include "detail/bpc_SleepButtonCounter.h"

namespace nn { namespace bpc { namespace driver {

void Initialize(bool recoveryFromRtcReset) NN_NOEXCEPT
{
    detail::InitializePowerReason();
    detail::InitializeExternalRtc(recoveryFromRtcReset);
    detail::InitializeSleepButtonCounter();
    detail::InitializeIrqHandler(detail::GetSleepButtonCounterMessageQueue());
    detail::InitializeBoardPowerController();
}

void Finalize() NN_NOEXCEPT
{
    detail::FinalizeBoardPowerController();
    detail::FinalizeIrqHandler();
    detail::FinalizeSleepButtonCounter();
}

void ShutdownSystem() NN_NOEXCEPT
{
    detail::ShutdownSystem();
}

void SuspendIrq() NN_NOEXCEPT
{
    // 起床直後にスレッドが回り、 Clear on read な割り込み要因レジスタを
    // nne::max7762x::irq::processInterrupt が読んで消費してしまうのを防ぐため、
    // ハンドラスレッドが割り込みイベントを待機しないステートに変えておく
    // ただし、RTC は WakeReason で扱わないため、生かしておく
    detail::SuspendIrqHandler();

    // 起床用の割り込みイベントを設定
    detail::ConfigWakeupEvents();
}

void ResumeIrq() NN_NOEXCEPT
{
    // 割り込み要因レジスタの読み出し
    detail::UpdateWakeupReason();

    // ハンドラスレッドで割り込みイベント待ちを再開
    detail::ResumeIrqHandler();

    // 電源ボタンのハンドリングステートを明示的に更新
    detail::StartSleepButtonStateUpdate();
}

void RebootSystem() NN_NOEXCEPT
{
    detail::RebootSystem();
}

nn::Result GetWakeupReason(WakeupReason* pOutWakeupReason) NN_NOEXCEPT
{
    return detail::GetWakeupReason(pOutWakeupReason);
}

nn::Result GetShutdownReason(ShutdownReason* pOutShutdownReason) NN_NOEXCEPT
{
    return detail::GetShutdownReason(pOutShutdownReason);
}

nn::Result GetAcOk(bool* pOutAcOk) NN_NOEXCEPT
{
    return detail::GetAcOk(pOutAcOk);
}

void GetSleepButtonState(SleepButtonState* pOutState) NN_NOEXCEPT
{
    detail::GetSleepButtonState(pOutState);
}

nn::Result GetPowerButtonEventPtr(nn::os::SystemEventType** pOutEventPtr, EventTarget target) NN_NOEXCEPT
{
    detail::GetPowerButtonEventPtr(pOutEventPtr, target);

    NN_RESULT_SUCCESS;
}

nn::Result GetBoardPowerControlEventPtr(nn::os::SystemEventType** pOutEventPtr, BoardPowerControlEventTarget target) NN_NOEXCEPT
{
    detail::GetBoardPowerControlEventPtr(pOutEventPtr, target);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::bpc::driver
