﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/bgtc/bgtc_Types.h>

namespace nn{ namespace bgtc{

class ClientBroker;
class ClientBroker
{
    NN_DISALLOW_COPY( ClientBroker );
    NN_DISALLOW_MOVE( ClientBroker );

public:

    class Entry : public nn::util::IntrusiveListBaseNode<Entry>
    {
        friend class ClientBroker;
    public:
        NN_IMPLICIT Entry(os::SystemEvent& eventTrigger, os::SystemEvent& eventAlarm);

    protected:
        void ScheduleTask(Interval interval);
        void SchedulePeriodicTask(Interval intervalFirst, Interval intervalPeriodic);
        void UnscheduleTask();
        Interval GetScheduledTaskInterval() const;
        void SignalScheduleEventIfElapsed(Time timeBase);

        void SetTaskProcessing(bool bTaskProcessing);

        bool IsTaskProcessing() const
        {
            return m_bTaskProcessing;
        }

        bool IsTaskScheduled() const
        {
            return m_TimeAlarm != 0;
        }

        Time GetScheduledTime() const
        {
            return m_TimeAlarm;
        }

        bool IsPeriodic() const
        {
            return m_bPeriodic;
        }

        void SetName(const char* pName, size_t length);

        const char* GetName() const
        {
            return m_ClientName;
        }

    private:
        Time m_TimeAlarm;
        Interval m_PeriodicInterval;
        os::SystemEvent& m_EventTrigger;
        os::SystemEvent& m_EventAlarm;
        char m_ClientName[8];
        bool m_bTaskProcessing;
        bool m_bAlarmActive;
        bool m_bPeriodic;
    };
    typedef nn::util::IntrusiveListBaseNodeTraits<Entry> EntryTraits;
    typedef nn::util::IntrusiveList<Entry, EntryTraits> EntryList;

    ClientBroker();
    ~ClientBroker();

    void Register(Entry& entry);
    void Unregister(Entry& entry);

    void ScheduleTask(Entry& entry, Interval interval);
    void SchedulePeriodicTask(Entry& entry, Interval intervalFirst, Interval intervalPeriodic);
    void UnscheduleTask(Entry& entry);
    void SetTaskProcessing(Entry& entry, bool bTaskProcessing);

    Interval GetScheduledTaskInterval(const Entry& entry) const;
    bool IsTaskScheduled(const Entry& entry) const;
    void SetClientName(Entry& entry, const char* pName, size_t length);

    bool GetNextScheduledTime(Time* pOut) const;
    bool GetNextScheduledTimeAsInterval(Interval* pOut) const;
    bool IsProcessingClientExists() const;

    void ResetTaskProcessingState();

    void DispatchScheduleEvent();
    void SetEnableScheduleEvent(bool bDispatchable);

    os::TimerEvent& GetNearestScheduleEvent()
    {
        return m_TimerEvent;
    }

    void ShowSchedule() const;

protected:
    void SignalScheduleEventIfElapsed();
    void OnScheduleChanged();

private:
    EntryList m_listEntry;
    mutable os::Mutex m_Mutex;
    os::TimerEvent m_TimerEvent;
    bool m_bDispatchable;
};

}}


