﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/service/bcat_Server.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <nn/bcat/detail/ipc/bcat_ServiceConfig.h>
#include <nn/bcat/detail/ipc/bcat_ServiceName.h>
#include <nn/bcat/detail/service/bcat_Capability.h>
#include <nn/bcat/detail/service/bcat_Service.h>
#include <nn/bcat/detail/service/core/bcat_BackgroundWorkerThread.h>
#include <nn/bcat/detail/service/core/bcat_PushNotificationLogManager.h>
#include <nn/bcat/detail/service/core/bcat_TaskManager.h>
#include <nn/bcat/detail/service/util/bcat_Account.h>
#include <nn/os.h>
#include <nn/npns.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/crypto/crypto_Md5Generator.h>

namespace nn { namespace bcat { namespace service {

namespace
{
    const int ParallelExecutionCountMax = 2;

    nn::os::ThreadType g_IpcThreads[ParallelExecutionCountMax] = {};
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_IpcThreadStacks[ParallelExecutionCountMax][24 * 1024];

    struct BcatServiceHipcSimpleAllInOneServerManagerOption
    {
        // これ以上のサイズのデータは、[BufferTransferMode(BufferTransferMode.MapAlias)] でマップ転送に切り替えること。
        static const size_t PointerTransferBufferSize = 1024;

        static const bool CanDeferInvokeRequest = false;

        static const int SubDomainCountMax = 64;
        static const int ObjectInSubDomainCountMax = 16;
    };

    class BcatServiceServerManager :
        public nn::sf::HipcSimpleAllInOneServerManager<64, detail::ipc::ServiceCount,
            BcatServiceHipcSimpleAllInOneServerManagerOption>
    {
    };

    BcatServiceServerManager* g_ServerManager = nullptr;
    std::aligned_storage<sizeof (BcatServiceServerManager)>::type g_ServerManagerStorage;
}

namespace
{
    void RegisterService(int index, const char* serviceName, Bit32 capability) NN_NOEXCEPT
    {
        detail::service::InitializeServiceCreator(index, serviceName, capability);

        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager->RegisterObjectForPort(detail::service::GetServiceCreator(index),
            detail::ipc::SessionCountMax, serviceName));
    }

    void IpcDispatcher(void*) NN_NOEXCEPT
    {
        NN_DETAIL_BCAT_HANDLING_DATA_CORRUPTION_SCOPED_DECLARE();

        g_ServerManager->LoopAuto();
    }
}

void StartServer() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] Start service...\n");

    g_ServerManager = new (&g_ServerManagerStorage) BcatServiceServerManager;

    // bcat:u
    RegisterService(0, detail::ipc::ServiceNameForGeneric,
        detail::service::Capability::Flag_Generic);

    // bcat:s
    RegisterService(1, detail::ipc::ServiceNameForSystemUser,
        detail::service::Capability::Flag_System);

    // bcat:m
    RegisterService(2, detail::ipc::ServiceNameForManager,
        detail::service::Capability::Flag_System | detail::service::Capability::Flag_Management);

    // bcat:a
    RegisterService(3, detail::ipc::ServiceNameForAdministrator,
        detail::service::Capability::Flag_All);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::TaskManager::GetInstance().Load());

    g_ServerManager->Start();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_IpcThreads[i],
            IpcDispatcher, nullptr, g_IpcThreadStacks[i], sizeof (g_IpcThreadStacks[i]), NN_SYSTEM_THREAD_PRIORITY(bcat, IpcServer)));

        nn::os::SetThreadNamePointer(&g_IpcThreads[i], NN_SYSTEM_THREAD_NAME(bcat, IpcServer));
        nn::os::StartThread(&g_IpcThreads[i]);
    }

    detail::service::core::BackgroundWorkerThread::Start();

    NN_DETAIL_BCAT_INFO("[bcat] Start service... done!\n");
}

void StopServer() NN_NOEXCEPT
{
    g_ServerManager->RequestStop();

    for (int i = 0; i < ParallelExecutionCountMax; i++)
    {
        nn::os::DestroyThread(&g_IpcThreads[i]);
    }

    g_ServerManager->~BcatServiceServerManager();
    g_ServerManager = nullptr;

    detail::service::FinalizeServiceCreators();

    detail::service::core::BackgroundWorkerThread::Stop();

    NN_DETAIL_BCAT_INFO("[bcat] End service.\n");
}

bool GetHalfAwakeScheduleInterval(nn::bgtc::Interval* outSeconds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSeconds);

    nn::TimeSpan interval;

    if (!detail::service::core::BackgroundWorkerThread::GetNextScheduleInterval(&interval))
    {
        return false;
    }

    // 分解能が秒になるので、+1 しておく。
    *outSeconds = static_cast<nn::bgtc::Interval>(interval.GetSeconds() + 1);

    return true;
}

void NotifySystemHalfAwaked() NN_NOEXCEPT
{
    detail::service::core::BackgroundWorkerThread::NotifySystemRunnableStateChanged(true);
}

void NotifySystemFullAwaked() NN_NOEXCEPT
{
    detail::service::core::TaskManager::GetInstance().NotifySleepAwaked();
    detail::service::core::BackgroundWorkerThread::NotifySystemRunnableStateChanged(true);
}

void NotifySystemSleepReady(bool isFromFullAwake) NN_NOEXCEPT
{
    NN_UNUSED(isFromFullAwake);

    detail::service::core::BackgroundWorkerThread::NotifySystemRunnableStateChanged(false);
}

void NotifySystemShutdownReady() NN_NOEXCEPT
{
    detail::service::core::BackgroundWorkerThread::NotifySystemRunnableStateChanged(false);
}

void WaitForSleepReady() NN_NOEXCEPT
{
}

void WaitForShutdownReady() NN_NOEXCEPT
{
}

bool NotifyNotificationReceived(const NotificationPayload& payload) NN_NOEXCEPT
{
    char formatType[8];
    size_t size;

    if (!payload.GetValue(&size, "type", formatType, sizeof (formatType)))
    {
        return false;
    }
    if (nn::util::Strncmp(formatType, "data", sizeof (formatType)) != 0)
    {
        return false;
    }

    detail::TopicId topicId;

    if (!payload.GetValue(&size, "topic_id", topicId.value, sizeof (topicId.value)))
    {
        NN_DETAIL_BCAT_INFO("[bcat] 'topic_id' is not included in the payload.\n");
        return false;
    }

    // TODO: TopicId の検証。

    char revision[64] = {};
    size_t revisionSize = 0;

    if (!payload.GetValue(&revisionSize, "revision", revision, sizeof (revision)))
    {
        NN_DETAIL_BCAT_INFO("[bcat] 'revision' is not included in the payload.\n");
        return false;
    }

    int64_t waitRange = 0;

    payload.GetValue(&waitRange, "wait_range");

    if (waitRange < 0 || waitRange > INT16_MAX)
    {
        NN_DETAIL_BCAT_INFO("[bcat] The range of wait time is invalid.\n");
        return false;
    }

    int16_t waitTime = static_cast<int16_t>(detail::service::util::GetRandom(0, waitRange + 1));

    Bit8 hash[16];
    nn::crypto::GenerateMd5Hash(hash, 16, revision, revisionSize);

    detail::RevisionHash revisionHash;
    std::memcpy(revisionHash.value, hash, sizeof (revisionHash.value));

    NN_DETAIL_BCAT_INFO("[bcat] The notification was received. (%s, %s, 0-%lld)\n", topicId.value, revision, waitRange);

    detail::service::core::TaskManager::GetInstance().NotifyNotificationReceived(topicId, revisionHash, waitTime);

    detail::service::core::PushNotificationLogManager::GetInstance().Notify(topicId);

    return true;
}

void NotifyNpnsStateChanged() NN_NOEXCEPT
{
    nn::npns::State state = nn::npns::GetState();

    if (state == nn::npns::State_Connected || state == nn::npns::State_ConnectedOnHalfAwake)
    {
        detail::service::core::BackgroundWorkerThread::NotifyNpnsConnected();
    }
}

void NotifyNetworkServiceAccountAvailabilityChanged() NN_NOEXCEPT
{
    if (detail::service::util::Account::IsNetworkServiceAccountAvailable())
    {
        detail::service::core::TaskManager::GetInstance().NotifyNintendoAccountLinked();
    }
}

void NotifyNetworkConnected() NN_NOEXCEPT
{
    detail::service::core::TaskManager::GetInstance().NotifyNetworkConnected();
}

}}}
