﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/service/bcat_NotificationPayload.h>
#include <nn/util/util_StringUtil.h>
#include <cstdlib>
#include <cstring>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace bcat { namespace service {

namespace
{
    bool IsSpace(char c) NN_NOEXCEPT
    {
        return (c == ' ' || c == '\t' || c == '\r' || c == '\n');
    }
    bool IsKeyChar(char c) NN_NOEXCEPT
    {
        return ((c >= '0' && c <= '9') ||(c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || c == '.' || c == '_' || c == '-');
    }
    bool IsStringValueChar(char c) NN_NOEXCEPT
    {
        return ((c >= 0x20 && c <= 0x7E) && c != '\"' && c != '\\');
    }
    bool IsIntegerValueChar(char c) NN_NOEXCEPT
    {
        return (c >= '0' && c <= '9');
    }
}

NotificationPayload::NotificationPayload(const char* payload, size_t size) NN_NOEXCEPT :
    m_Payload(payload),
    m_Size(size),
    m_IsValid(false)
{
    NN_SDK_REQUIRES_NOT_NULL(payload);

    m_IsValid = Verify();
}

bool NotificationPayload::GetValue(size_t* outSize, const char* key, char* buffer, size_t size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size> 0);

    const char* value = GetValuePointer(key);

    if (!value || value[0] != '"')
    {
        return false;
    }

    size_t actualSize = 1;

    while (value[actualSize] != '"')
    {
        actualSize++;
    }

    if (size < actualSize)
    {
        return false;
    }

    nn::util::Strlcpy(buffer, &value[1], static_cast<int>(actualSize));
    *outSize = actualSize;

    return true;
}

bool NotificationPayload::GetValue(int64_t* outValue, const char* key) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(key);

    const char* value = GetValuePointer(key);

    if (!value || !((value[0] >= '0' && value[0] <= '9') || value[0] == '-'))
    {
        return false;
    }

    *outValue = std::strtoll(value, nullptr, 10);

    return true;
}

bool NotificationPayload::IsValid() const NN_NOEXCEPT
{
    return m_IsValid;
}

bool NotificationPayload::Verify() const NN_NOEXCEPT
{
    // 必ず {} で囲まれていること。
    if (m_Size == 0 || m_Payload[0] != '{' || m_Payload[m_Size - 1] != '}')
    {
        return false;
    }

    size_t c = 1;

    // 終端。
    while (IsSpace(m_Payload[c]))
    {
        c++;
    }
    if (m_Payload[c] == '}')
    {
        return true;
    }

    while (NN_STATIC_CONDITION(true))
    {
        // キーの妥当性検査を行う。
        while (IsSpace(m_Payload[c]))
        {
            c++;
        }
        if (m_Payload[c++] != '"')
        {
            return false;
        }
        while (IsKeyChar(m_Payload[c]))
        {
            c++;
        }
        if (m_Payload[c++] != '"')
        {
            return false;
        }

        // 区切り文字の検査。
        while (IsSpace(m_Payload[c]))
        {
            c++;
        }
        if (m_Payload[c++] != ':')
        {
            return false;
        }

        // 値の妥当性検証。（文字列値、または、整数値）
        while (IsSpace(m_Payload[c]))
        {
            c++;
        }
        if (!(m_Payload[c] == '"' || (m_Payload[c] >= '0' && m_Payload[c] <= '9') || m_Payload[c] == '-'))
        {
            return false;
        }
        if (m_Payload[c++] == '"')
        {
            while (IsStringValueChar(m_Payload[c]))
            {
                c++;
            }
            if (m_Payload[c++] != '"')
            {
                return false;
            }
        }
        else
        {
            while (IsIntegerValueChar(m_Payload[c]))
            {
                c++;
            }
        }

        // 終端、または、区切り文字。
        while (IsSpace(m_Payload[c]))
        {
            c++;
        }
        if (m_Payload[c] == '}')
        {
            break;
        }
        if (m_Payload[c++] != ',')
        {
            return false;
        }
    }

    return (c == m_Size - 1);
}

const char* NotificationPayload::GetValuePointer(const char* key) const NN_NOEXCEPT
{
    if (!m_IsValid)
    {
        return nullptr;
    }

    size_t keyLength = static_cast<size_t>(nn::util::Strnlen(key, static_cast<int>(m_Size)));

    if (m_Size < keyLength + 2)
    {
        return nullptr;
    }

    size_t searchLength = m_Size - keyLength - 2;

    for (size_t i = 0; i < searchLength; i++)
    {
        if (m_Payload[i] == '"' && m_Payload[i + 1 + keyLength] == '"' &&
            nn::util::Strncmp(&m_Payload[i + 1], key, static_cast<int>(keyLength)) == 0)
        {
            size_t c = i + 2 + keyLength;

            while (IsSpace(m_Payload[c]))
            {
                c++;
            }
            if (m_Payload[c++] != ':')
            {
                continue;
            }
            while (IsSpace(m_Payload[c]))
            {
                c++;
            }

            return &m_Payload[c];
        }
    }

    return nullptr;
}

}}}
