﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/service/bcat_ArchiveDecoder.h>
#include <nn/bcat/service/bcat_ArchiveDecryptionKeyGenerator.h>
#include <nn/bcat/detail/service/util/bcat_Decryptor.h>
#include <nn/nn_StaticAssert.h>
#include <nn/crypto/detail/crypto_Clear.h>

namespace nn { namespace bcat { namespace service {

ArchiveDecoder::ArchiveDecoder() NN_NOEXCEPT :
    m_Destructor(nullptr),
    m_DecodeFunction(nullptr),
    m_IsInitialized(false)
{
    NN_STATIC_ASSERT(sizeof (m_DecryptorStorage) >= sizeof (detail::service::util::AesCtrDecryptor128));
    NN_STATIC_ASSERT(sizeof (m_DecryptorStorage) >= sizeof (detail::service::util::AesCtrDecryptor192));
    NN_STATIC_ASSERT(sizeof (m_DecryptorStorage) >= sizeof (detail::service::util::AesCtrDecryptor256));
}

ArchiveDecoder::~ArchiveDecoder() NN_NOEXCEPT
{
    if (m_Destructor)
    {
        m_Destructor(this);
    }
}

void ArchiveDecoder::Initialize(const ArchiveHeader& header, const ArchiveDecryptionKey& key, size_t blockIndex) NN_NOEXCEPT
{
    Bit8 iv[ArchiveBlockSize];

    if (header.encryptionType != EncryptionType_NoEncrypted)
    {
        CalculateIv(iv, header.iv, blockIndex);
    }

    switch (header.encryptionType)
    {
    case EncryptionType_NoEncrypted:
        break;
    case EncryptionType_AesCtr128:
        {
            new (&m_DecryptorStorage) detail::service::util::AesCtrDecryptor128;

            reinterpret_cast<detail::service::util::AesCtrDecryptor128*>(&m_DecryptorStorage)->Initialize(key.value, 16, iv, ArchiveBlockSize);

            m_Destructor = [](ArchiveDecoder* pThis)
            {
                reinterpret_cast<detail::service::util::AesCtrDecryptor128*>(&pThis->m_DecryptorStorage)->~AesCtrDecryptor();
            };
            m_DecodeFunction = [](ArchiveDecoder* pThis, void* dest, const void* src, size_t size)
            {
                reinterpret_cast<detail::service::util::AesCtrDecryptor128*>(&pThis->m_DecryptorStorage)->Update(dest, size, src, size);
            };
        }
        break;
    case EncryptionType_AesCtr192:
        {
            new (&m_DecryptorStorage) detail::service::util::AesCtrDecryptor192;

            reinterpret_cast<detail::service::util::AesCtrDecryptor192*>(&m_DecryptorStorage)->Initialize(key.value, 24, iv, ArchiveBlockSize);

            m_Destructor = [](ArchiveDecoder* pThis)
            {
                reinterpret_cast<detail::service::util::AesCtrDecryptor192*>(&pThis->m_DecryptorStorage)->~AesCtrDecryptor();
            };
            m_DecodeFunction = [](ArchiveDecoder* pThis, void* dest, const void* src, size_t size)
            {
                reinterpret_cast<detail::service::util::AesCtrDecryptor192*>(&pThis->m_DecryptorStorage)->Update(dest, size, src, size);
            };
        }
        break;
    case EncryptionType_AesCtr256:
        {
            new (&m_DecryptorStorage) detail::service::util::AesCtrDecryptor256;

            reinterpret_cast<detail::service::util::AesCtrDecryptor256*>(&m_DecryptorStorage)->Initialize(key.value, 32, iv, ArchiveBlockSize);

            m_Destructor = [](ArchiveDecoder* pThis)
            {
                reinterpret_cast<detail::service::util::AesCtrDecryptor256*>(&pThis->m_DecryptorStorage)->~AesCtrDecryptor();
            };
            m_DecodeFunction = [](ArchiveDecoder* pThis, void* dest, const void* src, size_t size)
            {
                reinterpret_cast<detail::service::util::AesCtrDecryptor256*>(&pThis->m_DecryptorStorage)->Update(dest, size, src, size);
            };
        }
        break;
    default:
        NN_SDK_REQUIRES(0, "The encryption type(%d) is not supported.", header.encryptionType);
    }

    m_IsInitialized = true;
}

void ArchiveDecoder::Initialize(const ArchiveHeader& header, const char* passphrase, const nn::ApplicationId& appId, size_t blockIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    ArchiveDecryptionKey key;
    ArchiveDecryptionKeyGenerator::Generate(&key, passphrase, appId, header.saltIndex);

    Initialize(header, key, blockIndex);

    // メモリ上に残り続けないよう、すぐにクリアする。
    nn::crypto::detail::ClearMemory(&key, sizeof (key));
}

void ArchiveDecoder::Decode(void* dest, const void* src, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(dest);
    NN_SDK_REQUIRES_NOT_NULL(src);

    NN_SDK_ASSERT(m_IsInitialized);

    if (size == 0)
    {
        return;
    }

    if (m_DecodeFunction)
    {
        m_DecodeFunction(this, dest, src, size);
    }
    else
    {
        std::memcpy(dest, src, size);
    }
}

void ArchiveDecoder::Decode(void* dest, const void* src, size_t size,
    const ArchiveHeader& header, const ArchiveDecryptionKey& key) NN_NOEXCEPT
{
    ArchiveDecoder decoder;

    decoder.Initialize(header, key);
    decoder.Decode(dest, src, size);
}

void ArchiveDecoder::Decode(void* dest, const void* src, size_t size,
    const ArchiveHeader& header, const char* passphrase, const nn::ApplicationId& appId) NN_NOEXCEPT
{
    ArchiveDecoder decoder;

    decoder.Initialize(header, passphrase, appId);
    decoder.Decode(dest, src, size);
}

void ArchiveDecoder::CalculateIv(Bit8* dest, const Bit8* src, size_t count) NN_NOEXCEPT
{
    std::memcpy(dest, src, ArchiveBlockSize);

    for (size_t c = 0; c < count; c++)
    {
        for (int i = ArchiveBlockSize - 1; i >= 0; i--)
        {
            if (++dest[i] != 0)
            {
                break;
            }
        }
    }
}

}}}
