﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_PassphraseManager.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

PassphraseManager::PassphraseManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_MutexForTemporary(true),
    m_Count(0)
{
}

nn::Result PassphraseManager::Get(char* passphrase, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(passphrase);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, static_cast<size_t>(PassphraseLengthMax + 1));
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    NN_UNUSED(size);

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakePassphrasePath(path, sizeof (path), appId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultPassphraseNotSaved());
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    size_t read;
    NN_RESULT_DO(nn::fs::ReadFile(&read, handle, 0, passphrase, PassphraseLengthMax));

    passphrase[read] = '\0';

    NN_RESULT_SUCCESS;
}

nn::Result PassphraseManager::Save(nn::ApplicationId appId, const char* passphrase, bool needsCommit) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    int length = nn::util::Strnlen(passphrase, PassphraseLengthMax + 1);

    NN_RESULT_THROW_UNLESS(length > 0 && length <= PassphraseLengthMax, ResultInvalidArgument());

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakePassphrasePath(path, sizeof (path), appId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    {
        nn::fs::FileHandle handle = {};
        bool isCreateNew = false;

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(FileSystem::CreateFile(path, length));
                NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));

                isCreateNew = true;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        if (!isCreateNew)
        {
            char prevPassphrase[PassphraseLengthMax + 1] = {};

            size_t prevLength;
            NN_RESULT_DO(nn::fs::ReadFile(&prevLength, handle, 0, prevPassphrase, PassphraseLengthMax));

            if (length == static_cast<int>(prevLength) && nn::util::Strncmp(passphrase, prevPassphrase, length) == 0)
            {
                NN_RESULT_SUCCESS;
            }
            if (length != static_cast<int>(prevLength))
            {
                NN_RESULT_DO(nn::fs::SetFileSize(handle, length));
            }
        }

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, passphrase, length,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    if (needsCommit)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(appId));
    }

    NN_RESULT_SUCCESS;
}

nn::Result PassphraseManager::Delete(nn::ApplicationId appId, bool needsCommit) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    char path[80] = {};
    DeliveryCacheStorageManager::GetInstance().MakePassphrasePath(path, sizeof (path), appId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    if (needsCommit)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(appId));
    }

    NN_RESULT_SUCCESS;
}

bool PassphraseManager::AddTemporary(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_MutexForTemporary)> lock(m_MutexForTemporary);

    int index = SearchRecord(appId);

    if (index == -1)
    {
        if (m_Count == TemporaryCountMax)
        {
            return false;
        }

        index = m_Count++;
    }

    m_Records[index].appId = appId;
    nn::util::Strlcpy(m_Records[index].passphrase, passphrase, sizeof (m_Records[index].passphrase));

    return true;
}

void PassphraseManager::RemoveTemporary(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_MutexForTemporary)> lock(m_MutexForTemporary);

    int index = SearchRecord(appId);

    if (index == -1)
    {
        return;
    }

    int moveCount = m_Count - 1 - index;

    if (moveCount > 0)
    {
        std::memmove(&m_Records[index], &m_Records[index + 1], sizeof (Record) * moveCount);
    }

    m_Count--;
}

nn::Result PassphraseManager::SaveTemporary(nn::ApplicationId appId, bool needsCommit, bool removeIfSuccess) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    Record record = {};

    {
        std::lock_guard<decltype (m_MutexForTemporary)> lock(m_MutexForTemporary);

        int index = SearchRecord(appId);

        if (index == -1)
        {
            NN_RESULT_THROW(ResultNotFound());
        }

        record = m_Records[index];
    }

    NN_RESULT_DO(PassphraseManager::GetInstance().Save(appId, record.passphrase, needsCommit));

    if (removeIfSuccess)
    {
        RemoveTemporary(appId);
    }

    NN_RESULT_SUCCESS;
}

int PassphraseManager::SearchRecord(nn::ApplicationId appId) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (appId == m_Records[i].appId)
        {
            return i;
        }
    }

    return -1;
}

}}}}}
