﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DeliveryListReader.h>
#include <nn/bcat/detail/service/msgpack/bcat_MsgpackReader.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

DeliveryListReader::DeliveryListReader() NN_NOEXCEPT :
    m_ListHeader(),
    m_ListHeaderBits(),
    m_CurrentDir(),
    m_CurrentDirBits(),
    m_CurrentFile(),
    m_CurrentFileBits(),
    m_IsWalkingEnabled(false),
    m_IsFileListingRequired(false),
    m_SpecifiedDirName(nullptr),
    m_SpecifiedDirFileCount(0),
    m_IsSpecifiedDir(false),
    m_IsSpecifiedDirFound(false),
    m_Dirs(nullptr),
    m_DirIndex(0),
    m_DirCountMax(0),
    m_Files(nullptr),
    m_FileIndex(0),
    m_FileCountMax(0),
    m_OutCount(nullptr),
    m_Callback(nullptr),
    m_Param(nullptr)
{
}

void DeliveryListReader::SetupForDirectoryListing(int* outCount, ListDirectory* outDirs, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES(outDirs);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_SDK_ASSERT(!m_Dirs && !m_Files);

    m_OutCount = outCount;

    m_Dirs = outDirs;
    m_DirCountMax = count;
}

void DeliveryListReader::SetupForFileListing(int* outCount, ListFile* outFiles, int count, const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outFiles);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    NN_SDK_ASSERT(!m_Dirs && !m_Files);

    m_OutCount = outCount;

    m_Files = outFiles;
    m_FileCountMax = count;

    m_SpecifiedDirName = dirName;
}

void DeliveryListReader::SetupForWalking(WalkCallback callback, void* param, WalkWorkBuffer& buffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(callback);

    NN_SDK_ASSERT(!m_Dirs && !m_Files);

    m_IsWalkingEnabled = true;

    m_Files = reinterpret_cast<ListFile*>(&buffer);
    m_FileCountMax = DeliveryCacheFileCountMaxPerDirectory;

    m_Callback = callback;
    m_Param = param;
}

nn::Result DeliveryListReader::Read(ListHeader* outHeader, nne::nlib::InputStream& stream) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Dirs || m_Files);

    detail::service::msgpack::MsgpackReader reader;

    NN_RESULT_TRY(reader.Parse(stream, *this, sizeof (Url)))
        NN_RESULT_CATCH(ResultAbortedByCallback)
        {
            NN_RESULT_THROW(ResultInvalidFormat());
        }
    NN_RESULT_END_TRY;

    if (m_IsWalkingEnabled)
    {
    }
    else if (m_Dirs)
    {
        *m_OutCount = m_DirIndex;
    }
    else if (m_Files)
    {
        NN_RESULT_THROW_UNLESS(m_IsSpecifiedDirFound, ResultNotFound());

        *m_OutCount = m_SpecifiedDirFileCount;
    }

    if (outHeader)
    {
        *outHeader = m_ListHeader;
    }

    NN_RESULT_SUCCESS;
}

bool DeliveryListReader::OnStartObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.directories[*]"))
    {
        std::memset(&m_CurrentDirBits, 0, sizeof (m_CurrentDirBits));
        std::memset(&m_CurrentDir, 0, sizeof (m_CurrentDir));
    }
    else if (m_IsFileListingRequired)
    {
        if (jsonPath.Match("$.directories[*].data_list[*]"))
        {
            std::memset(&m_CurrentFileBits, 0, sizeof (m_CurrentFileBits));
            std::memset(&m_CurrentFile, 0, sizeof (m_CurrentFile));
        }
    }

    return true;
}

bool DeliveryListReader::OnEndObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match("$.directories[*]"))
    {
        if (m_CurrentDirBits.name && m_CurrentDirBits.mode && m_CurrentDirBits.digest)
        {
            if (m_Dirs && m_DirIndex < m_DirCountMax)
            {
                m_Dirs[m_DirIndex++] = m_CurrentDir;
            }
            if (m_Callback && m_Files)
            {
                m_Callback(m_CurrentDir, m_Files, m_FileIndex, m_Param);
            }
            if (m_IsSpecifiedDir)
            {
                m_SpecifiedDirFileCount = m_FileIndex;
            }
        }
        else
        {
            return false;
        }
    }
    else if (m_IsFileListingRequired)
    {
        if (jsonPath.Match("$.directories[*].data_list[*]"))
        {
            if (m_CurrentFileBits.name && m_CurrentFileBits.dataId && m_CurrentFileBits.size &&
                m_CurrentFileBits.url && m_CurrentFileBits.digest)
            {
                if (m_FileIndex < m_FileCountMax)
                {
                    for (int i = 0; i < m_FileIndex; i++)
                    {
                        if (nn::util::Strnicmp(m_Files[i].name.value, m_CurrentFile.name.value, sizeof (m_Files[i].name.value)) == 0)
                        {
                            NN_DETAIL_BCAT_INFO("[bcat] [Warning] Duplication of the file name was detected. (name=%s, dataId=%llu/%llu)\n",
                                m_Files[i].name.value, m_Files[i].dataId, m_CurrentFile.dataId);
                            // 重複を検知した場合、後勝ちとする。
                            m_Files[i] = m_CurrentFile;

                            return true;
                        }
                    }

                    m_Files[m_FileIndex++] = m_CurrentFile;
                }
            }
            else
            {
                return false;
            }
        }
    }

    return true;
}

bool DeliveryListReader::OnStartArray(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Compare("$.directories[]"))
    {
        if (!(m_ListHeaderBits.topicId && m_ListHeaderBits.isInService &&
            m_ListHeaderBits.isNintendoAccountRequired && m_ListHeaderBits.requiredAppVersion))
        {
            return false;
        }
    }
    else if (m_Files)
    {
        if (jsonPath.Match("$.directories[*].data_list[]"))
        {
            m_IsSpecifiedDir = (m_SpecifiedDirName &&
                nn::util::Strnicmp(m_SpecifiedDirName, m_CurrentDir.name.value, sizeof (m_CurrentDir.name.value)) == 0);

            m_IsSpecifiedDirFound = (m_IsSpecifiedDirFound || m_IsSpecifiedDir);

            m_IsFileListingRequired = (m_IsWalkingEnabled || m_IsSpecifiedDir);
            m_FileIndex = 0;
        }
    }

    return true;
}

bool DeliveryListReader::OnBoolean(const detail::service::msgpack::JsonPath& jsonPath, bool value) NN_NOEXCEPT
{
    if (!m_ListHeaderBits.isNintendoAccountRequired &&
        jsonPath.Compare("$.na_required"))
    {
        m_ListHeader.isNintendoAccountRequired = value;
        m_ListHeaderBits.isNintendoAccountRequired = true;
    }

    return true;
}

bool DeliveryListReader::OnInteger(const detail::service::msgpack::JsonPath& jsonPath, const detail::service::msgpack::AnyInteger& value) NN_NOEXCEPT
{
    if (!m_ListHeaderBits.requiredAppVersion &&
        jsonPath.Compare("$.required_application_version"))
    {
        m_ListHeader.requiredAppVersion = static_cast<uint32_t>(value.u);
        m_ListHeaderBits.requiredAppVersion = true;
    }
    else if (m_IsFileListingRequired)
    {
        if (!m_CurrentFileBits.dataId &&
            jsonPath.Match("$.directories[*].data_list[*].data_id"))
        {
            m_CurrentFile.dataId = value.u;
            m_CurrentFileBits.dataId = true;
        }
        else if (!m_CurrentFileBits.size &&
            jsonPath.Match("$.directories[*].data_list[*].size"))
        {
            m_CurrentFile.size = value.s;
            m_CurrentFileBits.size = true;
        }
    }

    return true;
}

bool DeliveryListReader::OnString(const detail::service::msgpack::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT
{
    NN_UNUSED(length);
    NN_UNUSED(isOverflowed);

    if (!m_ListHeaderBits.topicId &&
        jsonPath.Compare("$.topic_id"))
    {
        nn::util::Strlcpy(m_ListHeader.topicId.value, value, sizeof (m_ListHeader.topicId.value));
        m_ListHeaderBits.topicId = true;

        // TODO: 検証
    }
    else if (!m_ListHeaderBits.isInService &&
        jsonPath.Compare("$.service_status"))
    {
        if (nn::util::Strncmp(value, "in_service", sizeof ("in_service")) == 0)
        {
            m_ListHeader.isInService = true;
        }
        else if (nn::util::Strncmp(value, "expired", sizeof ("expired")) == 0)
        {
            m_ListHeader.isInService = false;
        }
        else
        {
            return false;
        }
        m_ListHeaderBits.isInService = true;
    }
    else if (!m_CurrentDirBits.name &&
        jsonPath.Match("$.directories[*].name"))
    {
        nn::util::Strlcpy(m_CurrentDir.name.value, value, sizeof (m_CurrentDir.name.value));
        m_CurrentDirBits.name = true;

        return m_CurrentDir.name.IsValid();
    }
    else if (!m_CurrentDirBits.mode &&
        jsonPath.Match("$.directories[*].mode"))
    {
        if (nn::util::Strncmp(value, "sync", sizeof ("sync")) == 0)
        {
            m_CurrentDir.mode = DirectoryMode_Sync;
        }
        else if (nn::util::Strncmp(value, "copy", sizeof ("copy")) == 0)
        {
            m_CurrentDir.mode = DirectoryMode_Copy;
        }
        else
        {
            return false;
        }
        m_CurrentDirBits.mode = true;
    }
    else if (!m_CurrentDirBits.digest &&
        jsonPath.Match("$.directories[*].digest"))
    {
        if (!ParameterConverter::Convert(&m_CurrentDir.digest, value))
        {
            return false;
        }
        m_CurrentDirBits.digest = true;
    }
    else if (m_IsFileListingRequired)
    {
        if (!m_CurrentFileBits.name &&
            jsonPath.Match("$.directories[*].data_list[*].filename"))
        {
            nn::util::Strlcpy(m_CurrentFile.name.value, value, sizeof (m_CurrentFile.name.value));
            m_CurrentFileBits.name = true;

            return m_CurrentFile.name.IsValid();
        }
        else if (!m_CurrentFileBits.url &&
            jsonPath.Match("$.directories[*].data_list[*].url"))
        {
            nn::util::Strlcpy(m_CurrentFile.url.value, value, sizeof (m_CurrentFile.url.value));
            m_CurrentFileBits.url = true;
        }
        else if (!m_CurrentFileBits.digest &&
            jsonPath.Match("$.directories[*].data_list[*].digest"))
        {
            if (!ParameterConverter::Convert(&m_CurrentFile.digest, value))
            {
                return false;
            }
            m_CurrentFileBits.digest = true;
        }
    }

    return true;
}

}}}}}
