﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.generic.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheFileService.generic.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheDirectoryService.generic.h>
#include <nn/bcat/detail/service/bcat_ServiceMemoryManager.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <mutex>
#include <nn/fs.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace bcat { namespace detail { namespace service {

DeliveryCacheStorageServiceForGeneric::DeliveryCacheStorageServiceForGeneric() NN_NOEXCEPT :
    m_Mutex(true)
{
}

DeliveryCacheStorageServiceForGeneric::~DeliveryCacheStorageServiceForGeneric() NN_NOEXCEPT
{
    nn::fs::Unmount("bcat-dc");
}

nn::Result DeliveryCacheStorageServiceForGeneric::CreateFileService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheFileService>> outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    auto& allocator = ServiceMemoryManager::GetInstance().GetDeliveryCacheFileServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IDeliveryCacheFileService, DeliveryCacheFileServiceForGeneric>(&allocator,
        nn::sf::SharedPointer<DeliveryCacheStorageServiceForGeneric>(this, true));

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheStorageServiceForGeneric::CreateDirectoryService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheDirectoryService>> outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    auto& allocator = ServiceMemoryManager::GetInstance().GetDeliveryCacheDirectoryServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IDeliveryCacheDirectoryService, DeliveryCacheDirectoryServiceForGeneric>(&allocator,
        nn::sf::SharedPointer<DeliveryCacheStorageServiceForGeneric>(this, true));

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheStorageServiceForGeneric::EnumerateDeliveryCacheDirectory(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::bcat::DirectoryName>& outDirectoryNames) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::fs::DirectoryHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenDirectory(&handle, "bcat-dc:/", nn::fs::OpenDirectoryMode_Directory))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            *outCount = 0;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    int actualCount = 0;

    while (actualCount < static_cast<int>(outDirectoryNames.GetLength()))
    {
        nn::fs::DirectoryEntry entry = {};
        int64_t count;

        NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, handle, 1));

        if (count == 0)
        {
            break;
        }

        nn::util::Strlcpy(outDirectoryNames[actualCount++].value, entry.name, sizeof (outDirectoryNames[0].value));
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

}}}}
