﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheFileService.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheDirectoryService.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheStorageManager.h>
#include <nn/bcat/detail/service/core/bcat_PassphraseManager.h>
#include <nn/bcat/detail/service/core/bcat_TaskManager.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace bcat { namespace detail { namespace service {

DeliveryCacheStorageService::DeliveryCacheStorageService(nn::ApplicationId appId, const Capability& capability) NN_NOEXCEPT :
    m_Mutex(true),
    m_Capability(capability),
    m_AppId(appId),
    m_FileCount(0),
    m_DirCount(0)
{
}

DeliveryCacheStorageService::~DeliveryCacheStorageService() NN_NOEXCEPT
{
    nn::Result result = detail::service::core::PassphraseManager::GetInstance().SaveTemporary(m_AppId);

    if (result.IsFailure() && !ResultNotFound::Includes(result))
    {
        NN_DETAIL_BCAT_INFO("[bcat] PassphraseManager::SaveTemporary failed. code = %03d-%04d\n",
            result.GetModule(), result.GetDescription());
    }

    detail::service::core::DeliveryCacheStorageManager::GetInstance().Unmount(m_AppId);

    detail::service::core::TaskManager::GetInstance().NotifyStorageUnmounted(m_AppId);
}

void DeliveryCacheStorageService::NotifyFileServiceDeleted() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_SDK_ASSERT(m_FileCount > 0);

    m_FileCount--;
}

void DeliveryCacheStorageService::NotifyDirectoryServiceDeleted() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_SDK_ASSERT(m_DirCount > 0);

    m_DirCount--;
}

nn::Result DeliveryCacheStorageService::CreateFileService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheFileService>> outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_FileCount < detail::ipc::DeliveryCacheFileServiceCountPerProcess, ResultTooManyHandles());

    auto& allocator = ServiceMemoryManager::GetInstance().GetDeliveryCacheFileServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IDeliveryCacheFileService, DeliveryCacheFileService>(&allocator,
        nn::sf::SharedPointer<DeliveryCacheStorageService>(this, true), m_AppId, m_Capability);

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    m_FileCount++;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheStorageService::CreateDirectoryService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheDirectoryService>> outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_DirCount < detail::ipc::DeliveryCacheDirectoryServiceCountPerProcess, ResultTooManyHandles());

    auto& allocator = ServiceMemoryManager::GetInstance().GetDeliveryCacheDirectoryServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IDeliveryCacheDirectoryService, DeliveryCacheDirectoryService>(&allocator,
        nn::sf::SharedPointer<DeliveryCacheStorageService>(this, true), m_AppId, m_Capability);

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    m_DirCount++;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheStorageService::EnumerateDeliveryCacheDirectory(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::bcat::DirectoryName>& outDirectoryNames) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    detail::service::core::DirectoryMetaView view;

    NN_RESULT_DO(detail::service::core::DirectoryMetaView::LoadFromDeliveryCacheStorage(&view, m_AppId));

    int actualCount = 0;

    while (actualCount < static_cast<int>(outDirectoryNames.GetLength()))
    {
        DirectoryMeta meta = {};

        NN_RESULT_TRY(view.GetEntry(&meta, actualCount))
            NN_RESULT_CATCH(ResultNotFound)
            {
                break;
            }
        NN_RESULT_END_TRY;

        nn::util::Strlcpy(outDirectoryNames[actualCount++].value, meta.name.value, sizeof (outDirectoryNames[0].value));
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

}}}}
