﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/bcat_DeliveryCacheFile.h>
#include <nn/bcat/detail/bcat_ErrorHandler.h>
#include <nn/bcat/detail/bcat_ShimLibraryGlobal.h>
#include <nn/bcat/detail/ipc/bcat_ServiceConfig.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/util/util_StringUtil.h>

#define NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_FILE NN_DETAIL_FS_ACCESS_LOG_FORMAT_CLASSNAME(DeliveryCacheFile)

namespace nn { namespace bcat {

NN_STATIC_ASSERT(DeliveryCacheFile::HandleCountMax == detail::ipc::DeliveryCacheFileServiceCountPerProcess);

DeliveryCacheFile::DeliveryCacheFile() NN_NOEXCEPT :
    m_Handle(nullptr)
{

}

DeliveryCacheFile::~DeliveryCacheFile() NN_NOEXCEPT
{
    if (m_Handle)
    {
        const auto handle = nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle);
        NN_DETAIL_FS_ACCESS_LOG((Close(), nn::ResultSuccess()), handle, NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_FILE);
    }
}

nn::Result DeliveryCacheFile::Open(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    DirectoryName d = {};
    FileName f = {};

    const char* separator = std::strchr(path, '/');

    NN_RESULT_THROW_UNLESS(separator, ResultInvalidArgument());

    int dirCount = static_cast<int>(separator - path) + 1;

    NN_RESULT_THROW_UNLESS(dirCount <= sizeof (d.value), ResultInvalidArgument());

    nn::util::Strlcpy(d.value, path, dirCount);
    nn::util::Strlcpy(f.value, separator + 1, sizeof (f.value));

    NN_RESULT_DO(Open(d, f));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFile::Open(const DirectoryName& dirName, const FileName& fileName) NN_NOEXCEPT
{
    auto open = [&]() NN_NOEXCEPT -> nn::Result
    {
        NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());
        NN_RESULT_THROW_UNLESS(fileName.IsValid(), ResultInvalidArgument());

        NN_RESULT_THROW_UNLESS(!m_Handle, ResultAlreadyOpened());

        detail::ipc::IDeliveryCacheFileService* handle = nullptr;

        NN_DETAIL_BCAT_IPC_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateFileService(&handle));

        bool isSuccess = false;

        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                nn::sf::ReleaseSharedObject(handle);
            }
        };

        NN_DETAIL_BCAT_IPC_RESULT_DO(handle->Open(dirName, fileName));

        m_Handle = handle;
        detail::ShimLibraryGlobal::GetInstance().IncrementDeliveryCacheFileCount();

        isSuccess = true;

        NN_RESULT_SUCCESS;
    };

    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(open(), nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle),
        NN_DETAIL_FS_ACCESS_LOG_FORMAT_DIRECTORY_FILENAME NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_FILE, dirName.value, fileName.value));
    NN_RESULT_SUCCESS;
}

void DeliveryCacheFile::Close() NN_NOEXCEPT
{
    auto close = [&]() NN_NOEXCEPT
    {
        if (m_Handle)
        {
            nn::sf::ReleaseSharedObject(m_Handle);

            m_Handle = nullptr;
            detail::ShimLibraryGlobal::GetInstance().DecrementDeliveryCacheFileCount();
        }
    };
    const auto handle = nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle);
    NN_DETAIL_FS_ACCESS_LOG((close(), nn::ResultSuccess()), handle, NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_FILE);
}

nn::Result DeliveryCacheFile::Read(size_t* outSize, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0ll);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    auto read = [&]() NN_NOEXCEPT -> nn::Result
    {
        NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

        uint64_t actualSize = 0;

        NN_DETAIL_BCAT_IPC_RESULT_DO(m_Handle->Read(&actualSize, offset, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), size)));

        *outSize = static_cast<size_t>(actualSize);

        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(read(), nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle),
        NN_DETAIL_FS_ACCESS_LOG_FORMAT_OFFSET_AND_SIZE NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_FILE, offset, size));
    NN_RESULT_SUCCESS;
}

int64_t DeliveryCacheFile::GetSize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Handle);

    int64_t size = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Handle->GetSize(&size));

    return size;
}

Digest DeliveryCacheFile::GetDigest() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Handle);

    Digest digest = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Handle->GetDigest(&digest));

    return digest;
}

}}
