﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/audioctrl/detail/audioctrl_Log.h>
#include "audioctrl_PlayReportController.h"

#define NN_AUDIOCTRL_TRACE(...)
#if !defined(NN_AUDIOCTRL_TRACE)
#define NN_AUDIOCTRL_TRACE(format, ...) NN_DETAIL_STRUCTURED_SDK_LOG(audioctrl, Trace, 0, "[audioctrl:%s] " format, NN_CURRENT_FUNCTION_NAME, ##__VA_ARGS__)
#endif

namespace nn { namespace audioctrl { namespace server { namespace detail {

PlayReportController::PlayReportController() NN_NOEXCEPT
    : m_CommittedOutputTarget(PlayReportAudioOutputTarget_Invalid),
      m_CommittedVolumeData({}),
      m_UncommittedOutputTarget(PlayReportAudioOutputTarget_Invalid),
      m_UncommittedSpeakerVolume({}),
      m_UncommittedHeadphoneVolume({}),
      m_UncommittedIsHeadphonePowerLimitted(false),
      m_IsSpeakerVolumeNotified(false),
      m_IsHeadphoneVolumeNotified(false),
      m_IsHeadphonePowerLimitedNotified(false),
      m_IsUsbOutputDeviceVolumeNotified(false),
      m_IsOutputTargetNotified(false),
      m_IsReleasedHoldedEvents(false),
      m_CommitTimer(nn::os::EventClearMode_AutoClear),
      m_CommitDelayTime({}),
      m_OutputTargetUpdateEvent(nn::os::EventClearMode_AutoClear, true),
      m_VolumeUpdateEvent(nn::os::EventClearMode_AutoClear, true),
      m_Mutex(true),
      m_IsInitialized(false)
{

}

void PlayReportController::Initialize(nn::TimeSpan commitDelayTime) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(!m_IsInitialized);

    if(m_IsInitialized)
    {
        return;
    }

    // コミット遅延時間を保存
    m_CommitDelayTime = commitDelayTime;

    // コミットタイマーを非シグナル状態かつストップ状態に設定
    m_CommitTimer.Clear();
    m_CommitTimer.Stop();

    // TODO: Workaround for SIGLO-61827
    // Initialize() が呼ばれるより前に AudioHardware に API を呼ばれてしまう不具合 (SIGLO-61827) があるため、
    // ここで AudioHardware が書いた値をタイミング依存で上書きしてしまうことがあるため無効化
    // コンストラクタで値は設定されており、Initialize / Finalize が実機上で起動中に呼ばれることもないので、実質的には設定しなくても振舞は変わらない

    // 一度以上コミットしたかどうかのフラグをオフ
    // m_IsSpeakerVolumeNotified = false;
    // m_IsHeadphoneVolumeNotified = false;
    // m_IsHeadphonePowerLimitedNotified = false;
    // m_IsUsbOutputDeviceVolumeNotified = false;
    // m_IsOutputTargetNotified = false;

    // 通知イベントの保留の解除を行ったことがあるかのフラグをオフ
    // m_IsReleasedHoldedEvents = false;

    // 初期化済みフラグをオン
    m_IsInitialized = true;
}

void PlayReportController::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(!m_IsInitialized)
    {
        return;
    }

    // 初期化済みフラグをオフ
    m_IsInitialized = false;
}

bool PlayReportController::IsInitialized() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_IsInitialized;
}

PlayReportAudioOutputTarget PlayReportController::GetAudioOutputTarget() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    NN_AUDIOCTRL_TRACE("[GetAudioOutputTarget] %d\n", m_CommittedOutputTarget);

    return m_CommittedOutputTarget;
}

void PlayReportController::GetAudioVolumeData(PlayReportAudioVolumeData* pOutData) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);
    NN_AUDIOCTRL_TRACE("[GetAudioVolumeData] Speaker: %d %d HP: %d %d %d\n UsbOutputDeviceVolume: %d %d",
            m_CommittedVolumeData.speaker.volume,
            m_CommittedVolumeData.speaker.mute,
            m_CommittedVolumeData.headphone.volume,
            m_CommittedVolumeData.headphone.mute,
            m_CommittedVolumeData.isHeadphonePowerLimited,
            m_CommittedVolumeData.usbOutputDevice.volume,
            m_CommittedVolumeData.usbOutputDevice.mute
            );

    *pOutData = m_CommittedVolumeData;
}

void PlayReportController::AcquireAudioOutputDeviceUpdateEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(outHandle == nullptr)
    {
        return;
    }

    *outHandle = m_OutputTargetUpdateEvent.GetReadableHandle();
}

void PlayReportController::AcquireAudioVolumeUpdateEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(outHandle == nullptr)
    {
        return;
    }

    *outHandle = m_VolumeUpdateEvent.GetReadableHandle();
}

void PlayReportController::NotifyOutputDeviceUpdate(PlayReportAudioOutputTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // Workaround for SIGLO-61827
    // NN_SDK_ASSERT(m_IsInitialized);

    NN_AUDIOCTRL_TRACE("target(%d)\n", target);

    m_UncommittedOutputTarget = target;
    m_IsOutputTargetNotified = true;

    StartCommitTimer();
}

void PlayReportController::NotifyAudioVolumeUpdateForVolume(PlayReportAudioOutputTarget target, PlayReportAudioVolume volume) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // Workaround for SIGLO-61827
    // NN_SDK_ASSERT(m_IsInitialized);
    const auto IsValidTarget = (target == PlayReportAudioOutputTarget_Speaker || target == PlayReportAudioOutputTarget_Headphone || target == PlayReportAudioOutputTarget_UsbOutputDevice);
    NN_SDK_ASSERT(IsValidTarget);

    if(!IsValidTarget)
    {
        return;
    }

    NN_AUDIOCTRL_TRACE("target(%d) volume(%d) mute(%d)\n", target, volume.volume, volume.mute);

    if(target == PlayReportAudioOutputTarget_Speaker)
    {
        m_UncommittedSpeakerVolume = volume;
        m_IsSpeakerVolumeNotified = true;
    }
    else if(target == PlayReportAudioOutputTarget_Headphone)
    {
        m_UncommittedHeadphoneVolume = volume;
        m_IsHeadphoneVolumeNotified = true;
    }
    else if(target == PlayReportAudioOutputTarget_UsbOutputDevice)
    {
        m_UncommittedUsbOutputDeviceVolume = volume;
        m_IsUsbOutputDeviceVolumeNotified = true;
    }

    StartCommitTimer();
}

void PlayReportController::NotifyAudioVolumeUpdateForHeadphonePowerLimit(bool isHeadphonePowerLimited) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // Workaround for SIGLO-61827
    // NN_SDK_ASSERT(m_IsInitialized);

    NN_AUDIOCTRL_TRACE("isHeadphonePowerLimited(%d).\n", isHeadphonePowerLimited);

    m_UncommittedIsHeadphonePowerLimitted = isHeadphonePowerLimited;

    m_IsHeadphonePowerLimitedNotified = true;

    StartCommitTimer();
}

nn::os::TimerEvent& PlayReportController::GetCommitTimer() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_CommitTimer;
}

bool PlayReportController::ComparePlayReportAudioVolumes(const PlayReportAudioVolume& volumeA, const PlayReportAudioVolume& volumeB) NN_NOEXCEPT
{
    return (volumeA.volume == volumeB.volume) && (volumeA.mute == volumeB.mute);
}

bool PlayReportController::ShouldHoldSignalUpdateEvent() const NN_NOEXCEPT
{
    return !(m_IsSpeakerVolumeNotified
          && m_IsHeadphoneVolumeNotified
          && m_IsHeadphonePowerLimitedNotified
          && m_IsUsbOutputDeviceVolumeNotified
          && m_IsOutputTargetNotified);
}

void PlayReportController::Commit() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // Workaround for SIGLO-61827
    // NN_SDK_ASSERT(m_IsInitialized);

    bool isOutputTargetUpdateEventSignalNeeded = false;

    // 出力先情報が更新されているかチェック
    if(m_CommittedOutputTarget != m_UncommittedOutputTarget)
    {
        NN_AUDIOCTRL_TRACE("Output target is committed.\n");

        m_CommittedOutputTarget = m_UncommittedOutputTarget;

        isOutputTargetUpdateEventSignalNeeded = true;
    }

    bool isVolumeUpdateEventSignalNeeded = false;

    // スピーカーの音量情報が更新されているかチェック
    if(!ComparePlayReportAudioVolumes(m_CommittedVolumeData.speaker, m_UncommittedSpeakerVolume))
    {
        NN_AUDIOCTRL_TRACE("SpeakerVolume is committed.\n");

        m_CommittedVolumeData.speaker = m_UncommittedSpeakerVolume;
        isVolumeUpdateEventSignalNeeded = true;
    }

    // ヘッドホンの音量情報が更新されているかチェック
    if(!ComparePlayReportAudioVolumes(m_CommittedVolumeData.headphone, m_UncommittedHeadphoneVolume))
    {
        NN_AUDIOCTRL_TRACE("HeadphoneVolume is committed.\n");

        m_CommittedVolumeData.headphone = m_UncommittedHeadphoneVolume;
        isVolumeUpdateEventSignalNeeded = true;
    }

    // USB 出力デバイスの音量情報が更新されているかチェック
    if(!ComparePlayReportAudioVolumes(m_CommittedVolumeData.usbOutputDevice, m_UncommittedUsbOutputDeviceVolume))
    {
        NN_AUDIOCTRL_TRACE("UsbOutputDeviceVolume is committed.\n");

        m_CommittedVolumeData.usbOutputDevice = m_UncommittedUsbOutputDeviceVolume;
        isVolumeUpdateEventSignalNeeded = true;
    }

    // ヘッドホン出力制限情報が更新されているかチェック
    if(m_CommittedVolumeData.isHeadphonePowerLimited != m_UncommittedIsHeadphonePowerLimitted)
    {
        NN_AUDIOCTRL_TRACE("isHeadphonePowerLimited is committed.\n");

        m_CommittedVolumeData.isHeadphonePowerLimited = m_UncommittedIsHeadphonePowerLimitted;
        isVolumeUpdateEventSignalNeeded = true;
    }

    // イベントを保留するべきかどうかをチェック
    if(!ShouldHoldSignalUpdateEvent())
    {
        // イベントを保留していた場合はここで全ての更新イベントをシグナル状態にする
        if(!m_IsReleasedHoldedEvents)
        {
            m_IsReleasedHoldedEvents = true;

            isOutputTargetUpdateEventSignalNeeded = true;
            isVolumeUpdateEventSignalNeeded = true;
        }

        if(isOutputTargetUpdateEventSignalNeeded)
        {
            NN_AUDIOCTRL_TRACE("OutputTargetUpdate is signaled.\n");

            m_OutputTargetUpdateEvent.Signal();
        }

        if(isVolumeUpdateEventSignalNeeded)
        {
            NN_AUDIOCTRL_TRACE("VolumeUpdate is signaled.\n");

            m_VolumeUpdateEvent.Signal();
        }
    }

    // コミットタイマーをクリア
    m_CommitTimer.Clear();
}

void PlayReportController::StartCommitTimer() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // Workaround for SIGLO-61827
    // NN_SDK_ASSERT(m_IsInitialized);

    m_CommitTimer.StartOneShot(m_CommitDelayTime);
}

}}}}  // namespace nn::audioctrl::server::detail


