﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Mutex.h>
#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>
#include <nne/audio/audio.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "../audio_Mixer.h"
#include "../../common/audio_Util.h"
#include "../audio_AudioSessionTypes.h"

namespace nn {
namespace audio {
namespace detail {

class UacInDevice
{
    enum UacReadResult
    {
        UacReadResult_Success,
        UacReadResult_Failure,
        UacReadResult_FailureAlreadyCopiedToUser,
    };

    struct UserBuffer
    {
        UserBuffer()
            : buffer(nullptr)
            , size(0)
            , writePosition(0)
            {}
        UserBuffer(void* _buffer, size_t _size)
            : buffer(_buffer)
            , size(_size)
            , writePosition(0)
        {}
        void* buffer;
        size_t size;
        size_t writePosition;
    };
public:
    UacInDevice() NN_NOEXCEPT;
    bool Update() NN_NOEXCEPT;

    static bool IsSupported(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT;

    //Plug/Unplug
    void InitializeDevice(const char* deviceName, nn::cduac::Interface* interface, nn::cduac::Host* host, nn::cduac::Parser* parser, const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;
    void FinalizeDevice() NN_NOEXCEPT;

    bool OpenSession(server::SessionFormat& format, int sessionId) NN_NOEXCEPT;
    void CloseSession() NN_NOEXCEPT;

    //Game Start/Stop
    void StartDevice() NN_NOEXCEPT;
    void StopDevice() NN_NOEXCEPT;

    bool AppendBuffer(void* buffer, size_t bufferSize, nn::dd::ProcessHandle) NN_NOEXCEPT;
    void* GetBufferAddress() NN_NOEXCEPT;

    bool IsStarted() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;
    const char* GetName() NN_NOEXCEPT;

    int GetSessionId() NN_NOEXCEPT;

    int16_t GetMinVolume() NN_NOEXCEPT;
    int16_t GetMaxVolume() NN_NOEXCEPT;
    int16_t GetCurrentVolume() NN_NOEXCEPT;
    Result SetCurrentVolume(int16_t volume) NN_NOEXCEPT;
    Result SetCurrentGain(float gain) NN_NOEXCEPT;

    bool IsMuted() NN_NOEXCEPT;
    void SetMute(bool mute) NN_NOEXCEPT;

    nn::os::SystemEventType* GetBufferCompletionEvent() NN_NOEXCEPT;
    void ClearBufferCompletionEvent() NN_NOEXCEPT;
    void ClearBuffers() NN_NOEXCEPT;

    bool CanOpen() const NN_NOEXCEPT;
private:
    static void SetSamplingFrequency(nn::cduac::AudioSamplingFrequency* outFrequency, uint32_t frequency) NN_NOEXCEPT;
    static int GetEndpointSamplingRateFromFormatType(nn::cduac::AudioStreamingFormatType* format) NN_NOEXCEPT;
    static int SetEndpointSamplingRate(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT;
    static nn::audio::SampleFormat UacFormatToAudioFormat(uint8_t uacFormat, uint8_t sampleSizeBits) NN_NOEXCEPT;
    static uint8_t FindAltSetting(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT;

    void StartDeviceInternal() NN_NOEXCEPT;
    void StopDeviceInternal(bool isConnected) NN_NOEXCEPT;
    void SetUacEndpoint() NN_NOEXCEPT;
    void SetUacControls() NN_NOEXCEPT;
    int8_t* ReadFromDevice(UacReadResult* outResult) NN_NOEXCEPT;
    int8_t* GetNextReadyBuffer() NN_NOEXCEPT;
    void SubmitUacBuffer() NN_NOEXCEPT;
    UacReadResult CheckXferStatus() NN_NOEXCEPT;
    void ResyncUacDevice(bool waitCompletions) NN_NOEXCEPT;
    void CopyToUserBuffer(int8_t* buffer, UacReadResult result) NN_NOEXCEPT;
    void SetNextUacFrameNumber() NN_NOEXCEPT;
    void Depop(int16_t* outBuffer, uint32_t sampleCount) NN_NOEXCEPT;
    bool IsDeviceInitialized() NN_NOEXCEPT;

private:
    static const int UacFrameDurationMs = 5;
    static const int UacMaxSamplingRate = 48000;
    static const int UacMaxSamplesPerMs = UacMaxSamplingRate / 1000;
    static const int UacMaxSamplesPerFrame = UacMaxSamplesPerMs * UacFrameDurationMs;
    static const int UacMaxChannelCount = 2;
    static const int UacBufferMaxCount = 6;
    static const int UacSampleSize = 2;
    static const int UacBufferSize = NN_AUDIO_ALIGN_UP(UacMaxSamplesPerFrame * UacMaxChannelCount * UacSampleSize, nn::os::MemoryPageSize);
    static const int UserBufferMaxCount = nne::audio::gmix::NumAudioInBuffers;

    NN_AUDIO_ALIGNAS_UAC_BUFFER int8_t m_UacBuffer[UacBufferMaxCount][UacBufferSize];
    NN_AUDIO_ALIGNAS_UAC_BUFFER int8_t m_ReportBuffer[nn::os::MemoryPageSize];
    uint32_t m_InputBytes[UacFrameDurationMs];
    nn::usb::XferReport m_XferReport[UacFrameDurationMs * UacBufferMaxCount];
    nn::os::Mutex m_Mutex;
    //nn::os::Mutex m_FinalizeMutex;
    nn::cduac::Host* m_pHost;
    nn::cduac::Parser* m_pParser;
    nn::cduac::Interface* m_pInterface;
    nn::cduac::AudioControlInputTerminal* m_pInputTerminal;
    nn::cduac::AudioControlFeatureUnit* m_pFeatureUnit;
    uint16_t m_MaxPacketSize;
    nn::os::SystemEventType* m_pStateChangeEvent;
    const char* m_Name;
    Mixer m_Mixer;
    MixerFormat m_OutFormat;
    nn::applet::AppletResourceUserId m_AppletId;
    uint32_t m_NextUsbFrame;
    int m_UacBufferIndex;
    int m_UacBufferReadIndex;
    int m_SessionId;
    uint32_t m_OutCount;
    int16_t m_MinVolume;
    int16_t m_MaxVolume;
    int16_t m_CurrentVolume;
    float m_Gain;
    int16_t m_Controls[UacMaxChannelCount];
    uint8_t m_AltSetting;
    bool m_IsOpen;
    bool m_IsStarted;
    bool m_IsInitialized;
    bool m_IsDeviceInitialized;
    bool m_IsMute;
    bool m_IsDeviceStarted;
    bool m_NeedResync;
    bool m_SkipSubmitBuffer;
};

}  // namespace detail
}  // namespace audio
}  // namespace nn
