﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_ServiceMemoryPoolInfo.h"

#include <nn/audio/detail/audio_Log.h>
#include <nn/audio/audio_Result.h>
#include "../dsp/audio_Dsp.h"

namespace nn { namespace audio { namespace server {

MemoryPoolInfo::MemoryPoolInfo(Location location) NN_NOEXCEPT
    : m_CpuAddress(nullptr)
    , m_DspAddress(0)
    , m_Size(0)
    , m_Location(location)
{
}

inline bool IsActionRequired(MemoryPoolInfo::State state)
{
    return (state == nn::audio::MemoryPoolInfo::State_RequestAttach) ||
           (state == nn::audio::MemoryPoolInfo::State_RequestDetach);
}

CpuAddr MemoryPoolInfo::GetCpuAddress() const NN_NOEXCEPT
{
    return m_CpuAddress;
}

nn::audio::DspAddr MemoryPoolInfo::GetDspAddress() const NN_NOEXCEPT
{
    return m_DspAddress;
}

size_t MemoryPoolInfo::GetSize() const NN_NOEXCEPT
{
    return m_Size;
}

MemoryPoolInfo::Location MemoryPoolInfo::GetLocation() const NN_NOEXCEPT
{
    return m_Location;
}

void MemoryPoolInfo::SetCpuAddress(CpuAddr cpuAddr, size_t size) NN_NOEXCEPT
{
    m_CpuAddress = cpuAddr;
    m_Size = size;
}

void MemoryPoolInfo::SetDspAddress(DspAddr dspAddr) NN_NOEXCEPT
{
    m_DspAddress = dspAddr;
}

bool MemoryPoolInfo::Contains(CpuAddr addr, size_t size) const NN_NOEXCEPT
{
    return (m_CpuAddress.GetAddress() <= addr.GetAddress()) && ( (addr.GetAddress() + size) <= (m_CpuAddress.GetAddress() + m_Size));
}

bool MemoryPoolInfo::IsMapped() const NN_NOEXCEPT
{
    return (m_DspAddress != NULL);
}

nn::audio::DspAddr MemoryPoolInfo::Translate(CpuAddr addr, size_t size) const NN_NOEXCEPT
{
    if (Contains(addr, size) == false || IsMapped() == false)
    {
        return NULL;
    }

    auto offset = (addr.GetAddress() - GetCpuAddress().GetAddress());
    return static_cast<DspAddr>(GetDspAddress() + offset);
}

void MemoryPoolInfo::SetUsed(bool used) NN_NOEXCEPT
{
    m_Used = used;
}

bool MemoryPoolInfo::IsUsed() const NN_NOEXCEPT
{
    return m_Used;
}

void PoolMapper::ClearUseState(MemoryPoolInfo* pPoolInfos, int poolCount) NN_NOEXCEPT
{
    for (auto i = 0; i < poolCount; ++i)
    {
        pPoolInfos[i].SetUsed(false);
    }
}

MemoryPoolInfo* PoolMapper::FindMemoryPool(MemoryPoolInfo* pInfos, int poolCount, CpuAddr addr, size_t size) const NN_NOEXCEPT
{
    for (auto i = 0; i < poolCount; ++i)
    {
        if (pInfos[i].Contains(addr, size))
        {
            return &pInfos[i];
        }
    }
    return nullptr;
}

nn::audio::server::MemoryPoolInfo* PoolMapper::FindMemoryPool(CpuAddr addr, size_t size) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Infos);
    NN_SDK_ASSERT_GREATER(m_PoolCount, 0);

    return this->FindMemoryPool(m_Infos, m_PoolCount, addr, size);
}

bool PoolMapper::FillDspAddr(AddressInfo* pOutAddr, MemoryPoolInfo* pInfos, int poolCount) const NN_NOEXCEPT
{
    if (pOutAddr->GetCpuAddr() == nullptr)
    {
        pOutAddr->SetPool(nullptr);
        return false;
    }

    auto pool = PoolMapper::FindMemoryPool(pInfos, poolCount, pOutAddr->GetCpuAddr(), pOutAddr->GetSize());
    if (pool)
    {
        pOutAddr->SetPool(pool);
        return true;
    }

    // No pool found for this buffer.
    if (m_IsForceMapEnabled)
    {
        auto addr = dsp::MapUserPointer(m_ProcessHandle, pOutAddr->GetCpuAddr(), pOutAddr->GetSize());
        pOutAddr->SetForceMappedDspAddr(addr);

        // This additional map is W.A.
        // current map style expect implicitly that reference count are enough high. So as not to release mapped region.
        dsp::MapUserPointer(m_ProcessHandle, pOutAddr->GetCpuAddr(), pOutAddr->GetSize());
    }
    else
    {
        pOutAddr->SetPool(nullptr);
    }
    return false;
}

bool PoolMapper::FillDspAddr(AddressInfo* pOutAddr) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Infos);
    NN_SDK_ASSERT_GREATER(m_PoolCount, 0);

    return FillDspAddr(pOutAddr, m_Infos, m_PoolCount);
}

bool PoolMapper::TryAttachBuffer(common::BehaviorParameter::ErrorInfo* pOutErroInfo, AddressInfo* pOutAddr, CpuAddr cpuAddr, size_t size) NN_NOEXCEPT
{
    pOutAddr->Setup(cpuAddr, size);
    if (FillDspAddr(pOutAddr))
    {
        pOutErroInfo->SetResult(ResultSuccess(), 0);
        return true;
    }
    else
    {
        pOutErroInfo->SetResult(ResultNoMemoryPoolEntry(), pOutAddr->GetCpuAddr().GetAddress());
        return IsForceMapEnabled() ? true : false;
    }
}

bool PoolMapper::IsForceMapEnabled() const NN_NOEXCEPT
{
    return m_IsForceMapEnabled;
}

void PoolMapper::ForceUnmapPointer(AddressInfo* pAddr) const NN_NOEXCEPT
{
    if (m_IsForceMapEnabled == false)
    {
        return;
    }
    NN_SDK_ASSERT_NOT_NULL(m_Infos);
    NN_SDK_ASSERT_GREATER(m_PoolCount, 0);

    if (FindMemoryPool(m_Infos, m_PoolCount, pAddr->GetCpuAddr(), pAddr->GetSize()) == nullptr)
    {
        dsp::UnmapUserPointer(m_ProcessHandle, pAddr->GetCpuAddr(), 0);
    }
}

PoolMapper::PoolMapper(nn::dd::ProcessHandle processHandle, bool forceMapping) NN_NOEXCEPT
    : m_ProcessHandle(processHandle)
    , m_Infos(nullptr)
    , m_PoolCount(0)
    , m_IsForceMapEnabled(forceMapping)
{
}

PoolMapper::PoolMapper(nn::dd::ProcessHandle processHandle, MemoryPoolInfo* pInfos, int poolCount, bool forceMapping) NN_NOEXCEPT
    : m_ProcessHandle(processHandle)
    , m_Infos(pInfos)
    , m_PoolCount(poolCount)
    , m_IsForceMapEnabled(forceMapping)
{
}

nn::dd::ProcessHandle PoolMapper::GetProcessHandle(const MemoryPoolInfo* pInfo) const NN_NOEXCEPT
{
    return (pInfo->GetLocation() == MemoryPoolInfo::Location_Client) ? m_ProcessHandle :
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
           (pInfo->GetLocation() == MemoryPoolInfo::Location_Service) ? nn::dd::GetCurrentProcessHandle() :
#endif
        nn::os::InvalidNativeHandle;
}

nn::audio::DspAddr PoolMapper::Map(nn::dd::ProcessHandle processHandle, CpuAddr addr, size_t size) const NN_NOEXCEPT
{
    //NN_DETAIL_AUDIO_TRACE("[%s:%d] address:%016llx, %08x\n", __FUNCTION__, __LINE__, addr.GetAddress(), size);
    return dsp::MapUserPointer(processHandle, addr, size);
}

bool PoolMapper::Map(MemoryPoolInfo* pOutMemoryPool) const NN_NOEXCEPT
{
    DspAddr addr = Map(GetProcessHandle(pOutMemoryPool), pOutMemoryPool->GetCpuAddress(), pOutMemoryPool->GetSize());
    if (addr != NULL)
    {
        //NN_DETAIL_AUDIO_TRACE("map memory pool:%08x\n", addr);
        pOutMemoryPool->SetDspAddress(addr);
        return true;
    }
    return false;
}

bool PoolMapper::Unmap(nn::dd::ProcessHandle processHandle, CpuAddr addr, size_t size) const NN_NOEXCEPT
{
    //NN_DETAIL_AUDIO_TRACE("[%s:%d] address:%016llx, %08x\n", __FUNCTION__, __LINE__, addr.GetAddress(), size);
    dsp::UnmapUserPointer(processHandle, addr, size); // checked in routine that if there are any used buffer in this Pool.
    //TODO : there are no way to check if really unmapped for now. Fix me.
    return true;
}

bool PoolMapper::Unmap(MemoryPoolInfo* pOutMemoryPool) const NN_NOEXCEPT
{
    if ((pOutMemoryPool->IsUsed() == false)
        && Unmap(GetProcessHandle(pOutMemoryPool), pOutMemoryPool->GetCpuAddress(), pOutMemoryPool->GetSize()))
    {
        pOutMemoryPool->SetCpuAddress(nullptr, 0);
        pOutMemoryPool->SetDspAddress(NULL);
        return true;
    }
    return false;
}

PoolMapper::Error PoolMapper::Update(MemoryPoolInfo* pOutPoolInfo, const nn::audio::MemoryPoolInfo::InParameter* pInParameter, nn::audio::MemoryPoolInfo::OutStatus* pOutOutStatus) const NN_NOEXCEPT
{
    if (IsActionRequired(pInParameter->_state) == false)
    {
        return Error_NoError; // no operation.
    }

    // check if address & size is valid
    if ((pInParameter->_address.IsNullPtr()) ||
        (pInParameter->_size == 0) ||
        ((pInParameter->_address.GetAddress() % nn::audio::MemoryPoolType::AddressAlignment) != 0) ||
        ((pInParameter->_size % nn::audio::MemoryPoolType::SizeGranularity) != 0))
    {
        return Error_InvalidAddressOrSize;
    }

    // Map/Unmap.
    if (pInParameter->_state == nn::audio::MemoryPoolInfo::State_RequestAttach)
    {
        pOutPoolInfo->SetCpuAddress(pInParameter->_address, static_cast<size_t>(pInParameter->_size));
        if (Map(pOutPoolInfo) == false)
        {
            // Failed to map, cleanup intermediate state from PoolInfo
            pOutPoolInfo->SetCpuAddress(nullptr, 0);
            return Error_FailedToMap;
        }
        pOutOutStatus->_state = nn::audio::MemoryPoolInfo::State_Attached;
    }
    else if (pInParameter->_state == nn::audio::MemoryPoolInfo::State_RequestDetach)
    {
        if ((pOutPoolInfo->GetCpuAddress() != pInParameter->_address) ||
            (pOutPoolInfo->GetSize() != pInParameter->_size))
        {
            // keep all state roll backed
            // -> currently nothing todo.
            return Error_InvalidAddressOrSize;
        }
        if (Unmap(pOutPoolInfo) == false)
        {
            return Error_FailedToUnmap;
        }
        pOutOutStatus->_state = nn::audio::MemoryPoolInfo::State_Detached;
    }
    return Error_NoError;
}

bool PoolMapper::InitializeSystemPool(MemoryPoolInfo* pOutInfo, void* address, size_t size) const NN_NOEXCEPT
{
    if (pOutInfo->GetLocation() != MemoryPoolInfo::Location_Service)
    {
        return false;
    }
    pOutInfo->SetCpuAddress(address, size);
    return Map(pOutInfo);
}

AddressInfo::AddressInfo(CpuAddr cpuAddr, size_t size) NN_NOEXCEPT
    : m_CpuAddr(cpuAddr)
    , m_Size(size)
    , m_Pool(nullptr)
    , m_ForceMappedDspAddr(NULL)
{
}

AddressInfo::~AddressInfo() NN_NOEXCEPT
{
    m_CpuAddr = nullptr;
    m_Size = 0;
    m_Pool = nullptr;
    m_ForceMappedDspAddr = NULL;
}

bool AddressInfo::HasMappedMemoryPool() const NN_NOEXCEPT
{
    return (m_Pool != nullptr && m_Pool->IsMapped());
}

nn::audio::CpuAddr AddressInfo::GetCpuAddr() const NN_NOEXCEPT
{
    return m_CpuAddr;
}

size_t AddressInfo::GetSize() const NN_NOEXCEPT
{
    return m_Size;
}

void AddressInfo::Setup(CpuAddr addr, size_t size) NN_NOEXCEPT
{
    m_CpuAddr = addr;
    m_Size = size;
    m_Pool = nullptr;
    m_ForceMappedDspAddr = NULL;
}

bool AddressInfo::IsMapped() const NN_NOEXCEPT
{
    return HasMappedMemoryPool() || m_ForceMappedDspAddr;
}

nn::audio::DspAddr AddressInfo::GetReference(bool countReference /* = true */) const NN_NOEXCEPT
{
    if (HasMappedMemoryPool())
    {
        if (countReference)
        {
            m_Pool->SetUsed(true);
        }
        return m_Pool->Translate(m_CpuAddr, m_Size);
    }
    else
    {
        return m_ForceMappedDspAddr; // TODO: force map 無効時には常に nullptr であることのアサートがしたい。
    }
}

void AddressInfo::SetPool(MemoryPoolInfo* pool) NN_NOEXCEPT
{
    m_Pool = pool;
}

void AddressInfo::SetForceMappedDspAddr(DspAddr addr) NN_NOEXCEPT
{
    // This API is only for non-MemoryPool use cases.
    NN_SDK_ASSERT(HasMappedMemoryPool() == false);
    m_ForceMappedDspAddr = addr;
}

nn::audio::DspAddr AddressInfo::GetForceMappedDspAddr() const NN_NOEXCEPT
{
    // This API is only for non-MemoryPool use cases.
    NN_SDK_ASSERT(HasMappedMemoryPool() == false);
    return m_ForceMappedDspAddr;
}
}}} // namespace nn::audio::server
