﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef>
#include <nn/nn_StaticAssert.h>
#include <nn/audio/audio_Result.h>
#include "../common/audio_BehaviorParameters.h"
#include "../common/audio_Util.h"

#include "../audio_EffectManager.h"
#include "../audio_AuxTypes.h"
#include "audio_ServiceMemoryPoolInfo.h"

namespace nn {
namespace audio {
namespace server {

class NN_AUDIO_INFOTYPE_FILED_ALIGN EffectInfoBase
{
public:
    enum class UsageState : uint8_t
    {
        Invalid,
        Initialized, // effect is enabled, dsp command is not published yet.
        Running,     // effect is enabled  && dsp touches effect buffer.
        Stopped,     // effect is disabled && dsp does not touch effect buffer.
    };
    static const int BufferCount = 2;

    enum Index
    {
        Index_Invalid = -1,
        Index_Delay = 0,
        Index_Reverb = 0,
        Index_I3dl2Reverb = 0,
        Index_AuxSend = 0,
        Index_AuxReturn = 1,
    };

protected:
    ::nn::audio::EffectType m_Type;
    bool m_Enabled;
    bool m_ShouldSkip;
    UsageState m_UsageState;
    MixId m_AttachMixId;
    int32_t m_ProcessingOrder;
    int8_t _padding[8];
    server::AddressInfo m_WorkBuffer[BufferCount];

public:
    AuxBufferAddresses m_AuxAddresses; // TODO: merge this field to m_StateBuffer; SIGLO-33781

protected:
    static const size_t ParameterBufferSize = NN_AUDIO_ALIGN_UP(sizeof(nn::audio::common::BiggestType <
        BufferMixerParameter,
        DelayParameter,
        ReverbParameter,
        I3dl2ReverbParameter,
        AuxParameter,
        BiquadFilterEffectParameter
        >::Result), nn::audio::BufferAlignSize);
    NN_AUDIO_INFOTYPE_FILED_ALIGN int8_t m_ParameterBuffer[ParameterBufferSize];

    static const size_t StateBufferSize = NN_AUDIO_ALIGN_UP(sizeof(nn::audio::common::BiggestType<
        DelayState,
        ReverbState,
        I3dl2ReverbState,
        AuxBufferAddresses,
        BiquadFilterEffectState
        >::Result), nn::audio::BufferAlignSize);
    NN_AUDIO_INFOTYPE_FILED_ALIGN int8_t m_StateBuffer[StateBufferSize];

    DspAddr GetSingleBuffer(server::EffectInfoBase::Index index) const NN_NOEXCEPT;
    void SetUsage(UsageState state) NN_NOEXCEPT;
    bool ShouldUpdateWorkBufferInfo(const nn::audio::EffectInfo::InParameter* pUserInParam) const NN_NOEXCEPT;

public:
    EffectInfoBase() NN_NOEXCEPT;
    void CleanUp() NN_NOEXCEPT;
    void ForceUnmapBuffers(PoolMapper& poolMapper) NN_NOEXCEPT;

    bool IsEnabled() const NN_NOEXCEPT;
    bool ShouldSkip() const NN_NOEXCEPT;
    ::nn::audio::EffectType GetType() const NN_NOEXCEPT;
    MixId GetMixId() const NN_NOEXCEPT;
    int32_t GetProcessingOrder() const NN_NOEXCEPT;
    const void* GetParameter() const NN_NOEXCEPT;
    void* GetStateBuffer() NN_NOEXCEPT;
    UsageState GetUsage() const NN_NOEXCEPT;
    void StoreStatus(nn::audio::EffectInfo::OutStatus* pOutStatus, bool isRendererActive) const NN_NOEXCEPT;

    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT;
    virtual DspAddr GetWorkBuffer(Index index = Index_Invalid) const NN_NOEXCEPT;
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(EffectInfoBase, NN_AUDIO_ADDRESS_SELECT(864, 1200));
#endif


template< nn::audio::EffectType type, typename parameter>
class EffectInfo : public EffectInfoBase
{
public:
    EffectInfo() NN_NOEXCEPT
    {
        EffectInfoBase::CleanUp();
        m_Type = type;
    }
    parameter* GetParameter() NN_NOEXCEPT
    {
        return reinterpret_cast<parameter*>(m_ParameterBuffer);
    }
    const parameter* GetParameter() const NN_NOEXCEPT
    {
        return reinterpret_cast<const parameter*>(m_ParameterBuffer);
    }
};

class BufferMixerInfo final : public EffectInfo<nn::audio::EffectType_BufferMixer, BufferMixerParameter>
{
public:
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
};

class DelayInfo final : public EffectInfo<nn::audio::EffectType_Delay, DelayParameter>
{
public:
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
    virtual DspAddr GetWorkBuffer(Index index) const NN_NOEXCEPT NN_OVERRIDE;
};

class ReverbInfo final : public EffectInfo<nn::audio::EffectType_Reverb, ReverbParameter>
{
public:
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
    virtual DspAddr GetWorkBuffer(Index index) const NN_NOEXCEPT NN_OVERRIDE;
};

class I3dl2ReverbInfo final : public EffectInfo<nn::audio::EffectType_I3dl2Reverb, I3dl2ReverbParameter>
{
public:
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
    virtual DspAddr GetWorkBuffer(Index index) const NN_NOEXCEPT NN_OVERRIDE;
};

class AuxInfo final : public EffectInfo<nn::audio::EffectType_Aux, AuxParameter>
{
public:
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
    virtual DspAddr GetWorkBuffer(Index index) const NN_NOEXCEPT NN_OVERRIDE;
};

class BiquadFilterInfo final : public EffectInfo<nn::audio::EffectType_BiquadFilter, BiquadFilterEffectParameter>
{
public:
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
};

// Classes derived from server::EffectInfo should not have its own member fields to keep all server::Effect type has same size.
NN_STATIC_ASSERT(sizeof(EffectInfoBase) == sizeof(BufferMixerInfo));
NN_STATIC_ASSERT(sizeof(EffectInfoBase) == sizeof(DelayInfo) );
NN_STATIC_ASSERT(sizeof(EffectInfoBase) == sizeof(ReverbInfo));
NN_STATIC_ASSERT(sizeof(EffectInfoBase) == sizeof(I3dl2ReverbInfo));
NN_STATIC_ASSERT(sizeof(EffectInfoBase) == sizeof(AuxInfo));
NN_STATIC_ASSERT(sizeof(EffectInfoBase) == sizeof(BiquadFilterInfo));


// Data container & accessor for service side Effect related infos.
class EffectContext
{
    server::EffectInfoBase* m_Infos;
    int m_InfoCount;
public:
    EffectContext() NN_NOEXCEPT;
    void Initialize(server::EffectInfoBase* infos, int infoCount) NN_NOEXCEPT;
    server::EffectInfoBase& GetInfo(int id) const NN_NOEXCEPT;
    int GetCount() const NN_NOEXCEPT;
};

}  // namespace server
}  // namespace audio
}  // namespace nn
