﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <functional>
#include <nn/audio/audio_PerformanceMetricsTypes.private.h>
#include <nn/audio/audio_PerformanceMetrics.h>

#include "audio_ICommandBuffer.h"
#include "audio_BaseAspect.h"

namespace nn { namespace audio { namespace server {

class PerfMeasureAspect final
    : public BaseAspect<ICommandBuffer>
{
public:
    PerfMeasureAspect(ICommandBuffer* publisher, PerformanceManager* pPerformanceManager, NodeId parentId, PerformanceSysDetailType target) NN_NOEXCEPT;
    ~PerfMeasureAspect() NN_NOEXCEPT;

    template <class Func, class... Args>
    typename std::result_of<Func(decltype(m_Publisher), Args&&...)>::type
        Invoke(Func&& func,  Args&&... args)
        {
            auto c = std::bind(std::move(func), m_Publisher, std::forward<Args>(args)...)();

            if(m_pEstimatedProcessingTime != nullptr && c != nullptr)
            {
                *m_pEstimatedProcessingTime = c->header.estimatedProcessingTime;
            }

            return c;
        }
private:
    PerformanceEntryAddresses m_Addresses = {};
    NodeId m_ParentId;
    bool m_Published = false;
    PerformanceManager* m_PerformanceManager;
    uint32_t* m_pEstimatedProcessingTime = nullptr;
};

class PerformanceMeasuredCommandBuffer : public ICommandBuffer
{
public :
    PerformanceMeasuredCommandBuffer(ICommandBuffer* pCommandBuffer, PerformanceManager* pPerformanceManager) NN_NOEXCEPT;
    virtual PcmInt16DataSourceCommand* GeneratePcmInt16DataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, int32_t channel, NodeId nodeId) NN_NOEXCEPT;
    virtual AdpcmDataSourceCommand* GenerateAdpcmDataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual AuxCommand* GenerateAuxCommand(int mixBufferOffset, int8_t input, int8_t output, const AuxBufferAddresses* pAddresses, bool enabled, int32_t sampleCount, DspAddr sendBuffer, DspAddr returnBuffer, int updateCount, int offset, NodeId nodeId) NN_NOEXCEPT;
    virtual DelayCommand* GenerateDelayEffectCommand(int mixBufferOffset, const DelayParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT;
    virtual ReverbCommand* GenerateReverbEffectCommand(int mixBufferOffset, const ReverbParameter* pParameter, void* pState, bool enabled, bool longSizePreDelaySupported, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT;
    virtual I3dl2ReverbCommand* GenerateI3dl2ReverbEffectCommand(int mixBufferOffset, const I3dl2ReverbParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT;
    virtual CopyMixBufferCommand* GenerateCopyMixBufferCommand(int inputBufferIndex, int outputBufferIndex, NodeId nodeId) NN_NOEXCEPT;
    virtual BiquadFilterCommand* GenerateBiquadFilterCommand(int mixBufferOffset, const BiquadFilterParameter* pFilterParameter, BiquadFilterState* pState, int8_t input, int8_t output, bool needInitialization, NodeId nodeId) NN_NOEXCEPT;
    virtual MixCommand* GenerateMixCommand(int inputBufferIndex, int outputBufferIndex, float mixVolume, NodeId nodeId) NN_NOEXCEPT;
    virtual MixRampCommand* GenerateMixRampCommand(int inputBufferIndex, int outputBufferIndex, float mixVolume0, float mixVolume1, DspAddr voiceStateDspAddr, NodeId nodeId) NN_NOEXCEPT;
    virtual MixRampGroupedCommand* GenerateMixRampGroupedCommand(int mixBufferCount, int inputBufferIndex, int outputBufferIndexStart, const float* pMixVolume, const float* pMixVolumePrev, DspAddr lastSampleAddress, NodeId nodeId) NN_NOEXCEPT;
    virtual DepopPrepareCommand* GenerateDepopPrepareCommand(VoiceState* state, int32_t* depopBuffer, int mixBufferCount, int mixBufferOffset, NodeId nodeId, bool commandEnabled) NN_NOEXCEPT;
    virtual DepopForMixBuffersCommand* GenerateDepopForMixBuffersCommand(int32_t* depopBuffer, int mixBufferOffset, int mixBufferCount, NodeId nodeId, int32_t sampleRate) NN_NOEXCEPT;
    virtual ClearMixBufferCommand* GenerateClearMixBufferCommand(NodeId nodeId) NN_NOEXCEPT;
    virtual VolumeCommand* GenerateVolumeCommand(float volume, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual VolumeRampCommand* GenerateVolumeRampCommand(float volume0, float volume1, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual DeviceSinkCommand* GenerateDeviceSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, uint32_t sessionId, DspAddr mixBufferAddr, NodeId nodeId) NN_NOEXCEPT;
    virtual CircularBufferSinkCommand* GenerateCircularBufferSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, NodeId nodeId) NN_NOEXCEPT;
    virtual DownMix6chTo2chCommand* GenerateDownMixCommand(int mixBufferOffset, const int8_t* input, const int8_t* output, const int32_t* downMixParam, NodeId nodeId) NN_NOEXCEPT;
    virtual UpsampleCommand* GenerateUpsampleCommand(int mixBufferOffset, UpsamplerInfo* pInfo, int inputCount, const int8_t* input, int32_t buffeCount, int32_t sampleCount, int32_t sampleRate, NodeId nodeId) NN_NOEXCEPT;
    virtual PerformanceCommand* GeneratePerformanceCommand(const PerformanceEntryAddresses* entry, const PerformanceCommandType type, NodeId nodeId) NN_NOEXCEPT;
private:
    PerfMeasureAspect CreateAspect(NodeId parentId, PerformanceSysDetailType target) NN_NOEXCEPT;
private:
    ICommandBuffer* m_pCommandBuffer;
    PerformanceManager* m_PerformanceManager;
};

}}}  // namespace nn::audio::server
