﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/audio/detail/audio_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_Result.private.h>
#include <nn/audio/audio_SampleFormat.private.h>
#include "../audio_RendererInfo.h"

#include "audio_InfoUpdater.h"

namespace nn { namespace audio { namespace server {

InfoUpdater::InfoUpdater(uintptr_t inParameter, size_t inParameterSize, uintptr_t outStatus, size_t outStatusSize, nn::dd::ProcessHandle m_ClientProcessHandle, BehaviorInfo& behaviorInfo) NN_NOEXCEPT
    : m_InParameter(inParameter), m_InParameterOrigin(inParameter), m_InParameterSize(inParameterSize),
      m_OutStatus(outStatus), m_OutStatusOrigin(outStatus), m_OutStatusSize(outStatusSize),
      m_BehaviorInfo(behaviorInfo), m_ClientProcessHandle(m_ClientProcessHandle)
{
    m_InParameterHeader = reinterpret_cast<common::UpdateDataHeader*>(inParameter);
    m_InParameter += sizeof(common::UpdateDataHeader);
    m_OutStatusHeader = new(reinterpret_cast<void*>(m_OutStatus)) nn::audio::common::UpdateDataHeader();
    m_OutStatusHeader->Initialize(m_BehaviorInfo.GetProcessRevision());
    m_OutStatus += sizeof(common::UpdateDataHeader);
}

Result InfoUpdater::UpdateVoiceChannelResources(VoiceContext& context) NN_NOEXCEPT
{
    auto pInParameters = reinterpret_cast<const ::nn::audio::VoiceChannelResource::InParameter*>(m_InParameter);

    for(int i = 0; i < context.GetCount(); ++i)
    {
        context.GetChannelResource(i).Update(&pInParameters[i]);
    }

    auto consumedInParamSize = sizeof(::nn::audio::VoiceChannelResource::InParameter) * context.GetCount();
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::voiceChannelResource) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }

    m_InParameter += consumedInParamSize;
    NN_RESULT_SUCCESS;
}

Result InfoUpdater::UpdateVoices(VoiceContext& context, server::MemoryPoolInfo* pPoolInfos, int memoryPoolCount, DspAddr buffer) NN_NOEXCEPT
{
    server::PoolMapper poolMapper(m_ClientProcessHandle, pPoolInfos, memoryPoolCount, m_BehaviorInfo.IsMemoryPoolForceMappingEnabled());

    auto outStatus = reinterpret_cast<::nn::audio::VoiceInfo::OutStatus*>(m_OutStatus);
    auto pInParameters = reinterpret_cast<const ::nn::audio::VoiceInfo::InParameter*>(m_InParameter);

    for(int i = 0; i < context.GetCount(); ++i)
    {
        context.GetInfo(i)._inParameter._isInUse = false;
    }

    int count = 0;
    for (int i = 0; i < context.GetCount(); ++i)
    {
        VoiceState* pTmpVoiceStates[nn::audio::VoiceType::ChannelCountMax] = { nullptr };
        auto pInParameter = &pInParameters[i];

        // ボイスが無効な時は以下の処理を行わない
        if(pInParameter->_isInUse != true)
        {
            continue;
        }

        auto pVoice = &context.GetInfo(pInParameter->_id);

        NN_SDK_ASSERT(pInParameter->_id >= 0 && pInParameter->_id < context.GetCount());

        for (int j = 0; j < pInParameter->_channelCount; ++j)
        {
            const auto channelResourceId = pInParameter->_voiceChannelResourceIds[j];
            NN_SDK_ASSERT_RANGE(channelResourceId, 0, context.GetCount());

            pTmpVoiceStates[j] = &context.GetState(channelResourceId);
        }

        if (pInParameter->_isNew)
        {
            pVoice->Initialize();

            const auto channelCount = pInParameter->_channelCount;
            NN_SDK_ASSERT_MINMAX(channelCount, 1, ::nn::audio::VoiceType::ChannelCountMax);
            if(!(channelCount >= 1 && channelCount <= ::nn::audio::VoiceType::ChannelCountMax))
            {
                continue;
            }

            for(auto j = 0; j < channelCount; ++j)
            {
                memset(pTmpVoiceStates[j], 0, sizeof(VoiceState));
                if (pInParameter->_sampleFormat == SampleFormat_Opus)
                {
                    pTmpVoiceStates[j]->externalContext = buffer + pInParameter->_audioCodecBufferOffset;
                    pTmpVoiceStates[j]->externalContextSize = pInParameter->_audioCodecBufferSize;
                }
            }
        }

        common::BehaviorParameter::ErrorInfo errorInfo;
        pVoice->UpdateParameters(&errorInfo, pInParameter, poolMapper, m_BehaviorInfo);
        if (errorInfo.IsError())
        {
            m_BehaviorInfo.AppendError(errorInfo);
        }

        const int errorInfoCount = 2 * nn::audio::VoiceType::WaveBufferCountMax;
        common::BehaviorParameter::ErrorInfo errorInfos[errorInfoCount];
        memset(errorInfos, 0, sizeof(errorInfos));
        pVoice->UpdateWaveBuffers(errorInfos, errorInfoCount, pInParameter, pTmpVoiceStates, poolMapper, m_BehaviorInfo);
        for (auto& info : errorInfos)
        {
            if (info.IsError())
            {
                m_BehaviorInfo.AppendError(info);
            }
        }

        pVoice->WriteOutStatus(&outStatus[i], pInParameter, pTmpVoiceStates);

        count += pInParameter->_channelCount;
    }

    auto consumedInParamSize = sizeof(::nn::audio::VoiceInfo::InParameter) * context.GetCount();
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::voice) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }
    auto consumuedOutStatusSize = sizeof(::nn::audio::VoiceInfo::OutStatus) * context.GetCount();
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::voice, consumuedOutStatusSize);

    m_InParameter += consumedInParamSize;
    m_OutStatus += consumuedOutStatusSize;

    context.SetActiveCount(count);
    NN_RESULT_SUCCESS;
}


namespace {
void ResetEffectInfo(server::EffectInfoBase* pInfo, ::nn::audio::EffectType type) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    NN_SDK_ASSERT(pInfo->IsEnabled() == false);

    server::EffectInfoBase* info = nullptr;
    switch (type)
    {
    case ::nn::audio::EffectType_BufferMixer:
        info = new(pInfo) server::BufferMixerInfo();
        break;
    case ::nn::audio::EffectType_Delay:
        info = new(pInfo) server::DelayInfo();
        break;
    case ::nn::audio::EffectType_Reverb:
        info = new(pInfo) server::ReverbInfo();
        break;
    case ::nn::audio::EffectType_I3dl2Reverb:
        info = new(pInfo) server::I3dl2ReverbInfo();
        break;
    case ::nn::audio::EffectType_Aux:
        info = new(pInfo) server::AuxInfo();
        break;
    case ::nn::audio::EffectType_BiquadFilter:
        info = new(pInfo) server::BiquadFilterInfo();
        break;
    case ::nn::audio::EffectType_Invalid:
        info = new(pInfo) server::EffectInfoBase();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_SDK_ASSERT_NOT_NULL(info);
    NN_UNUSED(info);
}
}

Result InfoUpdater::UpdateEffects(EffectContext& context, bool isRendererActive, server::MemoryPoolInfo* pPoolInfos, int memoryPoolCount) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    auto pOutStatus = reinterpret_cast<::nn::audio::EffectInfo::OutStatus*>(m_OutStatus);
    auto pInParameter = reinterpret_cast<::nn::audio::EffectInfo::InParameter*>(m_InParameter);
    auto poolMapper = PoolMapper(m_ClientProcessHandle, pPoolInfos, memoryPoolCount, m_BehaviorInfo.IsMemoryPoolForceMappingEnabled());

    for (int i = 0; i < context.GetCount(); ++i)
    {
        server::EffectInfoBase& updateTarget = context.GetInfo(i);
        const auto& fromUser = pInParameter[i];
        auto& toUser = pOutStatus[i];

        if (updateTarget.GetType() != fromUser._type)
        {
            updateTarget.ForceUnmapBuffers(poolMapper);
            ResetEffectInfo(&updateTarget, fromUser._type);
        }
        common::BehaviorParameter::ErrorInfo errorInfo;
        updateTarget.Update(&errorInfo, &fromUser, poolMapper);
        if (errorInfo.IsError())
        {
            m_BehaviorInfo.AppendError(errorInfo);
        }
        updateTarget.StoreStatus(&toUser, isRendererActive);
    }

    auto consumedInParamSize = sizeof(::nn::audio::EffectInfo::InParameter) * context.GetCount();
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::effect) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }
    auto consumuedOutStatusSize = sizeof(::nn::audio::EffectInfo::OutStatus) * context.GetCount();
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::effect, consumuedOutStatusSize);

    m_InParameter += consumedInParamSize;
    m_OutStatus += consumuedOutStatusSize;

    return result;
}

namespace {
bool CheckMixInfoError(const nn::audio::MixInfo* pInMixInfos, int mixInfoCount, int mixBufferCount) NN_NOEXCEPT
{
    int bufferCount = 0;
    bool invalidDestination = false;
    for (auto i = 0; i < mixInfoCount; ++i)
    {
        const auto& pInfo = pInMixInfos[i];
        bufferCount += pInfo.bufferCount;
        if (pInfo.destinationMixId > mixInfoCount &&                // destinationId exceeded maximum number of mixInfo
            pInfo.destinationMixId != ::nn::audio::Invalid_MixId && // un-connected mixInfo's destinationid is initialized as Invalid_MixId
            pInfo.mixId != MixInfo::FinalMixId)                     // except for FinalMix.
        {
            invalidDestination = true;
            break;
        }
    }

    return ((bufferCount > mixBufferCount) || invalidDestination);
}
}

Result InfoUpdater::UpdateMixes(MixContext& mixContext, int mixBufferCount, const EffectContext& effectContext, const SplitterContext& splitterContext) NN_NOEXCEPT
{
    auto pInMixInfos = reinterpret_cast<const ::nn::audio::MixInfo*>(m_InParameter);

    // check total buffer count.
    if (CheckMixInfoError(pInMixInfos, mixContext.GetCount(), mixBufferCount))
    {
        return ResultInvalidUpdateInfo();
    }
    for (auto i = 0; i < mixContext.GetCount(); ++i)
    {
        auto& mixInfo = mixContext.GetInfo(i);
        if (mixInfo.isInUse != pInMixInfos[i].isInUse)
        {
            mixInfo.isInUse = pInMixInfos[i].isInUse;
            if (mixInfo.isInUse == false)
            {
                mixInfo.ClearEffectProcessingOrder();
            }
        }

        if (pInMixInfos[i].isInUse)
        {
            mixInfo.Update(mixContext.m_Edges, pInMixInfos[i], effectContext, splitterContext, m_BehaviorInfo);
        }
    }

    // sort infos
    if (m_BehaviorInfo.IsSplitterSupported() &&
        mixContext.TsortInfo(splitterContext) == false)
    {
        return ResultCycleDetected();
    }

    auto consumedInParamSize = sizeof(::nn::audio::MixInfo) * mixContext.GetCount();
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::mix) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }

    m_InParameter += consumedInParamSize;

    NN_RESULT_SUCCESS;
}

namespace {
void ResetSinkInfo(SinkInfoBase* pInfo, common::SinkType type) NN_NOEXCEPT
{
    SinkInfoBase* info = nullptr;
    switch (type)
    {
    case common::SinkType_CircularBuffer:
        info = new(pInfo) CircularBufferSinkInfo();
         break;
    case common::SinkType_Device:
        info = new(pInfo) DeviceSinkInfo();
        break;
    case common::SinkType_Invalid:
        info = new(pInfo) SinkInfoBase();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_SDK_ASSERT_NOT_NULL(pInfo);
    NN_UNUSED(info);
}
}

nn::Result InfoUpdater::UpdateSinks(SinkContext& sinkContext, server::MemoryPoolInfo* pPoolInfos, int memoryPoolCount) NN_NOEXCEPT
{
    auto pInSinkInfo = reinterpret_cast<const ::nn::audio::common::SinkInParameter*>(m_InParameter);
    auto pOutStatus = reinterpret_cast<::nn::audio::common::SinkOutStatus*>(m_OutStatus);
    auto poolMapper = PoolMapper(m_ClientProcessHandle, pPoolInfos, memoryPoolCount, m_BehaviorInfo.IsMemoryPoolForceMappingEnabled());
    for (int i = 0; i < sinkContext.GetCount(); ++i)
    {

        SinkInfoBase* serverSide = &sinkContext.GetInfo(i);
        const common::SinkInParameter* userSide = &pInSinkInfo[i];
        common::SinkOutStatus& toUserOutStat(pOutStatus[i]);

        if (serverSide->GetType() != userSide->_type)
        {
            serverSide->CleanUp();
            ResetSinkInfo(serverSide, userSide->_type);
        }

        common::BehaviorParameter::ErrorInfo errorInfo;
        serverSide->Update(&errorInfo, &toUserOutStat, userSide, poolMapper);
        if (errorInfo.IsError())
        {
            m_BehaviorInfo.AppendError(errorInfo);
        }
    }

    auto consumedInParamSize = sizeof(nn::audio::common::SinkInParameter) * sinkContext.GetCount();
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::sink) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }
    auto consumuedOutStatusSize = sizeof(nn::audio::common::SinkOutStatus) * sinkContext.GetCount();
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::sink, consumuedOutStatusSize);

    m_InParameter += consumedInParamSize;
    m_OutStatus += consumuedOutStatusSize;

    NN_RESULT_SUCCESS;
}

Result InfoUpdater::UpdateMemoryPools(server::MemoryPoolInfo* pOutInfos, int memoryPoolCount) NN_NOEXCEPT
{
    auto pInParameter = reinterpret_cast<const nn::audio::MemoryPoolInfo::InParameter*>(m_InParameter);
    auto pOutStatus = reinterpret_cast<nn::audio::MemoryPoolInfo::OutStatus*>(m_OutStatus);
    server::PoolMapper clientMemoryMapper(m_ClientProcessHandle, m_BehaviorInfo.IsMemoryPoolForceMappingEnabled());
    for (auto i = 0; i < memoryPoolCount; ++i)
    {
        auto err = clientMemoryMapper.Update(&pOutInfos[i], &pInParameter[i], &pOutStatus[i]);
        switch (err)
        {
        case nn::audio::server::PoolMapper::Error_NoError:
        case nn::audio::server::PoolMapper::Error_FailedToMap:
        case nn::audio::server::PoolMapper::Error_FailedToUnmap:
            // Error_FailedToUnmap は利用中の MemoryPool を RequrstDetach した時に返りうる
            break;
        case nn::audio::server::PoolMapper::Error_InvalidAddressOrSize:
            // precondition violation
            return ResultInvalidUpdateInfo();
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    auto consumedInParamSize = sizeof(nn::audio::MemoryPoolInfo::InParameter) * memoryPoolCount;
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::memoryPool) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }
    auto consumuedOutStatusSize = sizeof(nn::audio::MemoryPoolInfo::OutStatus) * memoryPoolCount;
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::memoryPool, consumuedOutStatusSize);

    m_InParameter += consumedInParamSize;
    m_OutStatus += consumuedOutStatusSize;

    NN_RESULT_SUCCESS;
}

Result InfoUpdater::UpdatePerformanceBuffer(void* performance, size_t performanceSize, PerformanceManager* manager) NN_NOEXCEPT
{
    auto pInParameter = reinterpret_cast<const nn::audio::PerformanceBufferInfo::InParameter*>(m_InParameter);
    auto pOutStatus = reinterpret_cast<nn::audio::PerformanceBufferInfo::OutStatus*>(m_OutStatus);

    if (manager)
    {
        pOutStatus->consumedSize = static_cast<uint32_t>(manager->CopyHistories(performance, performanceSize));
        manager->SetDetailTarget(pInParameter->detailTarget);
    }
    else
    {
        pOutStatus->consumedSize = 0;
    }

    auto consumedInParamSize = sizeof(nn::audio::PerformanceBufferInfo::InParameter);
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::performance) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }
    auto consumuedOutStatusSize = sizeof(nn::audio::PerformanceBufferInfo::OutStatus);
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::performance, consumuedOutStatusSize);

    m_InParameter += consumedInParamSize;
    m_OutStatus += consumuedOutStatusSize;

    NN_RESULT_SUCCESS;
}

nn::Result InfoUpdater::UpdateBehaviorInfo(BehaviorInfo* pBehaviorInfo) NN_NOEXCEPT
{
    auto pInParameter = reinterpret_cast<const nn::audio::common::BehaviorParameter::InParameter*>(m_InParameter);

    if (common::CheckValidRevision(pInParameter->_revision) == false ||
        pBehaviorInfo->m_UserLibRevision != pInParameter->_revision)
    {
        return ResultInvalidUpdateInfo();
    }

    // Clear last errors
    pBehaviorInfo->ClearError();
    // Update user specified AudioRenderer configurations.
    pBehaviorInfo->UpdateFlags(pInParameter->_parameter);

    auto consumedInParamSize = sizeof(nn::audio::common::BehaviorParameter::InParameter);
    if (m_InParameterHeader->GetSize(&common::UpdateDataSizes::behavior) != consumedInParamSize)
    {
        return ResultInvalidUpdateInfo();
    }

    m_InParameter += consumedInParamSize;

    NN_RESULT_SUCCESS;
}

nn::Result InfoUpdater::UpdateErrorInfo(BehaviorInfo* pBehaviorInfo) NN_NOEXCEPT
{
    auto pOutStatus = reinterpret_cast<nn::audio::common::BehaviorParameter::OutStatus*>(m_OutStatus);

    // Copy error infos to user side.
    pBehaviorInfo->CopyErrorInfo(pOutStatus->_infos, &pOutStatus->_infoCount);

    auto consumuedOutStatusSize = sizeof(nn::audio::common::BehaviorParameter::OutStatus);
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::behavior, consumuedOutStatusSize);
    m_OutStatus += consumuedOutStatusSize;

    NN_RESULT_SUCCESS;
}

nn::Result InfoUpdater::CheckConsumedSize() NN_NOEXCEPT
{
    if ((m_InParameter - m_InParameterOrigin != m_InParameterSize) ||
        (m_OutStatus - m_OutStatusOrigin != m_OutStatusSize))
    {
        NN_DETAIL_AUDIO_WARN(
            "[audio] update info size are invalid\n"
            "[audio] InParameter:0x%p - 0x%p != (0x%08x) or OutStatus:0x%p - 0x%p != (0x%08x)\n",
            m_InParameter, m_InParameterOrigin, m_InParameterSize,
            m_OutStatus, m_OutStatusOrigin, m_OutStatusSize);
        return ResultInvalidUpdateInfo();
    }
    NN_RESULT_SUCCESS;
}

nn::Result InfoUpdater::UpdateSplitterInfo(SplitterContext& sendContext) NN_NOEXCEPT
{
    size_t consumed = 0;
    if (sendContext.Update(m_InParameter, consumed) == false)
    {
        return ResultInvalidUpdateInfo();
    }

    m_InParameter += consumed;
    NN_RESULT_SUCCESS;
}

Result InfoUpdater::UpdateRendererInfo(int64_t elapsedFrameCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(elapsedFrameCount, 0);

    auto pRendererInfo = reinterpret_cast<RendererInfo*>(m_OutStatus);

    pRendererInfo->elapsedFrameCount = elapsedFrameCount;

    const auto infoSize = sizeof(*pRendererInfo);

    m_OutStatus += infoSize;
    m_OutStatusHeader->SetSize(&common::UpdateDataSizes::rendererInfo, infoSize);

    NN_RESULT_SUCCESS;
}

}}}  // namespace nn::audio::server
