﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_AudioOutTypes.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/audio/detail/audio_IAudioOutManager.h>
#include <nn/audio/detail/audio_IAudioOut.h>

namespace nn { namespace audio {  namespace server {

class AudioOutManagerImpl : public nn::sf::ISharedObject
{
public:
    AudioOutManagerImpl() NN_NOEXCEPT;
    virtual ~AudioOutManagerImpl() NN_NOEXCEPT;
    nn::Result OpenAudioOut(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioOut>> outAudioOut, nn::sf::InBuffer name, nn::audio::AudioOutParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result OpenAudioOutAuto(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioOut>> outAudioOut, nn::sf::InBuffer name, nn::audio::AudioOutParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result ListAudioOuts(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT;
    nn::Result ListAudioOutsAuto(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT;
    void ReportCloseSession(int sessionId) NN_NOEXCEPT;
    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;
    nn::os::Mutex* GetMutex() NN_NOEXCEPT;

public:
    static const int NumberOfSessions = 12;
    static const int SessionStackSize = 12 * 1024;

private:
    static void ThreadFunc(void* arg) NN_NOEXCEPT;
    void ThreadFuncImpl() NN_NOEXCEPT;
    nn::Result OpenAudioOutInternal(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioOut>> outAudioOut, nn::sf::InBuffer name, nn::audio::AudioOutParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioOutParameterInternal> audioOutInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result ListAudioOutsInternal(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT;

private:
    class AudioOutImpl;

    typedef nn::sf::ExpHeapAllocator MyAllocator;
    MyAllocator m_Allocator;

    //TODO SIGLO-71683: Move AudioOutImpl definition to a header, so sizeof(AudioOutImpl) can be used here.
    //For now, using MemoryPageSize. I confirmed that sizeof(AudioOutImpl) = 2120, so heap will have sufficient space.
    static const int HeapSizeForSessions = NumberOfSessions * nn::os::MemoryPageSize;
    static const int HeapBufferSize = HeapSizeForSessions + SessionStackSize;

    std::aligned_storage<HeapBufferSize>::type m_HeapBuffer;
    nn::lmem::HeapHandle m_HeapHandle;

    bool m_IsInitialized;

    class SessionsCircularBuffer
    {
    public:
        SessionsCircularBuffer()
            : m_NextAvailableIndex(0)
            , m_NextFreeIndex(0)
            , m_CurrentAmount(NumberOfSessions)
        {
            for( int i = 0; i < NumberOfSessions; ++i )
            {
                m_AvailableSessions[i] = i;
            }
        }

        nn::Result GetNextSessions(int* sessions)
        {
            NN_RESULT_THROW_UNLESS(m_CurrentAmount > 0, ResultOutOfResource());
            *sessions = m_AvailableSessions[m_NextAvailableIndex];
            m_NextAvailableIndex = ( m_NextAvailableIndex + 1 ) % NumberOfSessions;
            --m_CurrentAmount;
            NN_RESULT_SUCCESS;
        }

        void InsertSession(int sessionId)
        {
            NN_SDK_ASSERT(sessionId >= 0);
            NN_SDK_ASSERT(sessionId < NumberOfSessions);
            NN_SDK_ASSERT(m_CurrentAmount < NumberOfSessions, "Can't hold more sessions (max %d)", NumberOfSessions);

            m_AvailableSessions[m_NextFreeIndex] = sessionId;
            m_NextFreeIndex = ( m_NextFreeIndex + 1 ) % NumberOfSessions;
            ++m_CurrentAmount;
        }
    private:

        int m_AvailableSessions[NumberOfSessions];
        int m_NextAvailableIndex;
        int m_NextFreeIndex;
        int m_CurrentAmount;
    };

    SessionsCircularBuffer m_Sessions;

    char*            m_Stack;

    nn::os::SemaphoreType* m_pSemaphore;
    AudioOutImpl*          m_Session[NumberOfSessions];
    nn::applet::AppletResourceUserId m_AppletId[NumberOfSessions];
    nn::os::EventType      m_EventSemaphoreSignal;
    nn::os::Mutex     m_Mutex;

    std::atomic<bool> m_IsThreadActive;
    os::ThreadType    m_Thread;
    bool              m_IsAwake;

};

}}}  // namespace nn::audio::server
