﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <atomic>
#include <mutex>
#include <limits>

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/audio/audio_Result.h>

#include "../dsp/audio_Dsp.h"
#include "audio_AppletVolumeManager.h"

#define NN_AUDIO_APPLET_LOG(...)
#if !defined(NN_AUDIO_APPLET_LOG)
#include <nn/audio/detail/audio_Log.h>
#define NN_AUDIO_APPLET_LOG(format, ...) NN_DETAIL_AUDIO_TRACE("[audio][applet:%s] " format, __func__, ## __VA_ARGS__)
#endif

namespace nn { namespace audio {

namespace {

const float DefaultAppletVolume = 1.0f;
const float DefaultVolume = 1.0f;
const float DefaultDeviceVolume = 1.0f;
const float DefaultRecordVolume = 0.0f;

class LerpTarget
{
public:
    LerpTarget() NN_NOEXCEPT;
    void Start(float startVolume, float endVolume, int64_t duration, nn::os::Event* pEvent = nullptr) NN_NOEXCEPT;
    bool Lerp(float* outVolume) NN_NOEXCEPT;
    bool IsActive() const NN_NOEXCEPT;
    float GetEndVolume() const NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

private:
    nn::os::Event* m_TransferedEvent;
    int64_t m_StartTime;
    int64_t m_LerpDurationTicks;
    float m_StartVolume;
    float m_EndVolume;
    bool m_IsActive;
    nn::os::Mutex m_Mutex;
};

class DeviceSwitchEvent : public nn::util::IntrusiveListBaseNode<DeviceSwitchEvent>
{
public:
    DeviceSwitchEvent() NN_NOEXCEPT;
    ~DeviceSwitchEvent() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    bool IsInitialized() const NN_NOEXCEPT;
    nn::os::NativeHandle GetReadableHandle() const NN_NOEXCEPT;
    void Signal() NN_NOEXCEPT;

private:
    nn::os::SystemEventType m_SystemEvent;
    bool m_Initialized; // nn::os::SystemEventType._state を確認する API が存在しないため初期化状況を確認するように準備
};
using DeviceSwitchEventList = nn::util::IntrusiveList<DeviceSwitchEvent, nn::util::IntrusiveListBaseNodeTraits<DeviceSwitchEvent>>;

class AudioDeviceCompanion
{
public:
    enum class State
    {
        NoProcess,
        FadeOut,
        FadeIn,
    };

    using EventId = int;
    static const EventId InvalidEventId = -1;

    AudioDeviceCompanion() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    int GetDeviceIndex() const NN_NOEXCEPT;
    void SetDeviceIndex(int deviceIndex) NN_NOEXCEPT;
    int GetCurrentChannelCount() const NN_NOEXCEPT;
    void SetCurrentChannelCount(int channelCount) NN_NOEXCEPT;

    bool StartFadeOut(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT;
    bool StartFadeIn(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT;
    float GetMasterVolume() const NN_NOEXCEPT;
    bool LerpMasterVolume() NN_NOEXCEPT;
    void SignalDeviceSwitch() NN_NOEXCEPT;
    void SignalInputNotification() NN_NOEXCEPT;
    void SignalOutputNotification() NN_NOEXCEPT;

    EventId AllocDeviceSwitchEvent() NN_NOEXCEPT;
    EventId AllocDeviceInputEvent() NN_NOEXCEPT;
    EventId AllocDeviceOutputEvent() NN_NOEXCEPT;
    void FreeDeviceSwitchEvent(EventId allocationid) NN_NOEXCEPT;
    void FreeDeviceInputEvent(EventId allocationid) NN_NOEXCEPT;
    void FreeDeviceOutputEvent(EventId allocationid) NN_NOEXCEPT;
    nn::os::NativeHandle GetDeviceSwitchEventHandle(EventId allocationId) NN_NOEXCEPT;
    nn::os::NativeHandle GetDeviceInputEventHandle(EventId allocationId) NN_NOEXCEPT;
    nn::os::NativeHandle GetDeviceOutputEventHandle(EventId allocationId) NN_NOEXCEPT;

private:
    std::atomic<int> m_DeviceIndex;
    std::atomic<int> m_ChannelCount;
    std::atomic<float> m_DeviceMasterVolume;
    State m_State;
    LerpTarget m_LerpTarget;

    static const int DeviceEventCountMax = AppletVolumeManager::MaxRegisteredApplets;
    nn::os::Mutex m_EventAllocationMutex;
    DeviceSwitchEvent m_DeviceSwitchEvents[DeviceEventCountMax];
    DeviceSwitchEvent m_DeviceInputEvents[DeviceEventCountMax];
    DeviceSwitchEvent m_DeviceOutputEvents[DeviceEventCountMax];
    DeviceSwitchEventList m_RegisteredEvents;
    DeviceSwitchEventList m_RegisteredInputEvents;
    DeviceSwitchEventList m_RegisteredOutputEvents;
};
NN_DEFINE_STATIC_CONSTANT(const int AudioDeviceCompanion::DeviceEventCountMax);

struct AudioTypePermissions
{
    AudioTypePermissions(bool suspend = false, bool setVolume = false) NN_NOEXCEPT
        : canSuspend(suspend), canSetVolume(setVolume), suspendDisabled(false)
    {}
    bool canSuspend;
    bool canSetVolume;
    bool suspendDisabled;
};

struct AppletInfo
{
    int referenceCount;
    nn::applet::AppletResourceUserId id;
    float volume[AppletVolumeManager::NumSessionTypes];
    float recordVolume[AppletVolumeManager::NumSessionTypes];
    float deviceVolume[AppletVolumeManager::NumSessionTypes][AppletVolumeManager::MaxDevices];
    AudioDeviceCompanion::EventId deviceEventId;
    AudioDeviceCompanion::EventId inputEventId;
    AudioDeviceCompanion::EventId outputEventId;

    bool isSuspended[AppletVolumeManager::SessionType_Count];
    bool isSuspendedForDebug[AppletVolumeManager::SessionType_Count];

    AppletInfo() NN_NOEXCEPT
    {
        Clear();
    }

    void Clear()
    {
        referenceCount = 0 ;
        id = nn::applet::AppletResourceUserId::GetInvalidId();
        deviceEventId = AudioDeviceCompanion::InvalidEventId;
        inputEventId = AudioDeviceCompanion::InvalidEventId;
        outputEventId = AudioDeviceCompanion::InvalidEventId;
        for (auto& v : volume)
        {
            v = DefaultAppletVolume;
        }
        for (auto& v : recordVolume)
        {
            v = DefaultRecordVolume;
        }
        for (auto& volumes : deviceVolume)
        {
            for (auto& v : volumes)
            {
                v = DefaultDeviceVolume;
            }
        }
        for (auto& v : isSuspended)
        {
            v = false;
        }
        for (auto& v : isSuspendedForDebug)
        {
            v = false;
        }
    }
};

class Session
{
public:
    Session() NN_NOEXCEPT;
    void Initialize(AppletVolumeManager::SessionType type, int index, const AudioTypePermissions& permissions, AudioDeviceCompanion* deviceCompanion) NN_NOEXCEPT;
    void Open(const nn::applet::AppletResourceUserId& id, float volume, const float* pDeviceVolumes, bool isGmixSession) NN_NOEXCEPT;
    void Close() NN_NOEXCEPT;
    void StartSuspend(int64_t durationNano) NN_NOEXCEPT;
    void StartResume(int64_t durationNano) NN_NOEXCEPT;
    void StartSetAppletVolume(float volume, int64_t durationNano) NN_NOEXCEPT;
    void SetVolume(float volume) NN_NOEXCEPT;
    float GetVolume() const NN_NOEXCEPT;
    void SetRecordVolume(float volume, int64_t durationNano) NN_NOEXCEPT;
    void SuspendForDebugger(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    void ResumeForDebugger(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    float GetAppletVolume() const NN_NOEXCEPT;
    float GetSavedAppletVolume() const NN_NOEXCEPT;
    void SetDeviceVolume(AppletVolumeManager::DeviceType deviceIndex, float deviceVolume) NN_NOEXCEPT;
    float GetDeviceVolume(AppletVolumeManager::DeviceType deviceIndex) const NN_NOEXCEPT;
    AppletVolumeManager::SessionType GetType() const NN_NOEXCEPT;
    AppletVolumeManager::SessionState GetState() const NN_NOEXCEPT;
    bool IsOpen(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    bool IsOpen() NN_NOEXCEPT;
    bool Lerp() NN_NOEXCEPT;
    void Sleep() NN_NOEXCEPT;
    void Wake() NN_NOEXCEPT;
    void UpdateVolume(uint32_t durationMircoSeconds) const NN_NOEXCEPT;
    void Mute() const NN_NOEXCEPT;
    void Dump() NN_NOEXCEPT;
    void DisableSuspend(bool isSuspended) NN_NOEXCEPT;
    bool IsSuspendedForDebugger() const NN_NOEXCEPT;
private:
    void ResumeInnerSession() NN_NOEXCEPT;
    void SuspendInnerSession() NN_NOEXCEPT;
    void ResumeInnerSessionForDebugger() NN_NOEXCEPT;
    void SuspendInnerSessionForDebugger() NN_NOEXCEPT;

    struct ReservedState
    {
        AppletVolumeManager::SessionState state;
        int64_t durationNano;
    } m_ReservedState;

    LerpTarget m_LerpTarget;
    AppletVolumeManager::SessionType m_Type;
    AppletVolumeManager::SessionState m_State;
    bool m_IsSuspended;
    bool m_IsSuspendedForDebugger;
    nn::applet::AppletResourceUserId m_Id;
    int m_Index;
    float m_AppletVolume;
    float m_SavedAppletVolume;
    float m_RecordVolume;
    float m_Volume;
    float m_DeviceVolumes[AppletVolumeManager::MaxDevices];
    bool m_IsResuming;
    bool m_IsSuspending;
    bool m_IsGmixSession;
    AudioDeviceCompanion* m_DeviceCompanion;
    AudioTypePermissions  m_Permissions;
};

class AppletVolumeManagerImpl
{
    //////////////////////////////////////////////////////////////////////////
    // self controls
    //////////////////////////////////////////////////////////////////////////
public:
    AppletVolumeManagerImpl() NN_NOEXCEPT;
    Result Initialize() NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;
    void Sleep() NN_NOEXCEPT;
    void Wake() NN_NOEXCEPT;
    void Dump() NN_NOEXCEPT;

private:
    bool IsAsleep() const NN_NOEXCEPT;
    bool IsSessionRunning() const NN_NOEXCEPT;
    void SignalThread() NN_NOEXCEPT;
    void Wait() NN_NOEXCEPT;

    AppletInfo* FindRegisteredAppletInfo(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;

    //////////////////////////////////////////////////////////////////////////
    // Session controls
    //////////////////////////////////////////////////////////////////////////
public:
    Result RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;
    Result UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;
    bool IsAppletRegistered(nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

    void OpenSession(AppletVolumeManager::SessionType type, int index, const nn::applet::AppletResourceUserId& id, bool isGmixSession) NN_NOEXCEPT;
    void CloseSession(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT;
    Result SuspendSession(AppletVolumeManager::SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result ResumeSession(AppletVolumeManager::SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result ForceResumeSession(AppletVolumeManager::SessionType type, int sessionId, bool shouldMute, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    bool IsSessionRunning(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT;

    Result SetAppletVolume(AppletVolumeManager::SessionType type, float volume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result GetAppletVolume(float* outVolume, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result SetSessionRecordVolume(AppletVolumeManager::SessionType type, float volume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result GetSessionRecordVolume(float* outVolume, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result SetSessionVolume(AppletVolumeManager::SessionType type, int index, float volume, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    Result GetSessionVolume(float* outVolume, AppletVolumeManager::SessionType type, int index, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    void SetSessionDeviceVolume(float volume, AppletVolumeManager::DeviceType device, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    float GetSessionDeviceVolume(AppletVolumeManager::DeviceType device, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;

    void SetDeviceIndex(int deviceIndex, int channelCount) NN_NOEXCEPT;
    int GetDeviceIndex() const NN_NOEXCEPT;
    void QueryAudioDeviceSystemEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    void QueryAudioDeviceInputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    void QueryAudioDeviceOutputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    bool StartDeviceFadeOut(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT;
    bool StartDeviceFadeIn(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT;
    float GetDeviceMasterVolume() const NN_NOEXCEPT;
    int GetDeviceChannelCount() const NN_NOEXCEPT;

    void SignalInputEvents() NN_NOEXCEPT;
    void SignalOutputEvents() NN_NOEXCEPT;

    void SuspendForDebugger(AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;
    void ResumeForDebugger(AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT;

private:
    void RefreshVolumes(uint32_t durationMicro = 0) NN_NOEXCEPT;
    int LerpTargets() NN_NOEXCEPT;
    int GetMaxSessions(AppletVolumeManager::SessionType type) const NN_NOEXCEPT;
    Session* GetSessions(AppletVolumeManager::SessionType type) NN_NOEXCEPT;

    NN_OS_ALIGNAS_THREAD_STACK uint8_t m_ThreadStack[12 * 1024];
    nn::os::Mutex                      m_Mutex;
    nn::os::Mutex                      m_AppletMutex;
    nn::os::Event                      m_StartEvent;
    nn::os::ThreadType                 m_Thread;
    Session                            m_OutSessions[AppletVolumeManager::MaxOutSessions];
    Session                            m_RendererSessions[AppletVolumeManager::MaxRendererSessions];
    Session                            m_InSessions[AppletVolumeManager::MaxInSessions];
    Session                            m_RecorderSessions[AppletVolumeManager::MaxRecorderSessions];
    AppletInfo                         m_RegisteredApplets[AppletVolumeManager::MaxRegisteredApplets];
    int                                m_NumRegisteredApplets;
    std::atomic<bool>                  m_IsRunning;
    std::atomic<bool>                  m_IsAsleep;
    AudioDeviceCompanion               m_DeviceCompanion;
    nn::os::Event                      m_LerpCompleteEvent;

public:
    static void LerpThreadFunc(void* vManager) NN_NOEXCEPT;
};

// globals
AppletVolumeManagerImpl g_VolumeManagerImpl;

void LerpTarget::Stop() NN_NOEXCEPT
{
    if (m_TransferedEvent)
    {
        m_TransferedEvent->Signal();
    }
    m_IsActive = false;
}

float LerpTarget::GetEndVolume() const NN_NOEXCEPT
{
    return m_EndVolume;
}

bool LerpTarget::IsActive() const NN_NOEXCEPT
{
    return m_IsActive;
}

bool LerpTarget::Lerp(float* outVolume) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsActive);

    auto now = nn::os::GetSystemTick().GetInt64Value();
    if (now >= m_StartTime + m_LerpDurationTicks)
    {
        *outVolume = m_EndVolume;
        Stop();
        return false;
    }

    auto scale = static_cast<float>(now - m_StartTime) / m_LerpDurationTicks;
    auto dif = m_EndVolume - m_StartVolume;
    *outVolume = m_StartVolume + (dif * scale);

    return *outVolume != m_EndVolume;
}

void LerpTarget::Start(float startVolume, float endVolume, int64_t duration, nn::os::Event* pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_IsActive)
    {
        Stop();
    }

    m_TransferedEvent = pEvent;
    m_StartVolume = startVolume;
    m_EndVolume = endVolume;
    m_StartTime = nn::os::GetSystemTick().GetInt64Value();
    m_LerpDurationTicks = nn::os::ConvertToTick(nn::TimeSpan::FromNanoSeconds(duration)).GetInt64Value();
    m_IsActive = true;
}

LerpTarget::LerpTarget() NN_NOEXCEPT
    : m_TransferedEvent(nullptr)
    , m_StartTime(0ull)
    , m_LerpDurationTicks(0ull)
    , m_StartVolume(0.0f)
    , m_EndVolume(0.0f)
    , m_IsActive(false)
    , m_Mutex(false)
{
}

DeviceSwitchEvent::DeviceSwitchEvent() NN_NOEXCEPT
    : m_SystemEvent()
    , m_Initialized(false)
{
}

DeviceSwitchEvent::~DeviceSwitchEvent() NN_NOEXCEPT
{
    Finalize();
}

void DeviceSwitchEvent::Initialize() NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        nn::os::CreateSystemEvent(&m_SystemEvent, nn::os::EventClearMode_AutoClear, true);
        m_Initialized = true;
    }
}

void DeviceSwitchEvent::Finalize() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        nn::os::DestroySystemEvent(&m_SystemEvent);
        m_Initialized = false;
    }
}

bool DeviceSwitchEvent::IsInitialized() const NN_NOEXCEPT
{
    return m_Initialized;
}

nn::os::NativeHandle DeviceSwitchEvent::GetReadableHandle() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    return (m_Initialized) ?
        nn::os::GetReadableHandleOfSystemEvent(&m_SystemEvent) :
        nn::os::InvalidNativeHandle;
}

void DeviceSwitchEvent::Signal() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        nn::os::SignalSystemEvent(&m_SystemEvent);
    }
}

bool AudioDeviceCompanion::LerpMasterVolume() NN_NOEXCEPT
{
    if (m_LerpTarget.IsActive())
    {
        float volume;
        bool running = m_LerpTarget.Lerp(&volume);
        m_DeviceMasterVolume = volume;
        if (!running)
        {
            switch (m_State)
            {
            case State::FadeOut:
            case State::FadeIn:
                m_State = State::NoProcess;
                break;
            case State::NoProcess:
            default:
                break;
            }
        }
        return running;
    }

    return false;
}

float AudioDeviceCompanion::GetMasterVolume() const NN_NOEXCEPT
{
    return m_DeviceMasterVolume;
}

bool AudioDeviceCompanion::StartFadeIn(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT
{
    if (m_State == State::FadeIn || m_State == State::FadeOut)
    {
        return false;
    }
    m_State = State::FadeIn;
    m_LerpTarget.Start(0.0f, 1.0f, durationNano, pEvent);
    m_DeviceMasterVolume = 0.0f;
    return true;
}

bool AudioDeviceCompanion::StartFadeOut(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT
{
    if (m_State == State::FadeIn || m_State == State::FadeOut)
    {
        return false;
    }
    m_State = State::FadeOut;
    m_LerpTarget.Start(1.0f, 0.0f, durationNano, pEvent);
    m_DeviceMasterVolume = 1.0f;
    return true;
}

void AudioDeviceCompanion::SetDeviceIndex(int deviceIndex) NN_NOEXCEPT
{
    m_DeviceIndex.store(deviceIndex);
}

int AudioDeviceCompanion::GetDeviceIndex() const NN_NOEXCEPT
{
    return m_DeviceIndex.load();
}

AudioDeviceCompanion::AudioDeviceCompanion() NN_NOEXCEPT
    : m_EventAllocationMutex{false}
{
    Initialize();
}

void AudioDeviceCompanion::Initialize() NN_NOEXCEPT
{
    m_DeviceIndex = 0;
    m_DeviceMasterVolume = 1.0f;
    m_State = State::NoProcess;

    m_RegisteredEvents.clear();
    m_RegisteredInputEvents.clear();
    m_RegisteredOutputEvents.clear();
    for (auto& events : m_DeviceSwitchEvents)
    {
        if (events.IsInitialized())
        {
            events.Finalize();
        }
    }

    for (auto& events : m_DeviceInputEvents)
    {
        if (events.IsInitialized())
        {
            events.Finalize();
        }
    }

    for (auto& events : m_DeviceOutputEvents)
    {
        if (events.IsInitialized())
        {
            events.Finalize();
        }
    }
}

void AudioDeviceCompanion::SignalDeviceSwitch() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    for (auto& itr : m_RegisteredEvents)
    {
        itr.Signal();
    }
}

void AudioDeviceCompanion::SignalInputNotification() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    for (auto& itr : m_RegisteredInputEvents)
    {
        itr.Signal();
    }
}

void AudioDeviceCompanion::SignalOutputNotification() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    for (auto& itr : m_RegisteredOutputEvents)
    {
        itr.Signal();
    }
}

int AudioDeviceCompanion::GetCurrentChannelCount() const NN_NOEXCEPT
{
    return m_ChannelCount;
}

void AudioDeviceCompanion::SetCurrentChannelCount(int channelCount) NN_NOEXCEPT
{
    m_ChannelCount = channelCount;
}

AudioDeviceCompanion::EventId AudioDeviceCompanion::AllocDeviceSwitchEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    // TODO: 線形探索の削減
    for (auto idx = 0; idx < DeviceEventCountMax; ++idx)
    {
        if (m_DeviceSwitchEvents[idx].IsInitialized() == false)
        {
            m_DeviceSwitchEvents[idx].Initialize();
            m_RegisteredEvents.push_back(m_DeviceSwitchEvents[idx]);
            return static_cast<AudioDeviceCompanion::EventId>(idx); // 現状配列 index がそのまま id となる。
        }
    }
    return InvalidEventId;
}

AudioDeviceCompanion::EventId AudioDeviceCompanion::AllocDeviceInputEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    // TODO: 線形探索の削減
    for (auto idx = 0; idx < DeviceEventCountMax; ++idx)
    {
        if (m_DeviceInputEvents[idx].IsInitialized() == false)
        {
            m_DeviceInputEvents[idx].Initialize();
            m_RegisteredInputEvents.push_back(m_DeviceInputEvents[idx]);
            return static_cast<AudioDeviceCompanion::EventId>(idx); // 現状配列 index がそのまま id となる。
        }
    }
    return InvalidEventId;
}

AudioDeviceCompanion::EventId AudioDeviceCompanion::AllocDeviceOutputEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    // TODO: 線形探索の削減
    for (auto idx = 0; idx < DeviceEventCountMax; ++idx)
    {
        if (m_DeviceOutputEvents[idx].IsInitialized() == false)
        {
            m_DeviceOutputEvents[idx].Initialize();
            m_RegisteredOutputEvents.push_back(m_DeviceOutputEvents[idx]);
            return static_cast<AudioDeviceCompanion::EventId>(idx); // 現状配列 index がそのまま id となる。
        }
    }
    return InvalidEventId;
}

void AudioDeviceCompanion::FreeDeviceSwitchEvent(AudioDeviceCompanion::EventId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= id && id < DeviceEventCountMax);
    NN_SDK_ASSERT(m_DeviceSwitchEvents[id].IsInitialized() && m_DeviceSwitchEvents[id].IsLinked());

    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    if (id < 0 ||
        DeviceEventCountMax <= id ||
        m_DeviceSwitchEvents[id].IsInitialized() == false ||
        m_DeviceSwitchEvents[id].IsLinked() == false)
    {
        return;
    }

    m_RegisteredEvents.erase(m_RegisteredEvents.iterator_to(m_DeviceSwitchEvents[id]));
    m_DeviceSwitchEvents[id].Finalize();
}

void AudioDeviceCompanion::FreeDeviceInputEvent(AudioDeviceCompanion::EventId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= id && id < DeviceEventCountMax);
    NN_SDK_ASSERT(m_DeviceInputEvents[id].IsInitialized() && m_DeviceInputEvents[id].IsLinked());

    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    if (id < 0 ||
        DeviceEventCountMax <= id ||
        m_DeviceInputEvents[id].IsInitialized() == false ||
        m_DeviceInputEvents[id].IsLinked() == false)
    {
        return;
    }

    m_RegisteredInputEvents.erase(m_RegisteredInputEvents.iterator_to(m_DeviceInputEvents[id]));
    m_DeviceInputEvents[id].Finalize();
}

void AudioDeviceCompanion::FreeDeviceOutputEvent(AudioDeviceCompanion::EventId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= id && id < DeviceEventCountMax);
    NN_SDK_ASSERT(m_DeviceOutputEvents[id].IsInitialized() && m_DeviceOutputEvents[id].IsLinked());

    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    if (id < 0 ||
        DeviceEventCountMax <= id ||
        m_DeviceOutputEvents[id].IsInitialized() == false ||
        m_DeviceOutputEvents[id].IsLinked() == false)
    {
        return;
    }

    m_RegisteredOutputEvents.erase(m_RegisteredOutputEvents.iterator_to(m_DeviceOutputEvents[id]));
    m_DeviceOutputEvents[id].Finalize();
}

nn::os::NativeHandle AudioDeviceCompanion::GetDeviceSwitchEventHandle(AudioDeviceCompanion::EventId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= id && id < DeviceEventCountMax);
    NN_SDK_ASSERT(m_DeviceSwitchEvents[id].IsInitialized() && m_DeviceSwitchEvents[id].IsLinked());

    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    if (id < 0 ||
        DeviceEventCountMax <= id ||
        m_DeviceSwitchEvents[id].IsInitialized() == false ||
        m_DeviceSwitchEvents[id].IsLinked() == false)
    {
        return nn::os::InvalidNativeHandle;
    }

    return m_DeviceSwitchEvents[id].GetReadableHandle();
}

nn::os::NativeHandle AudioDeviceCompanion::GetDeviceInputEventHandle(AudioDeviceCompanion::EventId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= id && id < DeviceEventCountMax);
    NN_SDK_ASSERT(m_DeviceInputEvents[id].IsInitialized() && m_DeviceInputEvents[id].IsLinked());

    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    if (id < 0 ||
        DeviceEventCountMax <= id ||
        m_DeviceInputEvents[id].IsInitialized() == false ||
        m_DeviceInputEvents[id].IsLinked() == false)
    {
        return nn::os::InvalidNativeHandle;
    }

    return m_DeviceInputEvents[id].GetReadableHandle();
}

nn::os::NativeHandle AudioDeviceCompanion::GetDeviceOutputEventHandle(AudioDeviceCompanion::EventId id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 <= id && id < DeviceEventCountMax);
    NN_SDK_ASSERT(m_DeviceOutputEvents[id].IsInitialized() && m_DeviceOutputEvents[id].IsLinked());

    std::lock_guard<nn::os::Mutex> lock(m_EventAllocationMutex);

    if (id < 0 ||
        DeviceEventCountMax <= id ||
        m_DeviceOutputEvents[id].IsInitialized() == false ||
        m_DeviceOutputEvents[id].IsLinked() == false)
    {
        return nn::os::InvalidNativeHandle;
    }

    return m_DeviceOutputEvents[id].GetReadableHandle();
}

void Session::SuspendInnerSessionForDebugger() NN_NOEXCEPT
{
    if (!m_IsSuspendedForDebugger && !m_IsSuspended && m_Permissions.canSuspend && !m_Permissions.suspendDisabled && m_IsGmixSession)
    {
        NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d)\n", m_Type, m_Index);
        dsp::SessionSuspend(m_Type, m_Index);
        m_IsSuspended = true;
    }
    m_IsSuspendedForDebugger = true;
}

void Session::ResumeInnerSessionForDebugger() NN_NOEXCEPT
{
    if (m_IsSuspendedForDebugger && m_IsSuspended && m_Permissions.canSuspend && m_IsGmixSession)
    {
        NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d)\n", m_Type, m_Index);
        dsp::SessionResume(m_Type, m_Index);
        m_IsSuspended = false;
    }
    m_IsSuspendedForDebugger = false;
}

void Session::SuspendInnerSession() NN_NOEXCEPT
{
    if (!m_IsSuspendedForDebugger && !m_IsSuspended && m_State == AppletVolumeManager::SessionState_Running && m_Permissions.canSuspend && !m_Permissions.suspendDisabled && m_IsGmixSession)
    {
        NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d)\n", m_Type, m_Index);
        dsp::SessionSuspend(m_Type, m_Index);
        m_IsSuspended = true;
    }
    m_State = AppletVolumeManager::SessionState_Suspended;
}

void Session::ResumeInnerSession() NN_NOEXCEPT
{
    // a workaround for SIGLO-64030
    // call UpdateVolume(0) here because if AVM suspends session when gmix is doing volume fading, gmix holds a halfway volume.
    if (m_State == AppletVolumeManager::SessionState_Suspended)
    {
        UpdateVolume(0);
    }

    if (!m_IsSuspendedForDebugger && m_IsSuspended && m_State == AppletVolumeManager::SessionState_Suspended && m_Permissions.canSuspend && m_IsGmixSession)
    {
        NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d)\n", m_Type, m_Index);
        dsp::SessionResume(m_Type, m_Index);
        m_IsSuspended = false;
    }
    m_State = AppletVolumeManager::SessionState_Running;
}

void Session::Mute() const NN_NOEXCEPT
{
    if (m_Permissions.canSetVolume && m_IsGmixSession)
    {
        NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d) volume(%d) durationUsec(%u)\n", m_Type, m_Index, 0.f, 0);
        dsp::SetVolume(m_Type, m_Index, 0.f, 0);
    }
}

void Session::UpdateVolume(uint32_t durationMircoSeconds) const NN_NOEXCEPT
{
    if (m_Permissions.canSetVolume && m_IsGmixSession)
    {
        const auto volume = m_AppletVolume * m_DeviceVolumes[m_DeviceCompanion->GetDeviceIndex()] * m_DeviceCompanion->GetMasterVolume() * m_Volume;
        NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d) volume(%.2f) durationUsec(%u) m_Id.lower(%llu)\n", m_Type, m_Index, volume, durationMircoSeconds, m_Id.lower);
        dsp::SetVolume(m_Type, m_Index, volume, durationMircoSeconds);

        const auto recordVolume = m_RecordVolume * m_DeviceVolumes[m_DeviceCompanion->GetDeviceIndex()];
        dsp::SetRecordVolume(m_Type, m_Index, recordVolume, durationMircoSeconds);
    }
}

void Session::Wake() NN_NOEXCEPT
{
    if (m_State != AppletVolumeManager::SessionState_Closed)
    {
        if(!m_IsGmixSession)
        {
            return;
        }
        // Sync suspend state with gmix.
        if((m_State == AppletVolumeManager::SessionState_Suspended || m_IsSuspendedForDebugger) && m_Permissions.canSuspend)
        {
            NN_AUDIO_APPLET_LOG("m_Type(%d) m_Index(%d)", m_Type, m_Index);
            dsp::SessionSuspend(m_Type, m_Index);
            m_IsSuspended = true;
        }
    }
}

void Session::Sleep() NN_NOEXCEPT
{

}

bool Session::Lerp() NN_NOEXCEPT
{
    if (m_LerpTarget.IsActive())
    {
        float volume;
        bool running = m_LerpTarget.Lerp(&volume);
        m_AppletVolume = volume;
        if (!running)
        {
            if (m_IsResuming)
            {
                m_State = AppletVolumeManager::SessionState_Running;
            }
            else if (m_IsSuspending)
            {
                SuspendInnerSession();
            }

            if (m_ReservedState.state != AppletVolumeManager::SessionState_None)
            {
                NN_AUDIO_APPLET_LOG("re-trigger Suspend/Resume [%x]\n", m_ReservedState.state);
                if (m_ReservedState.state == AppletVolumeManager::SessionState_Running && m_IsSuspending)
                {
                    // Resume requested while Suspending
                    m_IsResuming = m_IsSuspending = false;
                    StartResume(m_ReservedState.durationNano);
                    running = true;
                }
                else if (m_ReservedState.state == AppletVolumeManager::SessionState_Suspended && m_IsResuming)
                {
                    // Suspend requested while Resuming
                    m_IsResuming = m_IsSuspending = false;
                    StartSuspend(m_ReservedState.durationNano);
                    running = true;
                }
                else
                {
                    NN_AUDIO_APPLET_LOG("unexpected condition %x, suspend:%d, resume:%d\n", m_ReservedState, m_IsSuspending, m_IsResuming);
                    m_IsResuming = m_IsSuspending = false;
                    // clear reserved state
                    m_ReservedState.state = AppletVolumeManager::SessionState_None;
                    m_ReservedState.durationNano = 0;
                }
            }
            else
            {
                m_IsResuming = m_IsSuspending = false;
            }
        }
        return running;
    }
    return false;
}

bool Session::IsOpen() NN_NOEXCEPT
{
    return m_State != AppletVolumeManager::SessionState_Closed;
}

bool Session::IsOpen(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return m_State != AppletVolumeManager::SessionState_Closed && m_Id == id;
}

nn::audio::AppletVolumeManager::SessionState Session::GetState() const NN_NOEXCEPT
{
    return m_State;
}

nn::audio::AppletVolumeManager::SessionType Session::GetType() const NN_NOEXCEPT
{
    return m_Type;
}

float Session::GetDeviceVolume(AppletVolumeManager::DeviceType deviceIndex) const NN_NOEXCEPT
{
    return m_DeviceVolumes[deviceIndex];
}

void Session::SetDeviceVolume(AppletVolumeManager::DeviceType deviceIndex, float deviceVolume) NN_NOEXCEPT
{
    m_DeviceVolumes[deviceIndex] = deviceVolume;
    UpdateVolume(0);
}

float Session::GetSavedAppletVolume() const NN_NOEXCEPT
{
    return m_SavedAppletVolume;
}

float Session::GetAppletVolume() const NN_NOEXCEPT
{
    return m_AppletVolume;
}

void Session::ResumeForDebugger(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    if (m_IsSuspendedForDebugger && m_Id == id)
    {
        ResumeInnerSessionForDebugger();
    }
}

void Session::SuspendForDebugger(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    if (!m_IsSuspendedForDebugger && m_Id == id)
    {
        SuspendInnerSessionForDebugger();
    }
}

void Session::StartSetAppletVolume(float volume, int64_t durationNano) NN_NOEXCEPT
{
    if (m_State == AppletVolumeManager::SessionState_Running)
    {
        m_LerpTarget.Start(m_AppletVolume, volume, durationNano);
    }
    // apply volume even when the session is suspended
    m_SavedAppletVolume = volume;
}

void Session::SetVolume(float volume) NN_NOEXCEPT
{
    // apply volume even when the session is suspended
    m_Volume = volume;
    if (m_State == AppletVolumeManager::SessionState_Running)
    {
        UpdateVolume(0);
    }
}

float Session::GetVolume() const NN_NOEXCEPT
{
    return m_Volume;
}

void Session::SetRecordVolume(float volume, int64_t durationNano) NN_NOEXCEPT
{
    m_RecordVolume = volume;
    if(m_IsGmixSession)
    {
        const auto v = volume * m_DeviceVolumes[m_DeviceCompanion->GetDeviceIndex()];
        dsp::SetRecordVolume(m_Type, m_Index, v, static_cast<uint32_t>(durationNano / 1000));
    }
}

void Session::StartResume(int64_t durationNano) NN_NOEXCEPT
{
    if (!m_IsResuming && m_IsSuspending == false)
    {
        m_IsResuming = true;
        m_IsSuspending = false;
        ResumeInnerSession();
        m_LerpTarget.Start(m_AppletVolume, m_SavedAppletVolume, durationNano);
        m_ReservedState.state = AppletVolumeManager::SessionState_None;
        m_ReservedState.durationNano = 0;
        NN_AUDIO_APPLET_LOG("normal call m_SavedAppletVolume %f\n", m_SavedAppletVolume);
    }
    else
    {
        m_ReservedState.state = AppletVolumeManager::SessionState_Running;
        m_ReservedState.durationNano = durationNano;
        NN_AUDIO_APPLET_LOG("called while Suspending!%s\n", "");
    }
}

void Session::StartSuspend(int64_t durationNano) NN_NOEXCEPT
{
    if (m_State != AppletVolumeManager::SessionState_Suspended && m_IsResuming == false && m_IsSuspending == false)
    {
        if (m_LerpTarget.IsActive())
        {
            m_AppletVolume = m_LerpTarget.GetEndVolume();
        }
        m_IsSuspending = true;
        m_IsResuming = false;
        ResumeInnerSession();
        m_LerpTarget.Start(m_AppletVolume, 0.0f, durationNano);
        m_ReservedState.state = AppletVolumeManager::SessionState_None;
        m_ReservedState.durationNano = 0;
        NN_AUDIO_APPLET_LOG("normal call m_SavedAppletVolume%f\n", m_SavedAppletVolume);
    }
    else
    {
        m_ReservedState.state = AppletVolumeManager::SessionState_Suspended;
        m_ReservedState.durationNano = durationNano;
        NN_AUDIO_APPLET_LOG("called while Suspending!%s\n", "");
    }
}

void Session::Close() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State != AppletVolumeManager::SessionState_Closed);
    m_State = AppletVolumeManager::SessionState_Closed;
    m_Id = nn::applet::AppletResourceUserId::GetInvalidId();
    m_LerpTarget.Stop();
}

void Session::Open(const nn::applet::AppletResourceUserId& id, float volume, const float* pDeviceVolumes, bool isGmixSession) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_State == AppletVolumeManager::SessionState_Closed);
    m_State = AppletVolumeManager::SessionState_Running;
    m_Id = id;
    m_IsSuspendedForDebugger = false;
    m_IsSuspended = false;
    m_AppletVolume = volume;
    m_SavedAppletVolume = volume;
    m_Volume = DefaultVolume;
    m_IsGmixSession = isGmixSession;
    for (auto i = 0; i < AppletVolumeManager::MaxDevices; ++i)
    {
        m_DeviceVolumes[i] = pDeviceVolumes[i];
    }
    UpdateVolume(0);
    DisableSuspend(false);
    m_ReservedState.state = AppletVolumeManager::SessionState_None;
    m_ReservedState.durationNano = 0;
    m_IsSuspending = false;
    m_IsResuming = false;
}

void Session::Initialize(AppletVolumeManager::SessionType type, int index, const AudioTypePermissions& permissions, AudioDeviceCompanion* deviceCompanion) NN_NOEXCEPT
{
    m_Type = type;
    m_Index = index;
    m_Permissions = permissions;
    m_DeviceCompanion = deviceCompanion;
    m_State = AppletVolumeManager::SessionState_Closed;
    m_Id = nn::applet::AppletResourceUserId::GetInvalidId();
    m_AppletVolume = DefaultAppletVolume;
    m_SavedAppletVolume = DefaultAppletVolume;
    m_Volume = DefaultVolume;

    m_ReservedState.state = AppletVolumeManager::SessionState_None;
    m_ReservedState.durationNano = 0;
}

void Session::Dump() NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("m_Type(%d) m_State(%d) m_Id.lower(%llu) m_Index(%d) m_AppletVolume(%.2f) m_SavedAppletVolume(%.2f) m_IsResuming(%d) m_IsSuspending(%d), m_IsSuspendedForDebugger(%d)\n",
        m_Type, m_State, m_Id.lower, m_Index, m_AppletVolume, m_SavedAppletVolume, m_IsResuming, m_IsSuspending, m_IsSuspendedForDebugger);
}

void Session::DisableSuspend(bool isSuspended) NN_NOEXCEPT
{
    m_Permissions.suspendDisabled = isSuspended;
}

bool Session::IsSuspendedForDebugger() const NN_NOEXCEPT
{
    return m_IsSuspendedForDebugger;
}

Session::Session() NN_NOEXCEPT
    : m_State(AppletVolumeManager::SessionState_Closed)
    , m_IsSuspended(false)
    , m_IsSuspendedForDebugger(false)
    , m_Id(nn::applet::AppletResourceUserId::GetInvalidId())
    , m_AppletVolume(DefaultAppletVolume)
    , m_SavedAppletVolume(DefaultAppletVolume)
    , m_RecordVolume(DefaultRecordVolume)
    , m_Volume(DefaultVolume)
{
    for (auto& vol : m_DeviceVolumes)
    {
        vol = DefaultDeviceVolume;
    }
}

void AppletVolumeManagerImpl::SignalThread() NN_NOEXCEPT
{
    m_StartEvent.Signal();
}

Session* AppletVolumeManagerImpl::GetSessions(AppletVolumeManager::SessionType type) NN_NOEXCEPT
{
    switch (type)
    {
    case AppletVolumeManager::SessionType_AudioRenderer:
        return &m_RendererSessions[0];
    case AppletVolumeManager::SessionType_AudioOut:
        return &m_OutSessions[0];
    case AppletVolumeManager::SessionType_AudioIn:
        return &m_InSessions[0];
    case AppletVolumeManager::SessionType_FinalOutputRecorder:
        return &m_RecorderSessions[0];
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int AppletVolumeManagerImpl::GetMaxSessions(AppletVolumeManager::SessionType type) const NN_NOEXCEPT
{
    switch (type)
    {
    case AppletVolumeManager::SessionType_AudioRenderer:
        return AppletVolumeManager::MaxRendererSessions;
    case AppletVolumeManager::SessionType_AudioOut:
        return AppletVolumeManager::MaxOutSessions;
    case AppletVolumeManager::SessionType_AudioIn:
        return AppletVolumeManager::MaxInSessions;
    case AppletVolumeManager::SessionType_FinalOutputRecorder:
        return AppletVolumeManager::MaxRecorderSessions;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void AppletVolumeManagerImpl::LerpThreadFunc(void* vManager) NN_NOEXCEPT
{
    AppletVolumeManagerImpl* manager = reinterpret_cast<AppletVolumeManagerImpl*>(vManager);
    while (manager->IsSessionRunning())
    {
        manager->Wait();
        if (!manager->IsSessionRunning())
        {
            break;
        }

        int count = 0;
        bool running = false;
        do
        {
            count = manager->LerpTargets();
            running = manager->m_DeviceCompanion.LerpMasterVolume();
            auto durationMicro = nn::TimeSpan::FromMilliSeconds(AppletVolumeManager::LerpIntervalMilliseconds).GetMicroSeconds();
            NN_SDK_ASSERT(std::numeric_limits<uint32_t>::max() > durationMicro);
            manager->RefreshVolumes(static_cast<uint32_t>(durationMicro));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(AppletVolumeManager::LerpIntervalMilliseconds));
        } while (running || count > 0);

        manager->m_LerpCompleteEvent.Signal();
    }
}

bool AppletVolumeManagerImpl::IsAppletRegistered(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_AppletMutex);
    return FindRegisteredAppletInfo(id) != nullptr;
}

bool AppletVolumeManagerImpl::IsSessionRunning(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);

    NN_UNUSED(maxSessions);
    NN_SDK_ASSERT(index < maxSessions);
    return sessions[index].GetState() == AppletVolumeManager::SessionState_Running && !sessions[index].IsSuspendedForDebugger();
}

bool AppletVolumeManagerImpl::IsSessionRunning() const NN_NOEXCEPT
{
    return m_IsRunning;
}

void AppletVolumeManagerImpl::CloseSession(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);

    NN_UNUSED(maxSessions);
    NN_SDK_ASSERT(index < maxSessions);
    sessions[index].Close();
}

void AppletVolumeManagerImpl::OpenSession(AppletVolumeManager::SessionType type, int index, const nn::applet::AppletResourceUserId& id, bool isGmixSession) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);

    NN_UNUSED(maxSessions);
    NN_SDK_ASSERT(index < maxSessions);
    float volume = DefaultAppletVolume;
    float recordVolume = DefaultRecordVolume;
    float deviceVolume[AppletVolumeManager::MaxDevices];
    for (auto& v : deviceVolume)
    {
        v = DefaultDeviceVolume;
    }
    bool shouldSuspend = false;
    bool shouldSuspendForDebug = false;
    {
        std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);
        for (int i = 0; i < m_NumRegisteredApplets; ++i)
        {
            if (m_RegisteredApplets[i].id == id)
            {
                volume = m_RegisteredApplets[i].volume[type];
                recordVolume = m_RegisteredApplets[i].recordVolume[type];
                for (auto j = 0; j < AppletVolumeManager::MaxDevices; ++j)
                {
                    deviceVolume[j] = m_RegisteredApplets[i].deviceVolume[type][j];
                }
                shouldSuspend = m_RegisteredApplets[i].isSuspended[type];
                shouldSuspendForDebug = m_RegisteredApplets[i].isSuspendedForDebug[type];
                break;
            }
        }
    }
    sessions[index].Open(id, volume, deviceVolume, isGmixSession);
    sessions[index].SetRecordVolume(recordVolume, 0);
    if (shouldSuspend)
    {
        sessions[index].StartSuspend(0);
        SignalThread();
    }
    if (shouldSuspendForDebug)
    {
        sessions[index].SuspendForDebugger(id);
    }
}

void AppletVolumeManagerImpl::Wait() NN_NOEXCEPT
{
    m_StartEvent.Wait();
}

AppletInfo* AppletVolumeManagerImpl::FindRegisteredAppletInfo(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    for (auto i = 0; i < m_NumRegisteredApplets; ++i)
    {
        if (m_RegisteredApplets[i].id == id)
        {
            return &m_RegisteredApplets[i];
        }
    }
    return nullptr;
}

bool AppletVolumeManagerImpl::IsAsleep() const NN_NOEXCEPT
{
    return m_IsAsleep;
}

int AppletVolumeManagerImpl::LerpTargets() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(!m_IsRunning)
    {
        return 0;
    }

    int count = 0;
    for (int i = 0; i < AppletVolumeManager::MaxRendererSessions; ++i)
    {
        if (m_RendererSessions[i].Lerp())
        {
            ++count;
        }
    }
    for (int i = 0; i < AppletVolumeManager::MaxOutSessions; ++i)
    {
        if (m_OutSessions[i].Lerp())
        {
            ++count;
        }
    }
    for (int i = 0; i < AppletVolumeManager::MaxInSessions; ++i)
    {
        if (m_InSessions[i].Lerp())
        {
            ++count;
        }
    }
    for( int i = 0; i < AppletVolumeManager::MaxRecorderSessions; ++i )
    {
        if(m_RecorderSessions[i].Lerp())
        {
            ++count;
        }
    }
    return count;
}

void AppletVolumeManagerImpl::ResumeForDebugger(AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    if (IsAsleep())
    {
        // skip operation.
        return;
    }

    {
        std::lock_guard<os::Mutex> lock(m_AppletMutex);
        auto info = FindRegisteredAppletInfo(id);
        if (info)
        {
            info->isSuspendedForDebug[type] = false;
        }
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);

    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].ResumeForDebugger(id);
        }
    }
    SignalThread();
}

void AppletVolumeManagerImpl::SuspendForDebugger(AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    if (IsAsleep())
    {
        // skip operation.
        return;
    }

    {
        std::lock_guard<os::Mutex> lock(m_AppletMutex);
        auto info = FindRegisteredAppletInfo(id);
        if (info)
        {
            info->isSuspendedForDebug[type] = true;
        }
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);

    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].SuspendForDebugger(id);
        }
    }
    SignalThread();
}

void AppletVolumeManagerImpl::RefreshVolumes(uint32_t durationMicro /*= 0*/) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    AppletVolumeManager::SessionType types[] = { AppletVolumeManager::SessionType_AudioOut, AppletVolumeManager::SessionType_AudioRenderer, AppletVolumeManager::SessionType_AudioIn };
    for (auto type : types)
    {
        Session* sessions = GetSessions(type);
        int maxSessions = GetMaxSessions(type);
        for (int i = 0; i < maxSessions; ++i)
        {
            if (sessions[i].IsOpen())
            {
                sessions[i].UpdateVolume(durationMicro);
            }
        }
    }
}

float AppletVolumeManagerImpl::GetSessionDeviceVolume(AppletVolumeManager::DeviceType device, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    float rt = 1.0f;
    std::lock_guard<nn::os::Mutex> lock(m_AppletMutex);
    for (int i = 0; i < m_NumRegisteredApplets; ++i)
    {
        if (m_RegisteredApplets[i].id == id)
        {
            rt = m_RegisteredApplets[i].deviceVolume[type][device];
            break;
        }
    }
    return rt;
}

void AppletVolumeManagerImpl::SetSessionDeviceVolume(float volume, AppletVolumeManager::DeviceType device, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    bool appletFound = false;
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    {
        std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);
        for (int i = 0; i < m_NumRegisteredApplets; ++i)
        {
            if (m_RegisteredApplets[i].id == id)
            {
                m_RegisteredApplets[i].deviceVolume[type][device] = volume;
                appletFound = true;
                break;
            }
        }
    }
    if (appletFound == false)
    {
        return;
    }

    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);
    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].SetDeviceVolume(device, volume);
        }
    }
}

nn::Result AppletVolumeManagerImpl::GetSessionRecordVolume(float* outVolume, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_AppletMutex);
    for (int i = 0; i < m_NumRegisteredApplets; ++i)
    {
        if (m_RegisteredApplets[i].id == id)
        {
            *outVolume = m_RegisteredApplets[i].recordVolume[type];
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultAppletResourceUserIdNotFound());
}

nn::Result AppletVolumeManagerImpl::SetSessionRecordVolume(AppletVolumeManager::SessionType type, float volume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    {
        bool appletFound = false;
        std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);
        for (int i = 0; i < m_NumRegisteredApplets; ++i)
        {
            if (m_RegisteredApplets[i].id == id)
            {
                m_RegisteredApplets[i].recordVolume[type] = volume;
                appletFound = true;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(appletFound == true, ResultAppletResourceUserIdNotFound());
    }

    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);
    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].SetRecordVolume(volume, durationNano);
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::SetSessionVolume(AppletVolumeManager::SessionType type, int index, float volume, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    {
        bool appletFound = false;
        std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);
        for (int i = 0; i < m_NumRegisteredApplets; ++i)
        {
            if (m_RegisteredApplets[i].id == id)
            {
                appletFound = true;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(appletFound == true, ResultAppletResourceUserIdNotFound());
    }

    Session* sessions = GetSessions(type);
    sessions[index].SetVolume(volume);

    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::GetSessionVolume(float* outVolume, AppletVolumeManager::SessionType type, int index, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    {
        bool appletFound = false;
        std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);
        for (int i = 0; i < m_NumRegisteredApplets; ++i)
        {
            if (m_RegisteredApplets[i].id == id)
            {
                appletFound = true;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(appletFound == true, ResultAppletResourceUserIdNotFound());
    }

    Session* sessions = GetSessions(type);
    *outVolume = sessions[index].GetVolume();

    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::GetAppletVolume(float* outVolume, AppletVolumeManager::SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_AppletMutex);
    for (int i = 0; i < m_NumRegisteredApplets; ++i)
    {
        if (m_RegisteredApplets[i].id == id)
        {
            *outVolume = m_RegisteredApplets[i].volume[type];
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultAppletResourceUserIdNotFound());
}

nn::Result AppletVolumeManagerImpl::SetAppletVolume(AppletVolumeManager::SessionType type, float volume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    {
        bool appletFound = false;
        std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);
        for (int i = 0; i < m_NumRegisteredApplets; ++i)
        {
            if (m_RegisteredApplets[i].id == id)
            {
                m_RegisteredApplets[i].volume[type] = volume;
                appletFound = true;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(appletFound == true, ResultAppletResourceUserIdNotFound());
    }

    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);
    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].StartSetAppletVolume(volume, durationNano);
        }
    }
    SignalThread();
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::ResumeSession(AppletVolumeManager::SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAppletRegistered(id) == true, ResultAppletResourceUserIdNotFound());

    {
        std::lock_guard<os::Mutex> lock(m_AppletMutex);
        auto info = FindRegisteredAppletInfo(id);
        if (info)
        {
            info->isSuspended[type] = false;
        }
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);
    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].StartResume(durationNano);
        }
    }
    SignalThread();
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::ForceResumeSession(AppletVolumeManager::SessionType type, int sessionId, bool shouldMute, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAppletRegistered(id) == true, ResultAppletResourceUserIdNotFound());

    {
        std::lock_guard<os::Mutex> lock(m_AppletMutex);
        auto info = FindRegisteredAppletInfo(id);
        if (info)
        {
            info->isSuspended[type] = false;
        }
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);
    NN_RESULT_THROW_UNLESS(sessionId < maxSessions, ResultOperationFailed());

    if (sessions[sessionId].IsOpen(id))
    {
        if (sessions[sessionId].GetState() == AppletVolumeManager::SessionState_Suspended)
        {
            sessions[sessionId].StartResume(0);
        }
        else if (sessions[sessionId].IsSuspendedForDebugger())
        {
            sessions[sessionId].ResumeForDebugger(id);
        }
        if (shouldMute)
        {
            sessions[sessionId].Mute();
        }
        sessions[sessionId].DisableSuspend(true);
    }

    SignalThread();
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::SuspendSession(AppletVolumeManager::SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsAppletRegistered(id) == true, ResultAppletResourceUserIdNotFound());

    {
        std::lock_guard<os::Mutex> lock(m_AppletMutex);
        auto info = FindRegisteredAppletInfo(id);
        if (info)
        {
            info->isSuspended[type] = true;
        }
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    Session* sessions = GetSessions(type);
    int maxSessions = GetMaxSessions(type);
    for (int i = 0; i < maxSessions; ++i)
    {
        if (sessions[i].IsOpen(id))
        {
            sessions[i].StartSuspend(durationNano);
        }
    }
    SignalThread();
    NN_RESULT_SUCCESS;
}

void AppletVolumeManagerImpl::Wake() NN_NOEXCEPT
{
    if (m_IsAsleep == false)
    {
        return;
    }
    m_IsAsleep = false;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    for (int i = 0; i < AppletVolumeManager::MaxRendererSessions; ++i)
    {
        m_RendererSessions[i].Wake();
    }
    for (int i = 0; i < AppletVolumeManager::MaxOutSessions; ++i)
    {
        m_OutSessions[i].Wake();
    }
    for (int i = 0; i < AppletVolumeManager::MaxInSessions; ++i)
    {
        m_InSessions[i].Wake();
    }
    for (int i = 0; i < AppletVolumeManager::MaxRecorderSessions; ++i)
    {
        m_RecorderSessions[i].Wake();
    }
    SignalThread();
}

void AppletVolumeManagerImpl::Sleep() NN_NOEXCEPT
{
    if (m_IsAsleep)
    {
        return;
    }
    m_IsAsleep = true;

    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        for (int i = 0; i < AppletVolumeManager::MaxRendererSessions; ++i)
        {
            m_RendererSessions[i].Sleep();
        }
        for (int i = 0; i < AppletVolumeManager::MaxOutSessions; ++i)
        {
            m_OutSessions[i].Sleep();
        }
        for (int i = 0; i < AppletVolumeManager::MaxInSessions; ++i)
        {
            m_InSessions[i].Sleep();
        }
        for (int i = 0; i < AppletVolumeManager::MaxRecorderSessions; ++i)
        {
            m_RecorderSessions[i].Sleep();
        }
    }

    m_LerpCompleteEvent.Clear();
    SignalThread();
    m_LerpCompleteEvent.Wait();
}

nn::Result AppletVolumeManagerImpl::UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_AppletMutex);
    for (int i = 0; i < m_NumRegisteredApplets; ++i)
    {
        if (m_RegisteredApplets[i].id == id)
        {
            NN_SDK_ASSERT(m_RegisteredApplets[i].referenceCount >= 1);
            --m_RegisteredApplets[i].referenceCount;

            if (m_RegisteredApplets[i].referenceCount == 0)
            {
                m_DeviceCompanion.FreeDeviceSwitchEvent(m_RegisteredApplets[i].deviceEventId);
                m_DeviceCompanion.FreeDeviceInputEvent(m_RegisteredApplets[i].inputEventId);
                m_DeviceCompanion.FreeDeviceOutputEvent(m_RegisteredApplets[i].outputEventId);
                --m_NumRegisteredApplets;
                m_RegisteredApplets[i] = m_RegisteredApplets[m_NumRegisteredApplets];
                m_RegisteredApplets[m_NumRegisteredApplets].Clear();
            }
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultAppletResourceUserIdNotFound());
}

nn::Result AppletVolumeManagerImpl::RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_AppletMutex);
    if (m_NumRegisteredApplets == AppletVolumeManager::MaxRegisteredApplets)
    {
        NN_RESULT_THROW(ResultMaxAppletResourceUserId());
    }

    int appletIndex;
    for (appletIndex = 0; appletIndex < m_NumRegisteredApplets; ++appletIndex)
    {
        if (m_RegisteredApplets[appletIndex].id == id)
        {
            ++m_RegisteredApplets[appletIndex].referenceCount;
            NN_RESULT_SUCCESS;
        }
    }

    NN_SDK_ASSERT(m_RegisteredApplets[appletIndex].referenceCount == 0);
    ++m_NumRegisteredApplets;
    m_RegisteredApplets[appletIndex].referenceCount = 1;
    m_RegisteredApplets[appletIndex].id = id;

    for (int sessionIndex = 0; sessionIndex < AppletVolumeManager::NumSessionTypes; ++sessionIndex)
    {
        m_RegisteredApplets[appletIndex].volume[sessionIndex] = DefaultAppletVolume;
        for (auto i = 0; i < AppletVolumeManager::MaxDevices; ++i)
        {
            m_RegisteredApplets[appletIndex].deviceVolume[sessionIndex][i] = DefaultDeviceVolume;
        }
        m_RegisteredApplets[appletIndex].recordVolume[sessionIndex] = DefaultRecordVolume;
    }

    m_RegisteredApplets[appletIndex].deviceEventId = m_DeviceCompanion.AllocDeviceSwitchEvent();
    m_RegisteredApplets[appletIndex].inputEventId = m_DeviceCompanion.AllocDeviceInputEvent();
    m_RegisteredApplets[appletIndex].outputEventId = m_DeviceCompanion.AllocDeviceOutputEvent();

    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::Finalize() NN_NOEXCEPT
{
    if(!m_IsRunning)
    {
        NN_RESULT_SUCCESS;
    }

    m_IsRunning = false;
    SignalThread();
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManagerImpl::Initialize() NN_NOEXCEPT
{
    if(m_IsRunning)
    {
        NN_RESULT_SUCCESS;
    }

    AudioTypePermissions rendererPermissions(false, true);
    for (int i = 0; i < AppletVolumeManager::MaxRendererSessions; ++i)
    {
        m_RendererSessions[i].Initialize(AppletVolumeManager::SessionType_AudioRenderer, i, rendererPermissions, &m_DeviceCompanion);
    }

    AudioTypePermissions outPermissions(true, true);
    for (int i = 0; i < AppletVolumeManager::MaxOutSessions; ++i)
    {
        m_OutSessions[i].Initialize(AppletVolumeManager::SessionType_AudioOut, i, outPermissions, &m_DeviceCompanion);
    }

    AudioTypePermissions inPermissions(true, true);
    for (int i = 0; i < AppletVolumeManager::MaxInSessions; ++i)
    {
        m_InSessions[i].Initialize(AppletVolumeManager::SessionType_AudioIn, i, inPermissions, &m_DeviceCompanion);
    }

    AudioTypePermissions recordPermissions (true, false);
    for( int i = 0; i < AppletVolumeManager::MaxRecorderSessions; ++i )
    {
        m_RecorderSessions[i].Initialize(AppletVolumeManager::SessionType_FinalOutputRecorder, i, recordPermissions, &m_DeviceCompanion);
    }

    for(auto& info : m_RegisteredApplets)
    {
        info.Clear();
    }
    m_DeviceCompanion.Initialize();

    m_NumRegisteredApplets = 0;
    m_StartEvent.Clear();
    m_IsAsleep = false;
    m_LerpCompleteEvent.Clear();

    m_IsRunning = true;
    nn::os::CreateThread(&m_Thread, LerpThreadFunc, this, m_ThreadStack, sizeof(m_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, AppletVolumeManager));
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audio, AppletVolumeManager));
    nn::os::StartThread(&m_Thread);
    NN_RESULT_SUCCESS;
}

AppletVolumeManagerImpl::AppletVolumeManagerImpl() NN_NOEXCEPT : m_Mutex(false), m_AppletMutex(false), m_StartEvent(nn::os::EventClearMode_AutoClear), m_NumRegisteredApplets(0), m_IsRunning(false), m_IsAsleep(false), m_LerpCompleteEvent(nn::os::EventClearMode_AutoClear)
{
}

void AppletVolumeManagerImpl::SetDeviceIndex(int deviceIndex, int channelCount) NN_NOEXCEPT
{
    m_DeviceCompanion.SetDeviceIndex(deviceIndex);
    m_DeviceCompanion.SetCurrentChannelCount(channelCount);
    RefreshVolumes();

    m_DeviceCompanion.SignalDeviceSwitch();
    m_DeviceCompanion.SignalOutputNotification();
}

bool AppletVolumeManagerImpl::StartDeviceFadeOut(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT
{
    auto rt = m_DeviceCompanion.StartFadeOut(pEvent, durationNano);
    SignalThread();
    return rt;
}

bool AppletVolumeManagerImpl::StartDeviceFadeIn(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT
{
    auto rt = m_DeviceCompanion.StartFadeIn(pEvent, durationNano);
    SignalThread();
    return rt;
}

float AppletVolumeManagerImpl::GetDeviceMasterVolume() const NN_NOEXCEPT
{
    return m_DeviceCompanion.GetMasterVolume();
}

int AppletVolumeManagerImpl::GetDeviceChannelCount() const NN_NOEXCEPT
{
    return m_DeviceCompanion.GetCurrentChannelCount();
}

void AppletVolumeManagerImpl::SignalInputEvents() NN_NOEXCEPT
{
    m_DeviceCompanion.SignalInputNotification();
}

void AppletVolumeManagerImpl::SignalOutputEvents() NN_NOEXCEPT
{
    m_DeviceCompanion.SignalOutputNotification();
}

int AppletVolumeManagerImpl::GetDeviceIndex() const NN_NOEXCEPT
{
    return m_DeviceCompanion.GetDeviceIndex();
}

void AppletVolumeManagerImpl::QueryAudioDeviceSystemEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    std::lock_guard<nn::os::Mutex> appletLock(m_AppletMutex);

    // find registered index
    int idx = 0;
    for (; idx < m_NumRegisteredApplets; ++idx)
    {
        if (m_RegisteredApplets[idx].id == id)
        {
            break;
        }
    }
    NN_SDK_ASSERT(idx < m_NumRegisteredApplets);
    if (idx >= m_NumRegisteredApplets)
    {
        *outHandle = nn::os::InvalidNativeHandle;
        return;
    }

    *outHandle = m_DeviceCompanion.GetDeviceSwitchEventHandle(m_RegisteredApplets[idx].deviceEventId);
}

void AppletVolumeManagerImpl::QueryAudioDeviceInputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    // find registered index
    int idx = 0;
    for (; idx < m_NumRegisteredApplets; ++idx)
    {
        if (m_RegisteredApplets[idx].id == id)
        {
            break;
        }
    }
    NN_SDK_ASSERT(idx < m_NumRegisteredApplets);
    if (idx >= m_NumRegisteredApplets)
    {
        *outHandle = nn::os::InvalidNativeHandle;
        return;
    }

    *outHandle = m_DeviceCompanion.GetDeviceInputEventHandle(m_RegisteredApplets[idx].inputEventId);
}

void AppletVolumeManagerImpl::QueryAudioDeviceOutputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    // find registered index
    int idx = 0;
    for (; idx < m_NumRegisteredApplets; ++idx)
    {
        if (m_RegisteredApplets[idx].id == id)
        {
            break;
        }
    }
    NN_SDK_ASSERT(idx < m_NumRegisteredApplets);
    if (idx >= m_NumRegisteredApplets)
    {
        *outHandle = nn::os::InvalidNativeHandle;
        return;
    }

    *outHandle = m_DeviceCompanion.GetDeviceOutputEventHandle(m_RegisteredApplets[idx].outputEventId);
}

void AppletVolumeManagerImpl::Dump() NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("m_NumRegisteredApplets(%d) m_IsRunning(%d) m_IsAsleep(%d)\n",
            m_NumRegisteredApplets,
            m_IsRunning.load(),
            m_IsAsleep.load());

    for (int i = 0; i < AppletVolumeManager::MaxRendererSessions; ++i)
    {
        m_RendererSessions[i].Dump();
    }

    for (int i = 0; i < AppletVolumeManager::MaxOutSessions; ++i)
    {
        m_OutSessions[i].Dump();
    }

    for (int i = 0; i < AppletVolumeManager::MaxInSessions; ++i)
    {
        m_InSessions[i].Dump();
    }

    for( int i = 0; i < AppletVolumeManager::MaxRecorderSessions; ++i )
    {
        m_RecorderSessions[i].Dump();
    }
}

} //namespace


Result AppletVolumeManager::Initialize() NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG();
    return g_VolumeManagerImpl.Initialize();
}

Result AppletVolumeManager::Finalize() NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG();
    return g_VolumeManagerImpl.Finalize();
}

Result AppletVolumeManager::RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("id.lower(%llu)\n", id.lower);
    return g_VolumeManagerImpl.RegisterAppletResourceUserId(id);
}

Result AppletVolumeManager::UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("id.lower(%llu)\n", id.lower);
    return g_VolumeManagerImpl.UnregisterAppletResourceUserId(id);
}

Result AppletVolumeManager::Sleep() NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG();
    g_VolumeManagerImpl.Sleep();
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::Wake() NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG();
    g_VolumeManagerImpl.Wake();
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::Suspend(SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) durationNano(%lld) id.lower(%llu)\n", type, durationNano, id.lower);
    return g_VolumeManagerImpl.SuspendSession(type, durationNano, id);
}

Result AppletVolumeManager::Resume(SessionType type, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) durationNano(%lld) id.lower(%llu)\n", type, durationNano, id.lower);
    return g_VolumeManagerImpl.ResumeSession(type, durationNano, id);
}

Result AppletVolumeManager::ForceResume(SessionType type, int sessionId, bool shouldMute, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return g_VolumeManagerImpl.ForceResumeSession(type, sessionId, shouldMute, id);
}

Result AppletVolumeManager::SetAppletVolume(SessionType type, float endVolume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) endVolume(%.2f) durationNano(%lld) id.lower(%llu)\n", type, endVolume, durationNano, id.lower);
    return g_VolumeManagerImpl.SetAppletVolume(type, endVolume, durationNano, id);
}

Result AppletVolumeManager::GetAppletVolume(float* outVolume, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return g_VolumeManagerImpl.GetAppletVolume(outVolume, type, id);
}

Result AppletVolumeManager::SetRecordVolume(SessionType type, float endVolume, int64_t durationNano, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) endRecordVolume(%.2f) durationNano(%lld) id.lower(%llu)\n", type, endVolume, durationNano, id.lower);
    return g_VolumeManagerImpl.SetSessionRecordVolume(type, endVolume, durationNano, id);
}

Result AppletVolumeManager::GetRecordVolume(float* outVolume, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return g_VolumeManagerImpl.GetSessionRecordVolume(outVolume, type, id);
}

Result AppletVolumeManager::SetVolume(SessionType type, int sessionId, float volume, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) sessionId(%d) volume(%.2f) id.lower(%llu)\n", type, sessionId, volume, id.lower);
    return g_VolumeManagerImpl.SetSessionVolume(type, sessionId, volume, id);
}

Result AppletVolumeManager::GetVolume(float* outVolume, SessionType type, int sessionId, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) sessionId(%d) volume(%.2f) id.lower(%llu)\n", type, sessionId, id.lower);
    return g_VolumeManagerImpl.GetSessionVolume(outVolume, type, sessionId, id);
}

Result AppletVolumeManager::SetDevice(AppletVolumeManager::DeviceType device, int channelCount) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("device(%d) channelCount(%d)\n", device, channelCount);
    g_VolumeManagerImpl.SetDeviceIndex(device, channelCount);
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::SetDeviceVolume(float volume, AppletVolumeManager::DeviceType device, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("volume(%.2f) device(%d) type(%d) id.lower(%llu)\n", volume, device, type, id.lower);
    g_VolumeManagerImpl.SetSessionDeviceVolume(volume, device, type, id);
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::GetDeviceVolume(float* outVolume, AppletVolumeManager::DeviceType device, SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    *outVolume = g_VolumeManagerImpl.GetSessionDeviceVolume(device, type, id);
    NN_RESULT_SUCCESS;
}

bool AppletVolumeManager::StartDeviceFadeOut(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("durationNano(%lld)\n", durationNano);
    return g_VolumeManagerImpl.StartDeviceFadeOut(pEvent, durationNano);
}

bool AppletVolumeManager::StartDeviceFadeIn(nn::os::Event* pEvent, int64_t durationNano) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("durationNano(%lld)\n", durationNano);
    return g_VolumeManagerImpl.StartDeviceFadeIn(pEvent, durationNano);
}

Result AppletVolumeManager::GetActiveDevice(AppletVolumeManager::DeviceType* pDevice) NN_NOEXCEPT
{
    *pDevice = static_cast<AppletVolumeManager::DeviceType>(g_VolumeManagerImpl.GetDeviceIndex());
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManager::QueryDeviceEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    g_VolumeManagerImpl.QueryAudioDeviceSystemEvent(outHandle, id);
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManager::QueryInputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    g_VolumeManagerImpl.QueryAudioDeviceInputNotificationEvent(outHandle, id);
    NN_RESULT_SUCCESS;
}

nn::Result AppletVolumeManager::QueryOutputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    g_VolumeManagerImpl.QueryAudioDeviceOutputNotificationEvent(outHandle, id);
    NN_RESULT_SUCCESS;
}

float AppletVolumeManager::GetDeviceMasterVolume() NN_NOEXCEPT
{
    return g_VolumeManagerImpl.GetDeviceMasterVolume();
}

int AppletVolumeManager::GetDeviceChannelCount() NN_NOEXCEPT
{
    return g_VolumeManagerImpl.GetDeviceChannelCount();
}

void AppletVolumeManager::SignalInputNotification() NN_NOEXCEPT
{
    g_VolumeManagerImpl.SignalInputEvents();
}

void AppletVolumeManager::SignalOutputNotification() NN_NOEXCEPT
{
    g_VolumeManagerImpl.SignalOutputEvents();
}

Result AppletVolumeManager::SuspendForDebugger(SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) id.lower(%llu)\n", type, id.lower);
    g_VolumeManagerImpl.SuspendForDebugger(type, id);
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::ResumeForDebugger(SessionType type, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) id.lower(%llu)\n", type, id.lower);
    g_VolumeManagerImpl.ResumeForDebugger(type, id);
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::OpenSession(SessionType type, int index, const nn::applet::AppletResourceUserId& id, bool isGmixSession) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) index(%d) id.lower(%llu)\n", type, index, id.lower);
    g_VolumeManagerImpl.OpenSession(type, index, id, isGmixSession);
    NN_RESULT_SUCCESS;
}

Result AppletVolumeManager::CloseSession(SessionType type, int index) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_LOG("type(%d) index(%d)\n", type, index);
    g_VolumeManagerImpl.CloseSession(type, index);
    NN_RESULT_SUCCESS;
}

bool AppletVolumeManager::IsRunning(SessionType type, int index) NN_NOEXCEPT
{
    return g_VolumeManagerImpl.IsSessionRunning(type, index);
}

void AppletVolumeManager::Dump() NN_NOEXCEPT
{
    g_VolumeManagerImpl.Dump();
}
}}  // namespace nn::audio
