﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../audio_DspCommon.h"
#include "../audio_EffectReverb.h"
#include "../audio_DspUtility.h"
#include "../audio_EffectReverbCoefficients.h"

namespace nn { namespace audio { namespace detail {

namespace {
void ProcessFeedback(qf * temps, qf * fdnFeedback, FXDelayLine * fdnDelay, qf * fdnLowPassHistory, qf * fdnLowPassCoef)
{
    // for (int i = 0; (i < ReverbParameter::FeedbackDelayNetworkCount); i++)
    // {
        // // attenuation:
        // temps[i] = MultiQfQf(fdnFeedback[i], DelayLineNextOut(&fdnDelay[i]));
        // // low-pass filter here:
        // temps[i] += MultiQfQf(fdnLowPassHistory[i], fdnLowPassCoef[i]);
        // fdnLowPassHistory[i] = temps[i];
    // }
    //unroll ReverbParameter::FeedbackDelayNetworkCount times (4)
    uint32_t fdn = 0;
    temps[fdn] = MultiQfQf(fdnFeedback[fdn], DelayLineNextOut(&fdnDelay[fdn]));
    temps[fdn] += MultiQfQf(fdnLowPassHistory[fdn], fdnLowPassCoef[fdn]);
    fdnLowPassHistory[fdn] = temps[fdn];
    ++fdn;
    temps[fdn] = MultiQfQf(fdnFeedback[fdn], DelayLineNextOut(&fdnDelay[fdn]));
    temps[fdn] += MultiQfQf(fdnLowPassHistory[fdn], fdnLowPassCoef[fdn]);
    fdnLowPassHistory[fdn] = temps[fdn];
    ++fdn;
    temps[fdn] = MultiQfQf(fdnFeedback[fdn], DelayLineNextOut(&fdnDelay[fdn]));
    temps[fdn] += MultiQfQf(fdnLowPassHistory[fdn], fdnLowPassCoef[fdn]);
    fdnLowPassHistory[fdn] = temps[fdn];
    ++fdn;
    temps[fdn] = MultiQfQf(fdnFeedback[fdn], DelayLineNextOut(&fdnDelay[fdn]));
    temps[fdn] += MultiQfQf(fdnLowPassHistory[fdn], fdnLowPassCoef[fdn]);
    fdnLowPassHistory[fdn] = temps[fdn];
}

void MixFeedbackMatrix(qf * outMatrix, qf * inMatrix)
{
    // Jot/Stautner/Puckette mixing matrix
    outMatrix[0] = inMatrix[1] + inMatrix[2];
    outMatrix[1] = -inMatrix[0] - inMatrix[3];
    outMatrix[2] = inMatrix[0] - inMatrix[3];
    outMatrix[3] = inMatrix[1] - inMatrix[2];
}
} // anonymous namespace

NN_FORCEINLINE void ApplyReverbEffect1ch(const ReverbParameter * reverb, ReverbState* state, const int sampleCount, const int32_t* pInData1, int32_t* pOutData1)
{
    qf outputs[nn::audio::ReverbParameter::ChannelCountMax];
    qf temps2[ReverbParameter::FeedbackDelayNetworkCount];
    qf fdnLowPassHistory[ReverbParameter::FeedbackDelayNetworkCount] = {
        state->_fdnLowPassHistory[0],
        state->_fdnLowPassHistory[1],
        state->_fdnLowPassHistory[2],
        state->_fdnLowPassHistory[3] }; // this needs to be restored after the loop below

    // early delay parameters
    FXDelayLine* earlyDelay = &state->_earlyDelay;
    uint32_t *earlyTap = &state->_earlyTap[0];
    qf *earlyCoef = &state->_earlyCoef[0];
    uint32_t lateTap = state->_lateTap;

    // FDN parameters
    qf *fdnFeedback = &state->_fdnFeedback[0];
    FXDelayLine *fdnDelay = &state->_fdnDelay[0];
    qf *fdnLowPassCoef = &state->_fdnLowPassCoef[0];
    FXDelayLine *allPass = &state->_allPass[0];

    const qf *base = earlyDelay->buff;
    const uint32_t delay = earlyDelay->delay;

    for (int j = 0; (j < sampleCount); j++)
    {
        outputs[0] = QF_CONST(0.0f); // zero outputs
        qf* in = earlyDelay->p;
        //early reflections
        // for (int i = 0; i < ReverbParameter::EarlyReflectionTaps; ++i)
        // {
            // outputs[0] += MultiQfQf(DelayLineTapOut(earlyDelay, earlyTap[i]), earlyCoef[i]);
        // }
        // unroll early reflections loop ReverbParameter::EarlyReflectionTaps times (10)
        uint32_t taps = 0;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        // get input samples (as int32_t)
        qf in_samp = MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData1[j]));
        DelayLineTick(earlyDelay, in_samp);

        // late reverb
        qf late_in = MultiQfQf(DelayLineTapOut(earlyDelay, lateTap), reverb->_lateGain);

        qf temps[ReverbParameter::FeedbackDelayNetworkCount];
        // get last samples out of delay lines, multiply by attenuation coef, and low-pass filter
        ProcessFeedback(temps, fdnFeedback, fdnDelay, fdnLowPassHistory, fdnLowPassCoef);
        MixFeedbackMatrix(temps2, temps);

        // add inputs, then all-pass, then feedback to delays
        for (int i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
        {
            temps[i] = AllPassTick(&allPass[i], late_in + temps2[i]);
            DelayLineTick(&fdnDelay[i], temps[i]);
        }

        // send outputs
        outputs[0] += (temps[0] + temps[2]) + (temps[1] + temps[3]); //add all channels of feedback delay network to the 1 channel

        pOutData1[j] = QfToQ8(MultiQfQf(outputs[0], reverb->_outGain)) + MultiQfQf(reverb->_dryGain, pInData1[j]);
    }
    state->_fdnLowPassHistory[0] = fdnLowPassHistory[0];
    state->_fdnLowPassHistory[1] = fdnLowPassHistory[1];
    state->_fdnLowPassHistory[2] = fdnLowPassHistory[2];
    state->_fdnLowPassHistory[3] = fdnLowPassHistory[3];
}

NN_FORCEINLINE void ApplyReverbEffect2ch(const ReverbParameter* reverb, ReverbState* state, int sampleCount, const int32_t* pInData0, const int32_t* pInData1, int32_t* pOutData0, int32_t* pOutData1)
{
    qf outputs[nn::audio::ReverbParameter::ChannelCountMax];
    qf temps2[ReverbParameter::FeedbackDelayNetworkCount];
    qf fdnLowPassHistory[ReverbParameter::FeedbackDelayNetworkCount] = {
        state->_fdnLowPassHistory[0],
        state->_fdnLowPassHistory[1],
        state->_fdnLowPassHistory[2],
        state->_fdnLowPassHistory[3] }; // this needs to be restored after the loop below

    // early delay parameters
    FXDelayLine* earlyDelay = &state->_earlyDelay;
    uint32_t *earlyTap = &state->_earlyTap[0];
    qf *earlyCoef = &state->_earlyCoef[0];
    uint32_t lateTap = state->_lateTap;
    // FDN parameters
    qf *fdnFeedback = &state->_fdnFeedback[0];
    FXDelayLine *fdnDelay = &state->_fdnDelay[0];
    qf *fdnLowPassCoef = &state->_fdnLowPassCoef[0];
    FXDelayLine *allPass = &state->_allPass[0];
    const qf *base = earlyDelay->buff;
    const uint32_t delay = earlyDelay->delay;
    for (int j = 0; (j < sampleCount); j++)
    {
        qf* in = earlyDelay->p;
        outputs[0] = outputs[1] = QF_CONST(0.0f); // zero outputs

        //early reflections
        // for (int i = 0; i < ReverbParameter::EarlyReflectionTaps; ++i)
        // {
            // outputs[__TapOutAssignments2ch[i]] += MultiQfQf(DelayLineTapOut(earlyDelay, earlyTap[i]), earlyCoef[i]);
        // }
        // unroll ReverbParameter::EarlyReflectionTaps times (10)
        uint32_t taps = 0;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]); ++taps;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        // get input samples (as int32_t)
        qf in_samp = MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData0[j])) + MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData1[j])); // Mix input channels
        DelayLineTick(earlyDelay, in_samp);

        // late reverb
        qf late_in = MultiQfQf(DelayLineTapOut(earlyDelay, lateTap), reverb->_lateGain);

        qf temps[ReverbParameter::FeedbackDelayNetworkCount];
        // get last samples out of delay lines, multiply by attenuation coef, and low-pass filter
        ProcessFeedback(temps, fdnFeedback, fdnDelay, fdnLowPassHistory, fdnLowPassCoef);
        MixFeedbackMatrix(temps2, temps);

        // add inputs, then all-pass, then feedback to delays
        //unroll ReverbParameter::FeedbackDelayNetworkCount times (4)
        // for (int i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
        // {
            // temps[i] = AllPassTick(&allPass[i], late_in + temps2[i]);
            // DelayLineTick(&fdnDelay[i], temps[i]);
        // }
        uint32_t fdn = 0;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        ++fdn;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        ++fdn;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        ++fdn;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        // send outputs
        outputs[0] += (temps[0] + temps[2]); // Left + LeftSur
        outputs[1] += (temps[1] + temps[3]); // Right + RightSur

        pOutData0[j] = QfToQ8(MultiQfQf(outputs[0], reverb->_outGain)) + MultiQfQf(reverb->_dryGain, pInData0[j]);
        pOutData1[j] = QfToQ8(MultiQfQf(outputs[1], reverb->_outGain)) + MultiQfQf(reverb->_dryGain, pInData1[j]);
    }
    state->_fdnLowPassHistory[0] = fdnLowPassHistory[0];
    state->_fdnLowPassHistory[1] = fdnLowPassHistory[1];
    state->_fdnLowPassHistory[2] = fdnLowPassHistory[2];
    state->_fdnLowPassHistory[3] = fdnLowPassHistory[3];
}

NN_FORCEINLINE void ApplyReverbEffect4ch(const ReverbParameter * reverb, ReverbState* state, int sampleCount,
                      const int32_t* pInData0, const int32_t* pInData1, const int32_t* pInData2, const int32_t* pInData3,
                      int32_t* pOutData0, int32_t* pOutData1, int32_t* pOutData2, int32_t* pOutData3)
{
qf outputs[nn::audio::ReverbParameter::ChannelCountMax];
    qf temps[ReverbParameter::FeedbackDelayNetworkCount];
    qf temps2[ReverbParameter::FeedbackDelayNetworkCount];
    qf fdnLowPassHistory[ReverbParameter::FeedbackDelayNetworkCount] = {
        state->_fdnLowPassHistory[0],
        state->_fdnLowPassHistory[1],
        state->_fdnLowPassHistory[2],
        state->_fdnLowPassHistory[3] }; // this needs to be restored after the loop below

    // early delay parameters
    FXDelayLine* earlyDelay = &state->_earlyDelay;
    uint32_t *earlyTap = &state->_earlyTap[0];
    qf *earlyCoef = &state->_earlyCoef[0];
    uint32_t lateTap = state->_lateTap;

    // FDN parameters
    qf *fdnFeedback = &state->_fdnFeedback[0];
    FXDelayLine *fdnDelay = &state->_fdnDelay[0];
    qf *fdnLowPassCoef = &state->_fdnLowPassCoef[0];
    FXDelayLine *allPass = &state->_allPass[0];

    const qf *base = earlyDelay->buff;
    const uint32_t delay = earlyDelay->delay;
    for (int j = 0; (j < sampleCount); j++)
    {
        qf* in = earlyDelay->p;
        // zero outputs
        outputs[0] = outputs[1] = outputs[2] = outputs[3] = QF_CONST(0.0f);
        uint32_t tap = 0;
        //early reflections unrolling ReverbParameter::EarlyReflectionTaps times (10)
        // for (int i = 0; (i < ReverbParameter::EarlyReflectionTaps); ++i)
        // {
            // outputs[__TapOutAssignments4ch[i]] += MultiQfQf(DelayLineTapOut(earlyDelay, earlyTap[i]), earlyCoef[i]);
        // }
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[0] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[1] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[2] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[2] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[3] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]); ++tap;
        outputs[3] += MultiQfQf(DelayLineTapOut(in, base, earlyTap[tap], delay), earlyCoef[tap]);

        // input
        qf in_samp = MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData0[j]))
                    + MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData1[j]))
                    + MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData2[j]))
                    + MultiQfQf(reverb->_reverbGain, Q8ToQf(pInData3[j])); // Mix input channels
        DelayLineTick(earlyDelay, in_samp);

        // late reverb
        qf late_in = MultiQfQf(DelayLineTapOut(earlyDelay, lateTap), reverb->_lateGain);

        // get last samples out of delay lines, multiply by attenuation coef, and low-pass filter
        ProcessFeedback(temps, fdnFeedback, fdnDelay, fdnLowPassHistory, fdnLowPassCoef);
        MixFeedbackMatrix(temps2, temps);

        // add inputs, then all-pass, then feedback to delays
        // for (int i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
        // {
            // temps[i] = AllPassTick(&allPass[i], late_in + temps2[i]);
            // DelayLineTick(&fdnDelay[i], temps[i]);
        // }
        //unroll ReverbParameter::FeedbackDelayNetworkCount times (4)
        uint32_t fdn = 0;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        ++fdn;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        ++fdn;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        ++fdn;
        temps[fdn] = AllPassTick(&allPass[fdn], late_in + temps2[fdn]);
        DelayLineTick(&fdnDelay[fdn], temps[fdn]);
        // send outputs
        outputs[0] += temps[0]; // Left
        outputs[1] += temps[1]; // Right
        outputs[2] += temps[2]; // Left Surround
        outputs[3] += temps[3]; // Right Surround

        pOutData0[j] = QfToQ8(MultiQfQf(outputs[0], reverb->_outGain)) + MultiQfQf(pInData0[j], reverb->_dryGain);
        pOutData1[j] = QfToQ8(MultiQfQf(outputs[1], reverb->_outGain)) + MultiQfQf(pInData1[j], reverb->_dryGain);
        pOutData2[j] = QfToQ8(MultiQfQf(outputs[2], reverb->_outGain)) + MultiQfQf(pInData2[j], reverb->_dryGain);
        pOutData3[j] = QfToQ8(MultiQfQf(outputs[3], reverb->_outGain)) + MultiQfQf(pInData3[j], reverb->_dryGain);
    }

    state->_fdnLowPassHistory[0] = fdnLowPassHistory[0];
    state->_fdnLowPassHistory[1] = fdnLowPassHistory[1];
    state->_fdnLowPassHistory[2] = fdnLowPassHistory[2];
    state->_fdnLowPassHistory[3] = fdnLowPassHistory[3];
}

NN_FORCEINLINE void ApplyReverbEffect6ch(const ReverbParameter * reverb, ReverbState* state, int sampleCount,
    const int32_t **ppInData, int32_t **ppOutData)
{
    qf outputs[nn::audio::ReverbParameter::ChannelCountMax];
    qf temps[ReverbParameter::FeedbackDelayNetworkCount];
    qf temps2[ReverbParameter::FeedbackDelayNetworkCount];
    qf fdnLowPassHistory[ReverbParameter::FeedbackDelayNetworkCount] = {
        state->_fdnLowPassHistory[0],
        state->_fdnLowPassHistory[1],
        state->_fdnLowPassHistory[2],
        state->_fdnLowPassHistory[3] }; // this needs to be restored after the loop below

                                        // early delay parameters
    FXDelayLine* earlyDelay = &state->_earlyDelay;
    uint32_t *earlyTap = &state->_earlyTap[0];
    qf *earlyCoef = &state->_earlyCoef[0];
    uint32_t lateTap = state->_lateTap;

    // FDN parameters
    qf *fdnFeedback = &state->_fdnFeedback[0];
    FXDelayLine *fdnDelay = &state->_fdnDelay[0];
    qf *fdnLowPassCoef = &state->_fdnLowPassCoef[0];
    FXDelayLine *allPass = &state->_allPass[0];
    const qf *base = earlyDelay->buff;
    const uint32_t delay = earlyDelay->delay;
    for (int j = 0; (j < sampleCount); j++)
    {
        // zero outputs
        outputs[0] = outputs[1] = outputs[2] = outputs[3] = outputs[4] = outputs[5] = QF_CONST(0.0f);
        qf* in = earlyDelay->p;
        //early reflections unrolling (ReverbParameter::EarlyReflectionTaps has been 10 for > 1 year)
        //for (int i = 0; (i < ReverbParameter::EarlyReflectionTaps); ++i)
        //{
            // qf value = MultiQfQf(DelayLineTapOut(earlyDelay, earlyTap[i]), earlyCoef[i]);
            // outputs[__TapOutAssignments6ch[i]] += value;
            // outputs[5] += value;
        //}
        uint32_t taps = 0;
        qf value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[0] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[0] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[1] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[1] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[4] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[4] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[2] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[2] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[3] += value;
        outputs[5] += value;
        ++taps;
        value = MultiQfQf(DelayLineTapOut(in, base, earlyTap[taps], delay), earlyCoef[taps]);
        outputs[3] += value;
        outputs[5] += value;

        outputs[5] = MultiQfQf(QF_CONST(0.2), outputs[5]);

        // input
        qf mono = MultiQfQf(reverb->_reverbGain, Q8ToQf(ppInData[0][j] + ppInData[1][j] + ppInData[2][j] + ppInData[3][j] + ppInData[4][j] + ppInData[5][j]));
        DelayLineTick(earlyDelay, mono);

        // late reverb
        qf late_in = MultiQfQf(DelayLineTapOut(earlyDelay, lateTap), reverb->_lateGain);

        // get last samples out of delay lines, multiply by attenuation coef, and low-pass filter
        ProcessFeedback(temps, fdnFeedback, fdnDelay, fdnLowPassHistory, fdnLowPassCoef);
        MixFeedbackMatrix(temps2, temps);

        // add inputs, then all-pass, then feedback to delays unroll
        // for (int i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++) unroll
        // {
            // temps[i] = AllPassTick(&allPass[i], late_in + temps2[i]);
            // DelayLineTick(&fdnDelay[i], temps[i]);
        // }
        uint32_t fdnIndex = 0;

        temps[fdnIndex] = AllPassTick(&allPass[fdnIndex], late_in + temps2[fdnIndex]);
        DelayLineTick(&fdnDelay[fdnIndex], temps[fdnIndex]);
        ++fdnIndex;
        temps[fdnIndex] = AllPassTick(&allPass[fdnIndex], late_in + temps2[fdnIndex]);
        DelayLineTick(&fdnDelay[fdnIndex], temps[fdnIndex]);
        ++fdnIndex;
        temps[fdnIndex] = AllPassTick(&allPass[fdnIndex], late_in + temps2[fdnIndex]);
        DelayLineTick(&fdnDelay[fdnIndex], temps[fdnIndex]);
        ++fdnIndex;
        temps[fdnIndex] = AllPassTick(&allPass[fdnIndex], late_in + temps2[fdnIndex]);
        DelayLineTick(&fdnDelay[fdnIndex], temps[fdnIndex]);

        qf tempc = DelayLineTick(&state->_centerDelay, MultiQfQf(QF_CONST(0.5), (temps[2] - temps[3])));

        // send outputs
        outputs[0] += temps[0]; // Left
        outputs[1] += temps[1]; // Right
        outputs[2] += temps[2]; // Right Surround
        outputs[3] += temps[3]; // Right Surround
        outputs[4] += tempc; // Center
        outputs[5] += temps[3]; // LFE

        ppOutData[0][j] = QfToQ8(MultiQfQf(outputs[0], reverb->_outGain)) + MultiQfQf(ppInData[0][j], reverb->_dryGain);
        ppOutData[1][j] = QfToQ8(MultiQfQf(outputs[1], reverb->_outGain)) + MultiQfQf(ppInData[1][j], reverb->_dryGain);
        ppOutData[2][j] = QfToQ8(MultiQfQf(outputs[2], reverb->_outGain)) + MultiQfQf(ppInData[2][j], reverb->_dryGain);
        ppOutData[3][j] = QfToQ8(MultiQfQf(outputs[3], reverb->_outGain)) + MultiQfQf(ppInData[3][j], reverb->_dryGain);
        ppOutData[4][j] = QfToQ8(MultiQfQf(outputs[4], reverb->_outGain)) + MultiQfQf(ppInData[4][j], reverb->_dryGain);
        ppOutData[5][j] = QfToQ8(MultiQfQf(outputs[5], reverb->_outGain)) + MultiQfQf(ppInData[5][j], reverb->_dryGain);
    }

    state->_fdnLowPassHistory[0] = fdnLowPassHistory[0];
    state->_fdnLowPassHistory[1] = fdnLowPassHistory[1];
    state->_fdnLowPassHistory[2] = fdnLowPassHistory[2];
    state->_fdnLowPassHistory[3] = fdnLowPassHistory[3];
} // NOLINT(readability/fn_size)

NN_FORCEINLINE void ApplyReverbEffectBypass(const int32_t **ppInData, int32_t **ppOutData, int channelCount, int sampleCount)
{
    for(int i = 0; i < channelCount; ++i)
    {
        if (ppOutData[i] != ppInData[i])
        {
            memcpy(ppOutData[i], ppInData[i], sampleCount* sizeof(int32_t));
        }
    }
}

}}} // namespace nn::audio::detail
