﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"
#include "audio_EffectI3dl2Reverb.h"
#include "audio_Axfx2DelayLine.h"
#include "audio_Qf.h"

// Note: NEON optimized version has not been implemented yet.
#if (defined(__ARM_NEON__) || defined(__ARM_NEON)) && 0
#include "detail/audio_EffectI3dl2Reverb.neon.h"
#else  // defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_EffectI3dl2Reverb.generic.h"
#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)

namespace nn {
namespace audio {

namespace {

// FDN delay lengths in msec when Density = 100% (the length for the ith delay is 380^(i/10), and then all values are normalized to sum to 550ms)
const f32 FdnDelayLenMax[I3dl2ReverbParameter::FeedbackDelayNetworkCount] = { 45.7042f, 82.7817f, 149.9383f, 271.5758f };

// delay lengths for AP filters, in msec (these values are chosen by hand)
const f32 I3dl2ApDelayLen[I3dl2ReverbParameter::FeedbackDelayNetworkCount] = { 17.0f, 13.0f,  9.0f, 7.0f}; // delay lengths for AP filters, in msec
const f32 I3dl2ApDelayLen2[I3dl2ReverbParameter::FeedbackDelayNetworkCount] = { 19.0f, 11.0f, 10.0f, 6.0f }; // delay lengths for 2nd set of AP filters, in msec

const f32 I3dl2ReverbLevelMax = 5000.f;
const f32 I3dl2ReverbMinReflectionDuration = 0.02f; // min duration of early reflections, in sec

// delay times for early reflection taps scaled [0.0, 1.0]
const f32 ErTimes[I3dl2ReverbParameter::EarlyReflectionTaps] =
{ 0.017136f, 0.059154f, 0.161733f, 0.390186f, 0.425262f, 0.455411f, 0.689737f, 0.745910f, 0.833844f, 0.859502f, 0.000000f, 0.075024f, 0.168788f, 0.299901f, 0.337443f, 0.371903f, 0.599011f, 0.716741f, 0.817859f, 0.851664f };

// FDN delay lengths when Density = 0%, in msec
const f32 FdnDelayLenMin[I3dl2ReverbParameter::FeedbackDelayNetworkCount] = { 5.0f, 6.0f, 13.0f, 14.0f };

f32 Pow10F32(f32 value)
{
    f32 result;
    qf valueQF = static_cast<qf>( value * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    qf resultQF = Pow10Qf(valueQF);
    result = (f32) resultQF / (f32) ( 1 << QF_FRACTIONAL_BIT_COUNT);
    return result;
}

void SetFdnCoefficientsI3dl2(const I3dl2ReverbParameter* reverb, I3dl2ReverbState* state, uint32_t fdn_num)
{
    f32 lo_db;
    f32 hi_db;
    f32 gamma;
    f32 rho;
    f32 c;
    f32 a1;
    f32 b0;
    f32 b1;
    f32 f_trans;
    f32 hf_ratio = reverb->_hfDecayRatio;

    lo_db = -60.0f * (Axfx2DelayGetDelay(&state->_fdnDelayLines[fdn_num])
          + Axfx2AllPassGetDelayLen(&state->_apDelayLine[fdn_num])
          + Axfx2AllPassGetDelayLen(&state->_ap2DelayLine[fdn_num])) / (reverb->_decayTime * reverb->_sampleRate);    // DC gain in dB

    hi_db = lo_db / hf_ratio;  // high freq gain in dB
    f_trans = 0.5f * reverb->_hfReference;     // transition freq is 1 octaves below reference freq.
    //c = 1.0f / tan(PI * f_trans / reverb->_sampleRate);
    qf cosFTransQf = CosQf(QF_CONST(128 * f_trans / reverb->_sampleRate));
    f32 cosFTransF32 = (f32) cosFTransQf / (f32) ( 1 << QF_FRACTIONAL_BIT_COUNT);
    qf sinFTransQf = SinQf(QF_CONST(128 * f_trans / reverb->_sampleRate));
    f32 sinFTransF32 = (f32) sinFTransQf / (f32) ( 1 << QF_FRACTIONAL_BIT_COUNT);
    c = cosFTransF32 / sinFTransF32;

    rho = Pow10F32((hi_db - lo_db) / 40.0f);
    gamma = Pow10F32((hi_db + lo_db) / 40.0f);
    gamma *= 0.7071f;     // fold in the matrix normalization

    b0 = gamma*(1.0f + rho*c) / (c + rho);
    b1 = gamma*(1.0f - rho*c) / (c + rho);
    a1 = (c - rho) / (c + rho);

    state->_fdnFilterCoef[0][fdn_num] = b0;
    state->_fdnFilterCoef[1][fdn_num] = b1;
    state->_fdnFilterCoef[2][fdn_num] = a1;
}

// sets all the params for the late reverb FDN.  Call this whenever parms changes.
void SetAllFdnParametersI3dl2(const I3dl2ReverbParameter* reverb, I3dl2ReverbState* state)
{
    int i;

    // coef for all-passes as a function of diffusion
    state->_apCoef = 0.6f * reverb->_diffusion * 0.01f;                 // scale coefficient to [0.0, 0.6]
    const auto sampleRate = reverb->_sampleRate / 1000; // [kHz]

    // set delay lengths for delay line and all-pass
    for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        const auto temp_len = FdnDelayLenMin[i] + (reverb->_density / 100.0f) * (FdnDelayLenMax[i] - FdnDelayLenMin[i]);
        Axfx2DelaySetDelay(&state->_fdnDelayLines[i], Axfx2DelayLineCalculateDelaySamples(sampleRate, temp_len));
        SetFdnCoefficientsI3dl2(reverb, state, i);
        Axfx2AllPassSetCoef(&state->_apDelayLine[i], state->_apCoef);
        Axfx2AllPassSetCoef(&state->_ap2DelayLine[i], -0.9f * state->_apCoef);
    }
}

void ChangeParametersBaseI3dl2(const I3dl2ReverbParameter* reverb, I3dl2ReverbState* state)
{
    f32 mb;
    const auto sampleRate = reverb->_sampleRate / 1000; // [kHz]

    mb = reverb->_room + reverb->_reflections;
    if (I3dl2ReverbLevelMax < mb)
    {
        mb = I3dl2ReverbLevelMax;
    }
    state->_earlyLevel = Pow10F32(mb / 2000.0f);

    mb = reverb->_room + reverb->_reverb;
    if (I3dl2ReverbLevelMax < mb)
    {
        mb = I3dl2ReverbLevelMax;
    }
    state->_lateLevel = Pow10F32(mb / 2000.0f);

    qf cosWHF = CosQf(QF_CONST(256 * reverb->_hfReference / reverb->_sampleRate));
    f32 cos_w_hf = (f32) cosWHF / (f32) ( 1 << QF_FRACTIONAL_BIT_COUNT);

    // calc coefficients for input LPF
    f32 hf_gain_lin, a, b;
    hf_gain_lin = Pow10F32(reverb->_roomHf / 2000.0f); // linear gain at hgh freqs
    if (hf_gain_lin >= 1.0f)
    {
        state->_lpfB0 = 1.0f;
        state->_lpfA1 = 0.0f;
    }
    else
    {
        a = 1.0f - hf_gain_lin;
        b = 2.0f - 2.0f * hf_gain_lin * cos_w_hf;

        qf tempQf = SqrtQf(QF_CONST(b * b - 4.0f * a * a));
        f32 tempF32 = (f32) tempQf / (f32) ( 1 << QF_FRACTIONAL_BIT_COUNT);
        state->_lpfA1 = (b - tempF32) / (2.0f * a);       // fb coef for lpf
        state->_lpfB0 = 1.0f - state->_lpfA1;                    // ff coef for lpf (DC normalized)
    }

    // early reflections stuff
    state->_erLateTap = Axfx2DelayLineCalculateDelaySamples(sampleRate, 1000.f * (reverb->_reflectionsDelay + reverb->_reverbDelay));

    SetAllFdnParametersI3dl2(reverb, state);
}

void ParametersCheckAndUpdateI3dl2(const I3dl2ReverbParameter* reverb, I3dl2ReverbState* state, bool clearFlag)
{
    uint32_t i;

    state->_dryGain = reverb->_dryGain;

    for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++) // 4
    {
        state->_fdnFilterZ[i] = 0.0f;
    }

    state->_lpfZ = 0.0f;

    ChangeParametersBaseI3dl2(reverb, state);

    if (clearFlag)
    {
        // clear late reverb stuff
        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++) // 4
        {
            Axfx2DelayClear(&state->_fdnDelayLines[i]);
            Axfx2DelayClear(&state->_apDelayLine[i]);
            Axfx2DelayClear(&state->_ap2DelayLine[i]);
        }

        // clear early reflection delay line
        Axfx2DelayClear(&state->_earlyDelayLine);

        // clear center delay
        Axfx2DelayClear(&state->_centerDelayLine);
    }
}

} // namespace

void InitializeI3dl2ReverbEffect(const I3dl2ReverbParameter *reverb, I3dl2ReverbState* state, void* workBuffer)
{
    memset(state, 0, sizeof(I3dl2ReverbState));

    uint32_t i;
    int32_t tempi;
    f32* buf = static_cast<f32*>(workBuffer);
    const auto sampleRate = reverb->_sampleRate / 1000; // [kHz]

    // late reverb stuff
    for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        tempi = Axfx2DelayLineCalculateDelaySamples(sampleRate, FdnDelayLenMax[i]);
        Axfx2DelayInitialize(&state->_fdnDelayLines[i], tempi, buf);
        buf += tempi + 1;

        tempi = Axfx2DelayLineCalculateDelaySamples(sampleRate, I3dl2ApDelayLen[i]);
        Axfx2AllPassInitialize(&state->_apDelayLine[i], tempi, 0.0, buf);
        buf += tempi + 1;

        tempi = Axfx2DelayLineCalculateDelaySamples(sampleRate, I3dl2ApDelayLen2[i]);
        Axfx2AllPassInitialize(&state->_ap2DelayLine[i], tempi, 0.0, buf);
        buf += tempi + 1;
    }

    tempi = Axfx2DelayLineCalculateDelaySamples(sampleRate, I3dl2ReverbParameter::LateReverbCenterDelay);
    Axfx2DelayInitialize(&state->_centerDelayLine, tempi, buf);
    buf += tempi + 1;

    // allocate early reflection delay line
    tempi = Axfx2DelayLineCalculateDelaySamples(sampleRate, I3dl2ReverbParameter::EarlyReflectionDelayMax + I3dl2ReverbParameter::LateReverbDelayMax);
    Axfx2DelayInitialize(&state->_earlyDelayLine, tempi, buf);
    buf += tempi + 1;

    UpdateI3dl2ReverbEffectParameter(reverb, state, true);
}

void UpdateI3dl2ReverbEffectParameter(const I3dl2ReverbParameter* reverb, I3dl2ReverbState* state, bool clearFlag)
{
    int32_t tap;
    uint32_t temp_max_dly;
    f32 temp_ms_dly;
    uint32_t temp_samp_dly;
    const auto sampleRate = reverb->_sampleRate / 1000; // [kHz]

    ParametersCheckAndUpdateI3dl2(reverb, state, clearFlag);

    temp_max_dly = Axfx2DelayGetMaxDelay(&state->_earlyDelayLine);

    f32 refl_time_scale = 1000.f * (I3dl2ReverbMinReflectionDuration + reverb->_reverbDelay * (1.0f - I3dl2ReverbMinReflectionDuration / I3dl2ReverbParameter::LateReverbDelayMax)); // linearly interpolate the duration of early reflections

    //  calculate early reflections
    for (tap = 0; tap < I3dl2ReverbParameter::EarlyReflectionTaps; tap++) // 20
    {
        temp_ms_dly = 1000.f * reverb->_reflectionsDelay + refl_time_scale * ErTimes[tap];
        temp_samp_dly = Axfx2DelayLineCalculateDelaySamples(sampleRate, temp_ms_dly);
        state->_erTapTimes[tap] = (temp_samp_dly < temp_max_dly) ? temp_samp_dly : temp_max_dly; // make sure we don't go over the max
    }
}

void ApplyI3dl2ReverbEffect(const I3dl2ReverbParameter *reverb, I3dl2ReverbState* state, bool enabled, const int32_t **ppInData, int32_t **ppOutData, const uint32_t sampleCount)
{
    NN_AUDIO_DSP_ASSERT((reverb->_numChannels == 1) || (reverb->_numChannels == 2) || (reverb->_numChannels == 4) || (reverb->_numChannels == 6)); // for klocwork inspection check.

    const auto channelCount = reverb->_numChannels;

    if (enabled)
    {
        switch (channelCount)
        {
        case 1:
            detail::ApplyI3dl2ReverbEffect1ch(state, ppInData, ppOutData, sampleCount);
            break;
        case 2:
            detail::ApplyI3dl2ReverbEffect2ch(state, ppInData, ppOutData, sampleCount);
            break;
        case 4:
            detail::ApplyI3dl2ReverbEffect4ch(state, ppInData, ppOutData, sampleCount);
            break;
        case 6:
            detail::ApplyI3dl2ReverbEffect6ch(state, ppInData, ppOutData, sampleCount);
            break;
        default: // default bypass
            detail::ApplyI3dl2ReverbEffectBypass(ppInData, ppOutData, channelCount, sampleCount);
            break;
        }
    }
    else // bypass
    {
        detail::ApplyI3dl2ReverbEffectBypass(ppInData, ppOutData, channelCount, sampleCount);
    }
}

size_t I3DL2ReverbGetRequiredDelayLineBufferSize(int fs, int channelCount)
{
    NN_UNUSED(channelCount);

    size_t samples = 0;

    const auto sampleRateKhz = fs / 1000; // [kHz]

    for (auto i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        samples += Axfx2DelayLineCalculateDelaySamples(sampleRateKhz, FdnDelayLenMax[i]) + 1;
        samples += Axfx2DelayLineCalculateDelaySamples(sampleRateKhz, I3dl2ApDelayLen[i]) + 1;
        samples += Axfx2DelayLineCalculateDelaySamples(sampleRateKhz, I3dl2ApDelayLen2[i]) + 1;
    }
    samples += Axfx2DelayLineCalculateDelaySamples(sampleRateKhz, I3dl2ReverbParameter::LateReverbCenterDelay) + 1;
    samples += Axfx2DelayLineCalculateDelaySamples(sampleRateKhz, I3dl2ReverbParameter::EarlyReflectionDelayMax + I3dl2ReverbParameter::LateReverbDelayMax) + 1;

    return samples * sizeof(float);
}

} // namespace audio
} // namespace nn
