﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
  Project:
  File:     src_bl.h
 *---------------------------------------------------------------------------*/
/*  \file  src_bl.h
 *  \brief dll interface wrapper header file over Bandlimited interpolation sample rate converter
 */
#pragma once

#include <stdint.h>
#include <nn/nn_Common.h>

namespace nn {
namespace audio {


typedef enum SRC_DOWNSAMPLER_CONSTANTS
{
    MAX_int16_t = (32767),
    MIN_int16_t = (-32768),
    NQRATIO  = 16,

    /*! \def Block SRC Related Constants
 */
    NN_AUDIO_SRC_BL_DN_NZ =  10,      /*!< number of zero crossings in the filter table, integer only */
    NN_AUDIO_SRC_BL_DN_NT =  32,      /*!< number of total bits for time register t */
    NN_AUDIO_SRC_BL_DN_NL =  8,       /*!< number of bits for L (samples per zero-crossing) */
    NN_AUDIO_SRC_BL_DN_NNU = 7,       /*!< number of bits for nu */

    NN_AUDIO_SRC_BL_DN_NP  =  (NN_AUDIO_SRC_BL_DN_NL + NN_AUDIO_SRC_BL_DN_NNU),   /*!< how many bits to the right of the binary point;*/
    NN_AUDIO_SRC_BL_DN_NN =   (NN_AUDIO_SRC_BL_DN_NT - NN_AUDIO_SRC_BL_DN_NP ),   /*!< number of bits for the integer part of t */
    NN_AUDIO_SRC_BL_DN_L  =   (1 << NN_AUDIO_SRC_BL_DN_NL),         /*!< the number of samples per zero crossingin the filter */
    NN_AUDIO_SRC_BL_DN_NH  =  (NN_AUDIO_SRC_BL_DN_NZ * NN_AUDIO_SRC_BL_DN_L),   /*!< length of filter table */

    NN_AUDIO_SRC_BL_DN_NU_MASK  =  (uint32_t) ((1 << NN_AUDIO_SRC_BL_DN_NNU) - 1 ),      /*!< a mask for getting nu */
    NN_AUDIO_SRC_BL_DN_RATIO_MAX = ((uint32_t) 1 << (NN_AUDIO_SRC_BL_DN_NP + NQRATIO)), /*!< maximum conversion ratio 2^n_p, Q "NQRATIO" format */
    NN_AUDIO_SRC_BL_DN_RATIO_MIN = (1 << (NQRATIO - 3)),  /*!< Q "NQRATIO" format - 3 octaves. the lower we allow rho_min to go, the more computation we need.
                                            minimum conversion ratio 0.125; downsample FsNew = FsOld / 8; max input samples needed 12 * 8 = 96 */
    NN_AUDIO_SRC_BL_DN_X_CB_SIZE = (((NN_AUDIO_SRC_BL_DN_NZ * 2 * (1 << NQRATIO)) / NN_AUDIO_SRC_BL_DN_RATIO_MIN) + 10),  /*!< add some creeping space */
    NN_AUDIO_SRC_BL_DN_CB_MID    = ((NN_AUDIO_SRC_BL_DN_X_CB_SIZE) / 2),

    NN_AUDIO_SRC_BL_DN_RATIO_48_441 = ((uint32_t)60211),
    NN_AUDIO_SRC_BL_DN_RATIO_48_96 = ((uint32_t)131072),
    NN_AUDIO_SRC_BL_DN_RATIO_48_32 = ((uint32_t)43691),

} SRC_DOWNSAMPLER_CONSTANTS;

/*! \struct DownSamplerState
 *  \brief  Define format of block SRC parameters
 *
 */
typedef struct DownSamplerState
{
    int32_t isInitialized;      // for lazy initialization
    uint32_t ratioRho;           /*!< Rho = FsNew / FsOld, Q15.16 format, defined by NQRATIO */
    uint32_t timeRegister;       /*!< t is a register which contains the output time in original samples.
                             t contains [ n_n bits for n | n_P bits for P ]
                             where n is the integer number of samples, and P is the fractional number of samples */
    uint32_t time_inc;           /*!< t_inc is how much to increment t by for every new sample of y */

    uint16_t nz_input;           /*!< how many steps through the filter will we make?  (for one side of the h filter)
                             or how many input samples are needed to calculate an output sample (2 * nz_input + 1) */
    uint16_t input_current;      /*!< array index to the current sample of the inputs, in input sample buffer, Q23.8 format */
    uint16_t input_write;        /*!< array index to the next sample writing location, in input sample buffer, Q23.8 format */
    uint16_t input_cb_lowerBound;
    uint16_t input_cb_upperBound;
    int16_t* h;                 /*!< h table */
    int32_t input_samples[NN_AUDIO_SRC_BL_DN_X_CB_SIZE];    /*!< Input Sample buffer for block SRC */
    int16_t numLeftovers;
    int32_t leftoverSamples[8];
} DownSamplerState;

/*! \fn  int32_t SrcProcessFrame(
*    DownSamplerState* pState,
*    int32_t numChannels,
*    int32_t channel,
*    int32_t* pIn,
*    int32_t* pOut,
*    int32_t inSamples,
*    int32_t outSamples)
*
*  \brief  run Resampler for rho < 1
*
*  \param[in]  DownSamplerState *pState
*  \param[in]  numChannels - number of interleaved channels in *pOut
*  \param[in]  channel - which channel position this resample is to occupy in the interleave
*  \param[in]  *pIn - pointer to single channel buffer of input samples
*  \param[in]  *pOut - pointer to interleaved output buffer base (ch 0)
*  \param[in]  inSamples - number of samples that must be consumed from *pIn
*  \param[in]  outSamples - number of samples that must be generated into *pOut
*  \return 0
*/
int32_t ResampleOutputFrame(DownSamplerState* pState, int32_t numChannels, int32_t channel,
    const int32_t* pIn, int16_t* pOut, int32_t inSamples, int32_t outSamples) NN_NOEXCEPT;

int32_t ResampleOutputFrame(DownSamplerState* pState, int32_t numChannels, int32_t channel,
    const int32_t* pIn, int16_t* pOut, int32_t inSamples, int32_t outSamples, uint32_t ratio) NN_NOEXCEPT;

int32_t ResampleOutputGetNextFrameSize(
    DownSamplerState* pState, int32_t numStates,
    int32_t inSamples, int32_t outSamples,
    int32_t* nextSamples) NN_NOEXCEPT;

}  // namespace audio
}  // namespace nn
