﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"

#include <nn/nn_StaticAssert.h>
#include <nn/audio/audio_Adpcm.h>
#include <nn/audio/audio_VoiceTypes.h>

#include "../common/audio_CommonWaveBuffer.h"
#include "../server/audio_VoiceState.h"

#include "audio_Cache.h"
#include "audio_DataSource.h"
#include "audio_Decode.h"
#include "audio_DspUtility.h"
#include "audio_Resample.h"

#include "opus.h"

// // define this if any pitch ratio should be supported
// #define NN_AUDIO_ALLOW_ANY_PITCH

namespace nn {
namespace audio {

namespace {
// TODO:
const int TempBufferCount = 16384 - 256;  // TODO: "- 256" to avoid SIGLO-67691
static int32_t tmp[TempBufferCount + VoiceState::SampleHistoryCount];  // TODO:
} // namespace anonymous

void DecodeFromWaveBuffers(const DecodeFromWaveBuffersArgs& args) NN_NOEXCEPT
{
    const auto& format = args.format;
    const auto& output = args.output;
    const auto& state = args.state;
    const auto& waveBuffer = args.waveBuffer;
    const auto& channel = args.channel;
    const auto& channelCount = args.channelCount;
    const auto& pitch = args.pitch;
    const auto& sourceSampleRate = args.sourceSampleRate;
    const auto& targetSampleRate = args.targetSampleRate;
    const auto& sampleCount = args.sampleCount;
    const auto& pCoef = args.pCoef;
    const auto& isPlayedSampleCountResetAtLoopPoint = args.isPlayedSampleCountResetAtLoopPoint;
    const auto& isPitchAndSrcSkipped = args.isPitchAndSrcSkipped;

    NN_AUDIO_DSP_ASSERT(output != nullptr);
    NN_AUDIO_DSP_ASSERT(targetSampleRate > 0);
    NN_AUDIO_DSP_ASSERT(sampleCount >= 0);
    NN_AUDIO_DSP_ASSERT(state != nullptr);
    NN_AUDIO_DSP_ASSERT(waveBuffer != nullptr);
    NN_AUDIO_DSP_ASSERT(channel >= 0);
    NN_AUDIO_DSP_ASSERT(channelCount >= 1);
    NN_AUDIO_DSP_ASSERT(pitch >= 0);
    NN_AUDIO_DSP_ASSERT(sourceSampleRate > 0);

    const auto resampleRate = static_cast<int32_t>(static_cast<float>(sourceSampleRate) / targetSampleRate * pitch);

    // Load state variables
    auto waveBufferConsumed = state->waveBufferConsumed;
    auto offset = state->offset;
    auto playedSampleCount = state->playedSampleCount;
    auto waveBufferIndex = state->waveBufferIndex;
    auto fraction = state->fraction;

    // TODO: Workaround for SIGLO-63280
#if !defined(NN_AUDIO_ALLOW_ANY_PITCH)
    {
        const auto sourceSampleCount = (sampleCount * resampleRate + fraction) >> 15;
        if(sourceSampleCount + VoiceState::SampleHistoryCount > TempBufferCount || sourceSampleCount < 0)
        {
            return;
        }
    }
#endif

    int32_t* outputBuffer = output;
    int remainSampleCount = sampleCount;
    bool isBufferStarved = false;

    const auto outputSampleCountForOneIteration = Min(((TempBufferCount << 15) - fraction) / resampleRate, sampleCount);

    while (remainSampleCount > 0 && isBufferStarved == false)
    {
        const auto outputSampleCount = Min(remainSampleCount, outputSampleCountForOneIteration);
        const auto sourceSampleCount = (outputSampleCount * resampleRate + fraction) >> 15;

        int32_t* p = tmp;
        auto decodedSamples = tmp;

        if(isPitchAndSrcSkipped == false)
        {
            // Restore last samples
            for (int i = 0; i < VoiceState::SampleHistoryCount; ++i)
            {
                *p++ = state->sampleHistory[i];
            }
        }

        int readCount = 0;
        while (readCount < sourceSampleCount)
        {
            NN_AUDIO_DSP_ASSERT(offset >= 0);
            NN_AUDIO_DSP_ASSERT(waveBufferIndex < VoiceType::WaveBufferCountMax);

            if(state->isWaveBufferValid[waveBufferIndex] == false)
            {
                isBufferStarved = true;
                break;
            }

            const auto pWaveBuffer = &waveBuffer[waveBufferIndex];

            if(format == SampleFormat_Adpcm && offset == 0 && pWaveBuffer->pContext)
            {
                // Copy loop context
                NN_STATIC_ASSERT(sizeof(state->context) >= sizeof(AdpcmContext));
                memcpy(state->context, reinterpret_cast<void*>(pWaveBuffer->pContext), sizeof(AdpcmContext));
            }

            int availableSampleCount;
            if(format == SampleFormat_Adpcm)
            {
                availableSampleCount = DecodeAdpcm(p, pWaveBuffer, offset, sourceSampleCount - readCount, pCoef, state->context);
            }
            else if(format == SampleFormat_PcmInt16)
            {
                availableSampleCount = DecodePcm16(p, pWaveBuffer, channel, channelCount, offset, sourceSampleCount - readCount);
            }
            else
            {
                NN_AUDIO_DSP_ASSERT(false, "Unsupported format.");
                availableSampleCount = 0; // set 0 to suppress warning
            }

            NN_AUDIO_DSP_ASSERT(availableSampleCount >= 0);

            readCount += availableSampleCount;
            p += availableSampleCount;
            offset += availableSampleCount;
            playedSampleCount += availableSampleCount;

            const auto isBufferEndReached = (offset >= pWaveBuffer->endSampleOffset - pWaveBuffer->startSampleOffset || availableSampleCount == 0);
            if(isBufferEndReached)
            {
                offset = 0;

                if (pWaveBuffer->loop)
                {
                    // To avoid infinity loop
                    if(availableSampleCount == 0)
                    {
                        isBufferStarved = true;
                        break;
                    }

                    if(isPlayedSampleCountResetAtLoopPoint)
                    {
                        playedSampleCount = 0;
                    }
                }
                else
                {
                    state->isWaveBufferValid[waveBufferIndex] = false;

                    ++waveBufferConsumed;
                    ++waveBufferIndex;

                    if (waveBufferIndex >= VoiceType::WaveBufferCountMax)
                    {
                        waveBufferIndex = 0;
                    }

                    if (pWaveBuffer->isEndOfStream)
                    {
                        playedSampleCount = 0;
                    }
                }
            }
        } // while (readCount < sourceSampleCount)

        if(isPitchAndSrcSkipped)
        {
            memcpy(outputBuffer, decodedSamples, readCount * sizeof(*p));
        }
        else
        {
            memset(p, 0, sizeof(*p) * (sourceSampleCount - readCount));
            Resample(outputBuffer, decodedSamples, resampleRate, &fraction, outputSampleCount);

            // Save last samples
            for (int i = 0; i < VoiceState::SampleHistoryCount; ++i)
            {
                state->sampleHistory[i] = decodedSamples[sourceSampleCount + i];
            }
        }

        outputBuffer += outputSampleCount;
        remainSampleCount -= outputSampleCount;

        NN_AUDIO_DSP_ASSERT(tmp <= p && p <= tmp + TempBufferCount - 1);
    } // while (remainSampleCount > 0 && isBufferStarved == false)

    NN_AUDIO_DSP_ASSERT(remainSampleCount == 0 || isBufferStarved);

    // Save state variables
    state->waveBufferConsumed = waveBufferConsumed;
    state->offset = offset;
    state->playedSampleCount = playedSampleCount;
    state->waveBufferIndex = waveBufferIndex;
    state->fraction = fraction;
} // NOLINT(impl/function_size)

namespace {

struct OpusContext
{
    static const size_t SampleCount = 2880;
    static const size_t WorkBufferSize = 48 * 1024;
    int16_t pcm[SampleCount];
    char work[WorkBufferSize];
    int current;
    int available;
    int frameSampleCount;
};

}

void GetOpusData(int32_t* output, VoiceState* state, const common::WaveBuffer waveBuffer[], int32_t _pitch, int sourceSampleRate, int targetSampleRate, int sampleCount, bool isPlayedSampleCountResetAtLoopPoint) NN_NOEXCEPT
{
    int32_t pitch = static_cast<int32_t>(static_cast<float>(sourceSampleRate) / targetSampleRate * _pitch);

    NN_AUDIO_TRACK_BUFFER(state, sizeof(VoiceState), dsp::CachedBuffer_RequireInval | dsp::CachedBuffer_RequireFlush);

#if !defined(NN_AUDIO_ALLOW_ANY_PITCH)
    // TODO: Workaround for SIGLO-63280
    {
        int32_t sourceSampleCount = (sampleCount * pitch + state->fraction) >> 15;
        if(sourceSampleCount + VoiceState::SampleHistoryCount > TempBufferCount || sourceSampleCount < 0)
        {
            return;
        }
    }
#endif  // !defined(NN_AUDIO_ALLOW_ANY_PITCH)

    if (state->externalContext == NULL)
    {
        return;
    }
    auto cxt = reinterpret_cast<OpusContext*>(state->externalContext);
    NN_AUDIO_DSP_ASSERT(state->externalContextSize >= sizeof(OpusContext));
    if (!state->isExternalContextInitialized)
    {
#if defined(_NX_AUDIO_DSP_)  // TODO: need to add dependency to nn::codec
        auto pOpusDecoder = reinterpret_cast<::OpusDecoder*>(cxt->work);
        if (opus_decoder_init(pOpusDecoder, sourceSampleRate, 1) != OPUS_OK)
        {
            // TODO: error handling
            return;
        }
#endif  // defined(_NX_AUDIO_DSP_)
        cxt->current = 0;
        cxt->available = 0;
        cxt->frameSampleCount = 0;
        state->isExternalContextInitialized = true;
    }

    int innerSampleCount = ((TempBufferCount << 15) - state->fraction) / pitch;
    innerSampleCount = innerSampleCount < sampleCount ? innerSampleCount : sampleCount;
    int remain = sampleCount;

    int32_t* pOut = output;

#if defined(NN_AUDIO_ALLOW_ANY_PITCH)
    while (remain > 0)
#endif  // defined(NN_AUDIO_ALLOW_ANY_PITCH)
    {
        int count = remain < innerSampleCount ? remain : innerSampleCount;
        int sourceSampleCount = (count * pitch + state->fraction) >> 15;
        int32_t* p = tmp;

        for (int i = 0; i < VoiceState::SampleHistoryCount; ++i)
        {
            *p++ = state->sampleHistory[i];
        }

        int readCount = 0;
        while (readCount < sourceSampleCount && state->isWaveBufferValid[state->waveBufferIndex])
        {
            auto pWaveBuffer = &waveBuffer[state->waveBufferIndex];
            if (cxt->available == 0)
            {
#if defined(_NX_AUDIO_DSP_)
                auto opusBuffer = reinterpret_cast<const unsigned char*>(pWaveBuffer->buffer);
                auto opusSize = pWaveBuffer->size;

                // TODO: this should be called exclusively with other opus calls...
                auto ret = opus_decode(reinterpret_cast<::OpusDecoder*>(cxt->work), opusBuffer, opusSize, cxt->pcm, static_cast<opus_int32>(sizeof(cxt->pcm) / sizeof(cxt->pcm[0])), 0);
                if (ret < 0)
                {
                    ret = 0;
                }
#else  // defined(_NX_AUDIO_DSP_)
                int ret = 0;
#endif  // defined(_NX_AUDIO_DSP_)

                cxt->current = pWaveBuffer->startSampleOffset;
                cxt->available = Min(ret, pWaveBuffer->endSampleOffset - pWaveBuffer->startSampleOffset);
                cxt->frameSampleCount = cxt->available;

                if(cxt->available == 0 && pWaveBuffer->loop)
                {
                    break;
                }
            }

            if (cxt->available > 0)
            {
                auto s = Min(cxt->available, sourceSampleCount - readCount);
                for (auto i = 0; i < s; ++i)
                {
                    *p++ = cxt->pcm[cxt->current + i];
                }
                cxt->available -= s;
                cxt->current += s;
                readCount += s;
            }
            if (cxt->available == 0)
            {
                if (pWaveBuffer->loop)
                {
                    cxt->current = pWaveBuffer->startSampleOffset;
                    cxt->available = cxt->frameSampleCount;
                    if(isPlayedSampleCountResetAtLoopPoint)
                    {
                        state->playedSampleCount = 0;
                    }
                }
                else
                {
                    state->isWaveBufferValid[state->waveBufferIndex] = false;
                    if (++state->waveBufferIndex >= VoiceType::WaveBufferCountMax)
                    {
                        state->waveBufferIndex = 0;
                    }
                    ++state->waveBufferConsumed;
                    if (pWaveBuffer->isEndOfStream)
                    {
                        state->playedSampleCount = 0;
                    }
                }
            }
        }
        memset(p, 0, sizeof(*p) * (sourceSampleCount - readCount));

        Resample(pOut, tmp, pitch, &state->fraction, count);
        pOut += count;
        remain -= count;

        for (int i = 0; i < VoiceState::SampleHistoryCount; ++i)
        {
            state->sampleHistory[i] = tmp[sourceSampleCount + i];
        }

#if !defined(NN_AUDIO_ALLOW_ANY_PITCH)
        NN_AUDIO_DSP_ASSERT(remain == 0);
#endif  // !defined(NN_AUDIO_ALLOW_ANY_PITCH)
    }
}  // NOLINT(impl/function_size)

}  // namespace audio
}  // namespace nn

