﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"

#include "audio_CommandProcessor.h"
#include "audio_DspUtility.h"
#include "audio_Cache.h"
#include "../audio_EffectPrivateTypes.h"
#include "../common/audio_Command.h"
#include "../common/audio_BuildDefinition.h"

#if defined(_NX_AUDIO_DSP_)
#include <os/kernel/thread.h>
#endif

#include "audio_DataSource.h"
#include "audio_Volume.h"
#include "audio_BiquadFilter.h"
#include "audio_Mix.h"
#include "audio_Downsampler.h"
#include "audio_Depop.h"

#if defined(NN_AUDIO_ENABLE_DEVICE_SINK)
#if defined(_NX_AUDIO_DSP_)
#include "audio_DeviceMixerManager.adsp.h"
#else
#include "audio_DeviceMixerManager.h"
#endif // defined(_NX_AUDIO_DSP_)
#endif // defined(NN_AUDIO_ENABLE_DEVICE_SINK)

#include "audio_EffectDelay.h"
#include "audio_EffectReverb.h"
#include "audio_EffectI3dl2Reverb.h"
#include "../audio_AuxTypes.h"
#include "audio_AuxBufferDsp.h"

// #define NN_AUDIO_ENABLE_PERFORMANCE_MEASURE_BY_THREAD_CYCLE

namespace nn { namespace audio { namespace dsp {

namespace
{
Time GetCurrentTime()
{
#if defined(NN_AUDIO_ENABLE_PERFORMANCE_MEASURE_BY_THREAD_CYCLE)
    return ADSP_HARDWARE_SELECT(thread_get_cycle_count(), nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()));
#else
    return ADSP_HARDWARE_SELECT(thread_get_cycle_count() / (get_cur_adsp_freq() / 1000 / 1000), nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()));
#endif
}

int32_t ConvertToMicroSeconds(Time time)
{
    return static_cast<int32_t>(ADSP_HARDWARE_SELECT(time, time.GetMicroSeconds()));
}

uint64_t GetCurrentThreadCycleCount()
{
    // This value is only used on NX ADSP.
    return ADSP_HARDWARE_SELECT(thread_get_cycle_count(), 0);
}

NN_FORCEINLINE
bool IsIndexInRange(int index, int min, int max) NN_NOEXCEPT
{
    return ((index >= min) && (index < max));
}

template<typename T>
bool VerifyCommand(const T* pCommand) NN_NOEXCEPT
{
    return (pCommand->header.id == T::Id) &&
           (pCommand->header.size == sizeof(T));
}

} // anonymous namespace

void CommandListProcessor::ProcessCommand(const PcmInt16DataSourceCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(PcmInt16DataSourceCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);

    const auto isPlayedSampleCountResetAtLoopPoint = (pCommand->behaviorFlags & DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint) != 0u;
    const auto isPitchAndSrcSkipped = (pCommand->behaviorFlags & DataSourceBehaviorFlag_IsPitchAndSrcSkipped) != 0u;

    const DecodeFromWaveBuffersArgs args = {
        SampleFormat_PcmInt16,
        output,
        reinterpret_cast<VoiceState*>(pCommand->state),
        pCommand->waveBuffer,
        pCommand->channel,
        pCommand->channelCount,
        pCommand->pitch,
        pCommand->sampleRate,
        m_TargetSampleRate,
        m_SampleCount,
        nullptr,
        isPlayedSampleCountResetAtLoopPoint,
        isPitchAndSrcSkipped,
    };

    DecodeFromWaveBuffers(args);
}

void CommandListProcessor::ProcessCommand(const AdpcmDataSourceCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(AdpcmDataSourceCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);

    const auto isPlayedSampleCountResetAtLoopPoint = (pCommand->behaviorFlags & DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint) != 0u;
    const auto isPitchAndSrcSkipped = (pCommand->behaviorFlags & DataSourceBehaviorFlag_IsPitchAndSrcSkipped) != 0u;
    const auto channel = 0;
    const auto channelCount = 1;

    const DecodeFromWaveBuffersArgs args = {
        SampleFormat_Adpcm,
        output,
        reinterpret_cast<VoiceState*>(pCommand->state),
        pCommand->waveBuffer,
        channel,
        channelCount,
        pCommand->pitch,
        pCommand->sampleRate,
        m_TargetSampleRate,
        m_SampleCount,
        reinterpret_cast<int16_t const*>(pCommand->adpcmParameter),
        isPlayedSampleCountResetAtLoopPoint,
        isPitchAndSrcSkipped,
    };

    DecodeFromWaveBuffers(args);
}

void CommandListProcessor::ProcessCommand(const OpusDataSourceCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(OpusDataSourceCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    auto state = reinterpret_cast<VoiceState*>(pCommand->state);
    const auto waveBuffer = pCommand->waveBuffer;
    const auto pitch = pCommand->pitch;
    const auto sourceSampleRate = pCommand->sampleRate;
    const auto isPlayedSampleCountResetAtLoopPoint = (pCommand->behaviorFlags & DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint) != 0u;
    GetOpusData(output, state, waveBuffer, pitch, sourceSampleRate, m_TargetSampleRate, m_SampleCount, isPlayedSampleCountResetAtLoopPoint);
}

void CommandListProcessor::ProcessCommand(const VolumeCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(VolumeCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    const auto volume = pCommand->volume;
    dsp::ApplyUniformGain(output, input, volume, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const VolumeRampCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(VolumeRampCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    const auto volume0 = pCommand->volume0;
    const auto volume1 = pCommand->volume1;
    NN_AUDIO_DSP_ASSERT(m_SampleCount > 0);
    const auto delta = (volume1 - volume0) / m_SampleCount;
    dsp::ApplyLinearEnvelopeGain(output, input, volume0, delta, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const BiquadFilterCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(BiquadFilterCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    const auto numerator = pCommand->numerator;
    const auto denominator = pCommand->denominator;
    auto state = reinterpret_cast<BiquadFilterState*>(pCommand->state);
    if (pCommand->needInitialization)
    {
        memset(state, 0, sizeof(BiquadFilterState));
    }
    NN_AUDIO_TRACK_BUFFER(state, sizeof(BiquadFilterState), CachedBuffer_RequireInval | CachedBuffer_RequireFlush);
    dsp::ApplyBiquadFilter(output, input, numerator, denominator, state, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const MixCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(MixCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    const auto volume = pCommand->volume;

    dsp::ApplyMix(output, input, volume, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const MixRampCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(MixRampCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    const auto volume0 = pCommand->volume0;
    const auto volume1 = pCommand->volume1;
    NN_AUDIO_DSP_ASSERT(m_SampleCount > 0);
    const auto delta = (volume1 - volume0) / m_SampleCount;

    auto x = dsp::ApplyMixRamp(output, input, volume0, delta, m_SampleCount);

    auto lastSample = reinterpret_cast<int32_t*>(pCommand->state);
    NN_AUDIO_TRACK_BUFFER(lastSample, sizeof(int32_t), CachedBuffer_RequireFlush);
    *lastSample = x;
}

void CommandListProcessor::ProcessCommand(const MixRampGroupedCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(MixRampGroupedCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const auto mixBufferCount = pCommand->mixBufferCount;
    NN_AUDIO_DSP_ASSERT(mixBufferCount >= 0 && mixBufferCount <= MixBufferCountMax);
    auto lastSamples = reinterpret_cast<int32_t*>(pCommand->state);
    NN_AUDIO_TRACK_BUFFER(lastSamples, sizeof(int32_t) * mixBufferCount, CachedBuffer_RequireInval | CachedBuffer_RequireFlush);
    for (auto i = 0; i < mixBufferCount; ++i)
    {
        if (!IsIndexInRange(pCommand->inputBufferIndices[i], 0, m_BufferCount) ||
            !IsIndexInRange(pCommand->outputBufferIndices[i], 0, m_BufferCount))
        {
            continue;
        }

        const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
        auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
        const auto volume0 = pCommand->volume0[i];
        const auto volume1 = pCommand->volume1[i];
        NN_AUDIO_DSP_ASSERT(m_SampleCount > 0);
        const auto delta = (volume1 - volume0) / m_SampleCount;
        if (volume0 != 0 || volume1 != 0)
        {
            lastSamples[i] = dsp::ApplyMixRamp(output, input, volume0, delta, m_SampleCount);
        }
        else
        {
            lastSamples[i] = 0;
        }
    }
}

void CommandListProcessor::ProcessCommand(const DepopPrepareCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(DepopPrepareCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const auto mixBufferCount = pCommand->mixBufferCount;
    const auto indexes = pCommand->outputBufferIndices;
    const auto lastSamples = reinterpret_cast<int32_t*>(pCommand->state);
    const auto depop = reinterpret_cast<int32_t*>(pCommand->depopBuffer);
    NN_AUDIO_TRACK_BUFFER(depop, sizeof(DepopPrepareCommand), CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(lastSamples, sizeof(int32_t) * MixBufferCountMax, CachedBuffer_RequireInval | CachedBuffer_RequireFlush);
    NN_AUDIO_TRACK_BUFFER(indexes, sizeof(int16_t) * mixBufferCount, CachedBuffer_RequireInval);

    for (int i = 0; i < mixBufferCount; ++i)
    {
        if (lastSamples[i] != 0)
        {
            auto index = indexes[i];
            depop[index] += lastSamples[i];
            lastSamples[i] = 0;
        }
    }
}

void CommandListProcessor::ProcessCommand(const DepopForMixBuffersCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(DepopPrepareCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const auto offset = pCommand->mixBufferOffset;
    const auto count = pCommand->mixBufferCount;
    const auto decay = pCommand->decay;
    const auto depop = reinterpret_cast<int32_t*>(pCommand->depopBuffer);
    const auto begin = offset;
    const auto end = Min(offset + count, m_BufferCount);
    NN_AUDIO_TRACK_BUFFER(depop, (end - begin) * sizeof(int32_t), CachedBuffer_LocalBuffer);
    for (auto i = begin; i < end; ++i)
    {
        if (depop[i] == 0)
        {
            continue;
        }

        auto output = GetMixBuffer(m_pMixBuffers, i, m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
        depop[i] = dsp::ApplyDepopMix(output, depop[i], decay, m_SampleCount);
    }
}

void CommandListProcessor::ProcessCommand(const DelayCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(DelayCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    auto delayConfig = &(pCommand->delayConfig);
    auto delayState = reinterpret_cast<DelayState*>(pCommand->delayState);
    auto enabled = pCommand->enabled;
    auto workBuffer = reinterpret_cast<void*>(pCommand->workBuffer);
    const auto numChannels = delayConfig->_numChannels;
    const int32_t* input[nn::audio::DelayParameter::ChannelCountMax];
    int32_t* output[nn::audio::DelayParameter::ChannelCountMax];
    for (int i = 0; i < numChannels; ++i)
    {
        if (!IsIndexInRange(pCommand->inputBufferIndex[i], 0, m_BufferCount) ||
            !IsIndexInRange(pCommand->outputBufferIndex[i], 0, m_BufferCount))
        {
            return;
        }

        input[i] = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex[i], m_SampleCount);
        output[i] = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex[i], m_SampleCount);
    }

    if(enabled)
    {
        if (delayConfig->_parameterStatus == EffectParameterStatus_Init)
        {
            InitializeDelayEffect(delayConfig, delayState, workBuffer);
        }
        else if (delayConfig->_parameterStatus == EffectParameterStatus_UpdateParam)
        {
            SetDelayEffectParameter(delayConfig, delayState);
        }
    }

    NN_AUDIO_TRACK_BUFFER(delayState, sizeof(DelayState), CachedBuffer_RequireFlush);
    ApplyDelayEffect(delayConfig, delayState, enabled, input, output, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const UpsampleCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(UpsampleCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const auto pInfo = reinterpret_cast<UpsamplerInfo*>(pCommand->pInfo);
    NN_AUDIO_TRACK_BUFFER(pInfo, sizeof(UpsamplerInfo), CachedBuffer_RequireInval);
    const auto count = Min(pCommand->mixBufferCount, pInfo->inputCount);
    const auto index = reinterpret_cast<int8_t*>(pCommand->input);

    NN_AUDIO_TRACK_BUFFER(index, count * sizeof(int8_t), CachedBuffer_RequireInval);
    auto pDestBuffers = reinterpret_cast<int32_t*>(pCommand->pDestBuffers);
    const auto inputSampleCount = pCommand->inputSampleCount;
    const auto outSampleCount = pInfo->sampleCount;

    for (int i = 0; i < count; ++i)
    {
        if (!IsIndexInRange(index[i], 0, m_BufferCount))
        {
            continue;
        }

        const auto pInData = GetMixBuffer(m_pMixBuffers, index[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(pInData, sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
        auto pOutData = GetMixBuffer(pDestBuffers, index[i], outSampleCount);
        NN_AUDIO_TRACK_BUFFER(pOutData, sizeof(int32_t) * outSampleCount, CachedBuffer_LocalBuffer);
        SrcProcessFrame(pOutData, pInData, outSampleCount, inputSampleCount, &pInfo->srcbl[i]);
    }
    dsp::FlushDataCache(pInfo->srcbl, sizeof(UpsamplerState) * count);
}

namespace {
NN_FORCEINLINE int32_t downMix(const int32_t downMixCoeff[], int32_t front, int32_t center, int32_t lfe, int32_t rear) NN_NOEXCEPT
{
    int64_t acc = 0;
    acc += static_cast<int64_t>(front)  * downMixCoeff[0];
    acc += static_cast<int64_t>(center) * downMixCoeff[1];
    acc += static_cast<int64_t>(lfe)    * downMixCoeff[2];
    acc += static_cast<int64_t>(rear)   * downMixCoeff[3];
    acc += (1 << DownMix6chTo2chCommand::CoeffShift) >> 1; // rounding error
    acc >>= DownMix6chTo2chCommand::CoeffShift;
    return static_cast<int32_t>(acc);
}
}

void CommandListProcessor::ProcessCommand(const DownMix6chTo2chCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(DownMix6chTo2chCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (!IsIndexInRange(pCommand->inputBufferIndices[nn::audio::ChannelMapping_FrontLeft], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->inputBufferIndices[nn::audio::ChannelMapping_FrontRight], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->inputBufferIndices[nn::audio::ChannelMapping_RearLeft], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->inputBufferIndices[nn::audio::ChannelMapping_RearRight], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->inputBufferIndices[nn::audio::ChannelMapping_FrontCenter], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->inputBufferIndices[nn::audio::ChannelMapping_LowFrequency], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndices[nn::audio::ChannelMapping_FrontLeft], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndices[nn::audio::ChannelMapping_FrontRight], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndices[nn::audio::ChannelMapping_RearLeft], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndices[nn::audio::ChannelMapping_RearRight], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndices[nn::audio::ChannelMapping_FrontCenter], 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndices[nn::audio::ChannelMapping_LowFrequency], 0, m_BufferCount))
    {
        return;
    }

    auto left      = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[nn::audio::ChannelMapping_FrontLeft], m_SampleCount);
    auto right     = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[nn::audio::ChannelMapping_FrontRight], m_SampleCount);
    auto rearLeft  = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[nn::audio::ChannelMapping_RearLeft], m_SampleCount);
    auto rearRight = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[nn::audio::ChannelMapping_RearRight], m_SampleCount);
    auto center    = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[nn::audio::ChannelMapping_FrontCenter], m_SampleCount);
    auto lfe       = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndices[nn::audio::ChannelMapping_LowFrequency], m_SampleCount);

    NN_AUDIO_TRACK_BUFFER(left     , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(right    , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(rearLeft , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(rearRight, sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(center   , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(lfe      , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);

    auto outputLeft      = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[nn::audio::ChannelMapping_FrontLeft], m_SampleCount);
    auto outputRight     = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[nn::audio::ChannelMapping_FrontRight], m_SampleCount);
    auto outputRearLeft  = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[nn::audio::ChannelMapping_RearLeft], m_SampleCount);
    auto outputRearRight = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[nn::audio::ChannelMapping_RearRight], m_SampleCount);
    auto outputCenter    = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[nn::audio::ChannelMapping_FrontCenter], m_SampleCount);
    auto outputLfe       = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndices[nn::audio::ChannelMapping_LowFrequency], m_SampleCount);

    NN_AUDIO_TRACK_BUFFER(outputLeft      , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(outputRight     , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(outputRearLeft  , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(outputRearRight , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(outputCenter    , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    NN_AUDIO_TRACK_BUFFER(outputLfe       , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);

    // mix down to LR channel
    for (auto i = 0; i < m_SampleCount; ++i)
    {
        outputLeft[i]  = downMix(pCommand->DownMixMatrixCoeff, left[i],  center[i], lfe[i], rearLeft[i]);
        outputRight[i] = downMix(pCommand->DownMixMatrixCoeff, right[i], center[i], lfe[i], rearRight[i]);
    }
    // clean up rest channel
    memset(outputRearLeft, 0, m_SampleCount * sizeof(int32_t));
    memset(outputRearRight, 0, m_SampleCount * sizeof(int32_t));
    memset(outputCenter, 0, m_SampleCount * sizeof(int32_t));
    memset(outputLfe, 0, m_SampleCount * sizeof(int32_t));
}

void CommandListProcessor::ProcessCommand(const AuxCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(AuxCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));


    if (!IsIndexInRange(pCommand->inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(pCommand->outputBufferIndex, 0, m_BufferCount))
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(input , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    auto output = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex, m_SampleCount);
    NN_AUDIO_TRACK_BUFFER(output , sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    auto pToUser = reinterpret_cast<AuxInfoDsp*>(pCommand->inputAuxInfo);
    auto pFromUser = reinterpret_cast<AuxInfoDsp*>(pCommand->outputAuxInfo);
    auto toUserBufferAddr = pCommand->inputBuffer;
    auto fromUserBufferAddr = pCommand->outputBuffer;
    auto countMax = pCommand->countMax;

    if (pCommand->enabled)
    {
        int32_t writeOffset = pCommand->writeOffset;
        int32_t updateCount = pCommand->updateCount;

        // write out to external buffer.
        dsp::WriteAuxBufferDsp(pToUser, toUserBufferAddr, countMax, input, m_SampleCount, writeOffset, updateCount);

        // read back from external buffer.
        int32_t read = dsp::ReadAuxBufferDsp(pFromUser, fromUserBufferAddr, countMax, output, m_SampleCount, writeOffset, updateCount);
        if ( read != m_SampleCount )
        {
            // TODO: (SIGLO-11898) need some notice
            // NN_DETAIL_AUDIO_TRACE("Read Return Buffer shorten. WO:%08x, RO:%08x, read:%08x\n", pFromUser->_writeOffsetCount, pFromUser->_readOffsetCount, read);
            memset(output + read, 0, (m_SampleCount - read) * sizeof(int32_t));
        }
    }
    else
    {
        dsp::ResetAuxBufferDsp(pToUser);
        dsp::ResetAuxBufferDsp(pFromUser);
        if (output != input )
        {
            memcpy(output, input, m_SampleCount * sizeof(int32_t));
        }
    }
}

#if defined(NN_AUDIO_ENABLE_DEVICE_SINK)
void CommandListProcessor::ProcessCommand(const DeviceSinkCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(DeviceSinkCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const char* name = pCommand->name;
    const auto index = pCommand->input;
    auto pSourceBuffers = reinterpret_cast<int32_t*>(pCommand->pMixBuffers);
    auto outSampleCount = 240;   // typical case

    auto pDevice = GetDevice(name, pCommand->sessionId);
    if (pDevice == nullptr)
    {
        return;
    }

    auto deviceBuffer = reinterpret_cast<int16_t*>(GetDeviceBuffer(pDevice));
    if(deviceBuffer == nullptr)
    {
        return;
    }

    const auto deviceChannelCount = GetDeviceChannelCount(pDevice);
    const auto deviceSampleRate = GetDeviceSampleRate(pDevice);
    const auto count = Min(deviceChannelCount, pCommand->inputCount);

    NN_AUDIO_TRACK_BUFFER(index, count * sizeof(int8_t), CachedBuffer_RequireInval);
    for (auto i = 0; i < count; ++i)
    {
        if (!IsIndexInRange(index[i], 0, m_BufferCount))
        {
            return;
        }
    }

    const auto inputSampleRate = 48000;
    if(deviceSampleRate == inputSampleRate)
    {
        for (int i = 0; i < count; ++i)
        {
            auto input = GetMixBuffer(pSourceBuffers, index[i], outSampleCount); // input always 48K
            NN_AUDIO_TRACK_BUFFER(input, outSampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
            auto output = &deviceBuffer[i];
            NN_AUDIO_TRACK_BUFFER(output, outSampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
            for (int j = 0; j < outSampleCount; ++j)
            {
                *output= static_cast<int16_t>(Clamp<16>(*input++));
                output = &output[deviceChannelCount];
            }
        }
    }
    else
    {
#if defined(NN_AUDIO_ENABLE_DEVICE_SINK_DOWNSAMPLER)
        auto pState = GetDeviceDownsampleState(pDevice);
        if (pState == nullptr)
        {
            return;
        }

        uint32_t ratio;
        switch(deviceSampleRate)
        {
        case 96000:
            ratio = NN_AUDIO_SRC_BL_DN_RATIO_48_96;
            break;
        case 44100:
        default:
            ratio = NN_AUDIO_SRC_BL_DN_RATIO_48_441;
            break;
        }

        outSampleCount = deviceSampleRate / 200 ; //first guess
        ResampleOutputGetNextFrameSize(pState, pDevice->InputCountMax, 240, outSampleCount, &outSampleCount);
        for (int i = 0; i < count; ++i)
        {
            auto input = GetMixBuffer(pSourceBuffers, index[i], 240); // input always 48K
            ResampleOutputFrame(&pState[i], deviceChannelCount, i,
                input, deviceBuffer, 240, outSampleCount, ratio);
                NN_AUDIO_TRACK_BUFFER(input, 240 * sizeof(int32_t), CachedBuffer_RequireFlush);
        }
#else
        NN_AUDIO_DSP_ASSERT(false, "DeviceSink does not supprot sample rate(%d).", deviceSampleRate);
#endif // defined(NN_AUDIO_ENABLE_DEVICE_SINK_DOWNSAMPLER)
    }
}
#endif // defined(NN_AUDIO_ENABLE_DEVICE_SINK)

void CommandListProcessor::ProcessCommand(const ReverbCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(ReverbCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    auto reverbConfig = &(pCommand->reverbConfig);
    NN_AUDIO_TRACK_BUFFER(reverbConfig, sizeof(*reverbConfig), CachedBuffer_Uncached);
    auto state = reinterpret_cast<ReverbState*>(pCommand->reverbState);
    auto enabled = pCommand->enabled;
    auto longSizePreDelaySupported = pCommand->longSizePreDelaySupported;
    auto workBuffer = reinterpret_cast<void*>(pCommand->workBuffer);
    const int32_t* input[nn::audio::ReverbParameter::ChannelCountMax];
    int32_t* output[nn::audio::ReverbParameter::ChannelCountMax];
    const auto numChannels = reverbConfig->_numChannels;
    for (int i = 0; i < numChannels; ++i)
    {
        if (!IsIndexInRange(pCommand->inputBufferIndex[i], 0, m_BufferCount) ||
            !IsIndexInRange(pCommand->outputBufferIndex[i], 0, m_BufferCount))
        {
            return;
        }

        input[i] = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(input[i], sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
        output[i] = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(output[i], sizeof(int32_t) * m_SampleCount, CachedBuffer_LocalBuffer);
    }

    if(enabled)
    {
        NN_AUDIO_TRACK_BUFFER(state, sizeof(ReverbState), dsp::CachedBuffer_LocalBuffer);
        if (reverbConfig->_parameterStatus == EffectParameterStatus_Init)
        {
            InitializeReverbEffect(reverbConfig, state, workBuffer, longSizePreDelaySupported);
        }
        else if (reverbConfig->_parameterStatus == EffectParameterStatus_UpdateParam)
        {
            UpdateReverbEffectParameter(reverbConfig, state);
        }
    }

    ApplyReverbEffect(reverbConfig, state, enabled, input, output, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const I3dl2ReverbCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(I3dl2ReverbCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    auto i3dl2ReverbConfig = &(pCommand->i3dl2ReverbConfig);
    NN_AUDIO_TRACK_BUFFER(i3dl2ReverbConfig, sizeof(*i3dl2ReverbConfig), CachedBuffer_Uncached);
    auto state = reinterpret_cast<I3dl2ReverbState*>(pCommand->i3dl2ReverbState);
    auto enabled = pCommand->enabled;
    auto workBuffer = reinterpret_cast<void*>(pCommand->workBuffer);
    const int32_t* input[nn::audio::I3dl2ReverbParameter::ChannelCountMax];
    int32_t* output[nn::audio::I3dl2ReverbParameter::ChannelCountMax];
    const auto numChannels = i3dl2ReverbConfig->_numChannels;
    for (int i = 0; i < numChannels; ++i)
    {
        if (!IsIndexInRange(pCommand->inputBufferIndex[i], 0, m_BufferCount) ||
            !IsIndexInRange(pCommand->outputBufferIndex[i], 0, m_BufferCount))
        {
            return;
        }

        input[i] = GetMixBuffer(m_pMixBuffers, pCommand->inputBufferIndex[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(input[i], m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
        output[i] = GetMixBuffer(m_pMixBuffers, pCommand->outputBufferIndex[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(output[i], m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
    }

    // Update parameters
    if(enabled)
    {
        NN_AUDIO_TRACK_BUFFER(state, sizeof(I3dl2ReverbState), dsp::CachedBuffer_LocalBuffer);

        const auto parameterStatus = i3dl2ReverbConfig->_parameterStatus;
        if(parameterStatus == EffectParameterStatus_Init)
        {
            InitializeI3dl2ReverbEffect(i3dl2ReverbConfig, state, workBuffer);
        }
        else if(parameterStatus == EffectParameterStatus_UpdateParam)
        {
            UpdateI3dl2ReverbEffectParameter(i3dl2ReverbConfig, state, false);
        }
    }

    ApplyI3dl2ReverbEffect(i3dl2ReverbConfig, state, enabled, input, output, m_SampleCount);
}

void CommandListProcessor::ProcessCommand(const PerformanceCommand* pCommand, Time startTime) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(PerformanceCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (pCommand->type == PerformanceCommandType_Begin)
    {
        auto pStartTime = reinterpret_cast<int32_t*>(pCommand->baseAddress + pCommand->offsetToStartTime);

        Time curTime = GetCurrentTime() - m_SkipTime;
        NN_AUDIO_TRACK_BUFFER(pStartTime, sizeof(int32_t), CachedBuffer_RequireInval | CachedBuffer_RequireFlush);
        *pStartTime = ConvertToMicroSeconds(curTime - startTime);
    }
    else if (pCommand->type == PerformanceCommandType_End)
    {
        auto pStartTime = reinterpret_cast<int32_t*>(pCommand->baseAddress + pCommand->offsetToStartTime);
        auto pCount = reinterpret_cast<int32_t*>(pCommand->baseAddress + pCommand->offsetToEntryCount);
        auto pProcessingTime = reinterpret_cast<int32_t*>(pCommand->baseAddress + pCommand->offsetToProcessingTime);

        // temp fix for SIGLO-19385
        //Is this invalidate required? Can pStartTime be modified on CPU before Begin and End commands?
        //dsp::InvalidateDataCache(pStartTime, sizeof(int32_t));

        NN_AUDIO_TRACK_BUFFER(pStartTime, sizeof(int32_t), CachedBuffer_RequireInval);
        NN_AUDIO_TRACK_BUFFER(pProcessingTime, sizeof(int32_t), CachedBuffer_RequireInval | CachedBuffer_RequireFlush);
        NN_AUDIO_TRACK_BUFFER(pCount, sizeof(int32_t), CachedBuffer_RequireInval | CachedBuffer_RequireFlush);

        Time curTime = GetCurrentTime() - m_SkipTime;
        *pProcessingTime = ConvertToMicroSeconds(curTime - startTime) - *pStartTime;
        *pCount += 1; // 計測するごとに一つカウントアップ

        // temp fix for SIGLO-19385
    }
    else // PerformanceCommandType_Invalid
    {
        // Pass
    }
}

void CommandListProcessor::ProcessCommand(const CircularBufferSinkCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(CircularBufferSinkCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const auto index = pCommand->input;
    const auto count = pCommand->inputCount;
    auto pSourceBuffers = m_pMixBuffers;

    auto sampleOffset = pCommand->currentOffset;
    const auto circularBufferAddress = reinterpret_cast<int8_t*>(pCommand->circularBuffer);
    const auto bufferSize = pCommand->circularBufferSize;

    NN_AUDIO_TRACK_BUFFER(circularBufferAddress, bufferSize, CachedBuffer_RequireFlush);

    if(bufferSize <= 0)
    {
        return;
    }

    for (int i = 0; i < count; ++i)
    {
        if (!IsIndexInRange(index[i], 0, m_BufferCount))
        {
            continue;
        }

        auto input = GetMixBuffer(pSourceBuffers, index[i], m_SampleCount);
        NN_AUDIO_TRACK_BUFFER(input, m_SampleCount * sizeof(int32_t), CachedBuffer_LocalBuffer);
        auto output = reinterpret_cast<int16_t*>(circularBufferAddress + sampleOffset);

        NN_AUDIO_TRACK_BUFFER(output, m_SampleCount * sizeof(int16_t), CachedBuffer_LocalBuffer);
        for (int j = 0; j < m_SampleCount; ++j)
        {
            output[j] = static_cast<int16_t>(Clamp<16>(input[j]));
        }

        sampleOffset += m_SampleCount * sizeof(output[0]);
        if (sampleOffset >= bufferSize)
        {
            sampleOffset = 0;
        }
    }
}

void CommandListProcessor::ProcessCommand(const ClearMixBufferCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(ClearMixBufferCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    if (m_pMixBuffers)
    {
        memset(m_pMixBuffers, 0, sizeof(int32_t) * m_BufferCount * m_SampleCount);
    }
}

void CommandListProcessor::ProcessCommand(const CopyMixBufferCommand* pCommand) NN_NOEXCEPT
{
    NN_AUDIO_TRACK_BUFFER(pCommand, sizeof(CopyMixBufferCommand), CachedBuffer_Uncached);
    NN_AUDIO_DSP_ASSERT(VerifyCommand(pCommand));

    const auto inputBufferIndex = pCommand->inputBufferIndex;
    const auto outputBufferIndex = pCommand->outputBufferIndex;

    // Check parameters
    if (!IsIndexInRange(inputBufferIndex, 0, m_BufferCount) ||
        !IsIndexInRange(outputBufferIndex, 0, m_BufferCount) ||
        m_pMixBuffers == nullptr)
    {
        return;
    }

    const auto input = GetMixBuffer(m_pMixBuffers, inputBufferIndex, m_SampleCount);
    auto output = GetMixBuffer(m_pMixBuffers, outputBufferIndex, m_SampleCount);

    if(output != input)
    {
        memcpy(output, input, m_SampleCount * sizeof(int32_t));
    }
}

void CommandListProcessor::Setup(const void* pCommandList, size_t commandListSize) NN_NOEXCEPT
{
    NN_AUDIO_DSP_ASSERT(commandListSize >= sizeof(CommandListHeader));

    m_pCommandList = pCommandList;
    m_CommandListSize = commandListSize;

    const auto pCommandListHeader = reinterpret_cast<const CommandListHeader*>(m_pCommandList);
    m_pCurrentCommand = reinterpret_cast<const uint8_t*>(m_pCommandList) + sizeof(CommandListHeader);
    m_BufferCount = pCommandListHeader->bufferCount;
    m_SampleCount = pCommandListHeader->sampleCount;
    m_TargetSampleRate = pCommandListHeader->sampleRate;
    m_CommandCount = pCommandListHeader->commandCount;
    m_pMixBuffers = reinterpret_cast<int32_t*>(pCommandListHeader->buffer);
    m_ProcessedCommandCount = 0;
}

void CommandListProcessor::SetProcessTimeMax(uint32_t processTimeMax) NN_NOEXCEPT
{
    m_ProcessTimeMax = processTimeMax;
}

uint64_t CommandListProcessor::Process() NN_NOEXCEPT
{
    static int frames;
    auto pCommand = m_pCurrentCommand;

    const auto beginCycleCount = GetCurrentThreadCycleCount();

    // Record processing start time if this is the first time.
    if(m_ProcessedCommandCount == 0)
    {
        m_StartTime = GetCurrentTime();
        m_SkipTime = 0;
    }
    else
    {
        m_SkipTime += GetCurrentTime() - m_EndTime;
    }

    ++frames;
    for (; m_ProcessedCommandCount < m_CommandCount; ++m_ProcessedCommandCount)
    {
        auto pCommandHeader = reinterpret_cast<const CommandHeader*>(pCommand);
        NN_AUDIO_DSP_ASSERT(pCommandHeader->magic == CommandHeaderMagicNumber);
        NN_AUDIO_DSP_ASSERT(m_CommandListSize >= (pCommand - reinterpret_cast<const uint8_t*>(m_pCommandList)) + static_cast<uintptr_t>(pCommandHeader->size));

        if(pCommandHeader->isEnabled)
        {
            const auto elapsedCycleCount = GetCurrentThreadCycleCount() - beginCycleCount;

            if(elapsedCycleCount + pCommandHeader->estimatedProcessingTime > m_ProcessTimeMax)
            {
                break;
            }

            switch (pCommandHeader->id)
            {
                case CommandId_PcmInt16DataSource:
                {
                    ProcessCommand(reinterpret_cast<const PcmInt16DataSourceCommand*>(pCommand));
                    break;
                }
                case CommandId_AdpcmDataSource:
                {
                    ProcessCommand(reinterpret_cast<const AdpcmDataSourceCommand*>(pCommand));
                    break;
                }
                case CommandId_OpusDataSource:
                {
                    ProcessCommand(reinterpret_cast<const OpusDataSourceCommand*>(pCommand));
                    break;
                }
                case CommandId_Volume:
                {
                    ProcessCommand(reinterpret_cast<const VolumeCommand*>(pCommand));
                    break;
                }
                case CommandId_VolumeRamp:
                {
                    ProcessCommand(reinterpret_cast<const VolumeRampCommand*>(pCommand));
                    break;
                }
                case CommandId_BiquadFilter:
                {
                    ProcessCommand(reinterpret_cast<const BiquadFilterCommand*>(pCommand));
                    break;
                }
                case CommandId_Mix:
                {
                    ProcessCommand(reinterpret_cast<const MixCommand*>(pCommand));
                    break;
                }
                case CommandId_MixRamp:
                {
                    ProcessCommand(reinterpret_cast<const MixRampCommand*>(pCommand));
                    break;
                }
                case CommandId_MixRampGrouped:
                {
                    ProcessCommand(reinterpret_cast<const MixRampGroupedCommand*>(pCommand));
                    break;
                }
                case CommandId_DepopPrepare:
                {
                    ProcessCommand(reinterpret_cast<const DepopPrepareCommand*>(pCommand));
                    break;
                }
                case CommandId_DepopForMixBuffers:
                {
                    ProcessCommand(reinterpret_cast<const DepopForMixBuffersCommand*>(pCommand));
                    break;
                }
                case CommandId_Delay:
                {
                    ProcessCommand(reinterpret_cast<const DelayCommand*>(pCommand));
                    break;
                }
                case CommandId_Reverb:
                {
                    ProcessCommand(reinterpret_cast<const ReverbCommand*>(pCommand));
                    break;
                }
                case CommandId_I3dl2Reverb:
                {
                    ProcessCommand(reinterpret_cast<const I3dl2ReverbCommand*>(pCommand));
                    break;
                }
                case CommandId_Upsample:
                {
                    ProcessCommand(reinterpret_cast<const UpsampleCommand*>(pCommand));
                    break;
                }
                case CommandId_DownMix6chTo2ch:
                {
                    ProcessCommand(reinterpret_cast<const DownMix6chTo2chCommand*>(pCommand));
                    break;
                }
#if defined(NN_AUDIO_ENABLE_DEVICE_SINK)
                case CommandId_DeviceSink:
                {
                    ProcessCommand(reinterpret_cast<const DeviceSinkCommand*>(pCommand));
                    break;
                }
#endif
                case CommandId_CircularBufferSink:
                {
                    ProcessCommand(reinterpret_cast<const CircularBufferSinkCommand*>(pCommand));
                    break;
                }
                case CommandId_Aux:
                {
                    ProcessCommand(reinterpret_cast<const AuxCommand*>(pCommand));
                    break;
                }
                case CommandId_Performance:
                {
                    ProcessCommand(reinterpret_cast<const PerformanceCommand*>(pCommand), m_StartTime);
                    break;
                }
                case CommandId_ClearMixBuffer:
                {
                    ProcessCommand(reinterpret_cast<const ClearMixBufferCommand*>(pCommand));
                    break;
                }
                case CommandId_CopyMixBuffer:
                {
                    ProcessCommand(reinterpret_cast<const CopyMixBufferCommand*>(pCommand));
                    break;
                }
                default:
                {
                    NN_AUDIO_DSP_LOG("Unknown command (%d)\n", pCommandHeader->id);
                    NN_AUDIO_DSP_ASSERT(false);
                    break;
                }
            }
        }

        pCommand += pCommandHeader->size;
    }

    m_EndTime = GetCurrentTime();
    m_pCurrentCommand = pCommand;

    return GetCurrentThreadCycleCount() - beginCycleCount;
} // NOLINT(impl/function_size)

int32_t CommandListProcessor::GetCommandCount() const NN_NOEXCEPT
{
    return m_CommandCount;
}

int32_t CommandListProcessor::GetSampleCount() const NN_NOEXCEPT
{
    return m_SampleCount;
}

int32_t CommandListProcessor::GetTargetSampleRate() const NN_NOEXCEPT
{
    return m_TargetSampleRate;
}

int32_t* CommandListProcessor::GetMixBuffers() const NN_NOEXCEPT
{
    return m_pMixBuffers;
}

int32_t CommandListProcessor::GetBufferCount() const NN_NOEXCEPT
{
    return m_BufferCount;
}

int32_t CommandListProcessor::GetProcessedCommandCount() const NN_NOEXCEPT
{
    return m_ProcessedCommandCount;
}

int32_t CommandListProcessor::GetRemainCommandCount() const NN_NOEXCEPT
{
    return m_CommandCount - m_ProcessedCommandCount;
}

}}}  // namespace nn::audio::dsp

