﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <limits>
#include <nn/nn_Macro.h>
#include <nn/util/util_BytePtr.h>

#include "audio_BuildDefinition.h"
#include "audio_BitArray.h"

namespace nn { namespace audio { namespace common {

template< typename IndexType >
class IndexAllocator
{
public:
    NN_STATIC_ASSERT(std::is_unsigned<IndexType>::value);

    IndexAllocator()
        : m_Range(0)
        , m_NextStartIndex(0)
    {}

    ~IndexAllocator()
    {
        m_Range = 0;
    }

    /**
     * @brief           必要なバッファサイズを計算します。
     * @param[in]       size_t requiredIndexCount 管理したいインデックス数を指定します。
     * @return          バッファサイズ
     */
    static size_t CalcBufferSize(size_t requiredIndexCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(std::numeric_limits<IndexType>::max() > requiredIndexCount);
        return BitArray<IndexType>::CalcBufferSize(static_cast<uint32_t>(requiredIndexCount));
    }

    /**
     * @brief           初期化します。
     * @param[in]       size_t indexCount  管理したいインデックス数を指定します。
     * @param[in]       void * buffer      CalcBufferSize() のサイズをもつバッファ
     * @param[in]       size_t bufferSize  buffer のバッファサイズ
     */
    void Initialize(size_t indexCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(CalcBufferSize(indexCount) <= bufferSize);
        m_IndexPool.Initialize(buffer, bufferSize);

        // 端数部分を 使用済みとして Mark
        auto f = m_IndexPool.Size() - indexCount;
        for (uint32_t i = 0; i < f; ++i)
        {
            m_IndexPool.Set(static_cast<uint32_t>(indexCount + i));
        }

        m_Range = indexCount;
        m_NextStartIndex = 0;
    }

    /**
     * @brief           requestCount 分のインデックスを確保します。
     * @param[in]       IndexType * pOutInd
     * @param[in]       int requestCount
     * @return          成功したら true を返し、 pOutInd にインデックスが書き込まれます。
     * @detail
     * - ここで確保されたインデックスは、連続であることが保証されます。
     *   いいかえると、断片化が起きえますので、その可能性を承知の上で利用してください。
     * - 内部では m_NextStartIndex の位置を起点とした線形探索を行い、最初に見つかった
     *   requestCount 分の連続領域を返す実装となっています。
     */
    bool Alloc(IndexType* pOutInd, int requestCount) NN_NOEXCEPT
    {
        bool found = false;
        int count = 0;
        IndexType ind = 0;

        for (uint32_t i = m_NextStartIndex; found == false && i < m_Range; i++)
        {
            if (m_IndexPool[i])
            {
                count = 0;
            }
            else
            {
                count++;
                if (count >= requestCount)
                {
                    found = true;
                    ind = static_cast<IndexType>(i - (count - 1));
                }
            }
        }
        count = 0;

        for (uint32_t i = 0; found == false && i < m_NextStartIndex; i++)
        {
            if (m_IndexPool[i])
            {
                count = 0;
            }
            else
            {
                count++;
                if (count >= requestCount)
                {
                    found = true;
                    ind = static_cast<IndexType>(i - (count - 1));
                }
            }
        }

        if (found)
        {
            m_NextStartIndex = ind + static_cast<IndexType>(requestCount);
            *pOutInd = ind;

            for (auto i = ind; i < ind + static_cast<IndexType>(requestCount) && i < m_Range; i++)
            {
                m_IndexPool.Set(i);
            }
        }
        return found;
    }

    /**
     * @brief           インデックスを解放します。
     * @param[in]       int startIndex 解放するインデックスの先頭
     * @param[in]       int count      解放する数
     * @detail
     * - startIndex から始まる count 分の連続領域を解放します。
     */
    void Free(int startIndex, int count) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(startIndex >= 0 && static_cast<size_t>(startIndex + count) <= m_Range);
        for (int i = startIndex; i < startIndex + count && static_cast<size_t>(i) < m_Range; i++)
        {
            m_IndexPool.Reset(i);
        }
    }

    /**
     * @brief           現在の空きインデックス数の総数を返します。
     * @detail
     * - ここで帰る値は空きインデックスの総数であり、必ずしも連続していることが保証されないことに注意してください。
     *   ここで変える値を引数に指定して Alloc() を呼んだとしても、失敗する可能性があります。
     */
    size_t GetTotalFreeSize() const NN_NOEXCEPT
    {
        return m_IndexPool.Size() - m_IndexPool.Count();
    }

private:
    BitArray<IndexType> m_IndexPool;
    size_t m_Range;
    IndexType m_NextStartIndex;
};


// Generarl usage of index allocator.
template <typename Type>
class InstanceAllocator
{
public:
    /**
     * @brief           ワークバッファに要求されるアドレスアライメントを返します。
     * @return          アライメントサイズ
     * @detail
     * - Initialize() に指定する buffer のアドレスは、ここで返る値にアライメントされている必要があります。
     */
    static size_t GetWorkBufferAlignSize() NN_NOEXCEPT
    {
        return NN_ALIGNOF(Type) > NN_ALIGNOF(InternalAllocator) ? NN_ALIGNOF(Type) : NN_ALIGNOF(InternalAllocator);
    }

    /**
     * @brief           ワークバッファサイズを計算します。
     * @param[in]       size_t instanceCount 管理したいインスタンス数
     * @return          ワークバッファサイズ
     */
    static size_t GetWorkBufferSize(size_t instanceCount) NN_NOEXCEPT
    {
        size_t size = 0;
        size += nn::util::align_up(InternalAllocator::CalcBufferSize(instanceCount), NN_ALIGNOF(Type));
        size += sizeof(Type) * instanceCount;
        return nn::util::align_up(size, GetWorkBufferAlignSize());
    }

public:
    InstanceAllocator() NN_NOEXCEPT
        : m_Instances(nullptr)
    {}

    virtual ~InstanceAllocator() NN_NOEXCEPT
    {
        m_Instances = nullptr;
    }

    /**
     * @brief           初期化します。
     * @param[in]       size_t instanceCount 管理したいインスタンス数
     * @param[in]       void * buffer        GetWorkBufferSize() のサイズを持ち、GetWorkBufferAlignSize() でアラインされたバッファ
     * @param[in]       size_t bufferSize    buffer のサイズ
     */
    void Initialize(size_t instanceCount, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(buffer != nullptr && bufferSize >= GetWorkBufferSize(instanceCount));
        NN_SDK_ASSERT_ALIGNED(buffer, GetWorkBufferAlignSize());

        auto ptr = nn::util::BytePtr(buffer);
        size_t bufSize = nn::util::align_up(InternalAllocator::CalcBufferSize(instanceCount), NN_ALIGNOF(Type));
        m_IdAllocator.Initialize(instanceCount, ptr.Get(), bufSize);
        m_Instances = (ptr + bufSize).Get<Type>();
        m_InsanceCount = static_cast<int>(instanceCount);
        m_AllocatedCount = 0;
    }

    /**
     * @brief           インスタンスを確保します。
     * @param[in]       int count 確保したいインスタンス数
     * @return          インスタンス配列の先頭アドレス
     */
    Type* Alloc(int count = 1) NN_NOEXCEPT
    {
        uint32_t id = 0;
        if (m_IdAllocator.Alloc(&id, count) == false)
        {
            return nullptr;
        }
        m_AllocatedCount += count;
        NN_SDK_ASSERT(m_AllocatedCount <= m_InsanceCount);

        return &m_Instances[id];
    }

    /**
     * @brief           インスタンスを解放します。
     * @param[in]       const Type * ptr 解放するインスタンス配列の先頭アドレス
     * @param[in]       int count        解放するインスタンス数
     */
    void Free(const Type* ptr, int count = 1) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Instances <= ptr);

        int startId = static_cast<int>(ptr - m_Instances);
        m_IdAllocator.Free(startId, count);
        m_AllocatedCount -= count;
        NN_SDK_ASSERT(m_AllocatedCount >= 0);

    }

    /**
     * @brief           確保済みインスタンス数を返します。
     */
    size_t GetAllocatedCount() const NN_NOEXCEPT
    {
        return m_AllocatedCount;
    }

    /**
     * @brief           確保済みインスタンス数を返します。
     * @detail
     * - 空インスタンス数をチェックするのみで、それらが連続であることを保証するものではない。
     *   よって、これが true を返すからと言って、必ずしも Alloc() が成功するわけではない。
     */
    bool HasInstances(int count) const NN_NOEXCEPT
    {
        auto rest = m_InsanceCount - m_AllocatedCount;
        NN_SDK_ASSERT(m_IdAllocator.GetTotalFreeSize() == static_cast<size_t>(rest));
        return rest >= count;
    }

private:
    using InternalAllocator = IndexAllocator<uint32_t>;
    InternalAllocator m_IdAllocator;
    Type* m_Instances;

    // mIdAllocator の GetTotalFreeSize() を利用すればこれらの変数は不要だが、
    // 計算効率の為にキャッシュしておく
    int32_t m_InsanceCount;
    int32_t m_AllocatedCount;
};

}}}  // namespace nn::audio::common
