﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined(_NX_AUDIO_DSP_)
#include "../dsp/audiodsp.h"
#define NN_AUDIO_COMMAND_DUMP_LOG(...) printf(__VA_ARGS__)
#else
#include <nn/nn_SdkLog.h>
#include <nn/audio/detail/audio_Log.h>
#define NN_AUDIO_COMMAND_DUMP_LOG(...) NN_DETAIL_AUDIO_INFO(__VA_ARGS__)
#endif

#include "audio_Command.h"

// TODO: Dump qf format
// TODO: Dump more detail information if these're called on ADSP (VoiceState etc...)

namespace {

void DumpValue(const char* name, int value)
{
    NN_UNUSED(name);
    NN_UNUSED(value);

    NN_AUDIO_COMMAND_DUMP_LOG(" %s (%d)\n", name, value);
}

void DumpValue(const char* name, uint32_t value)
{
    NN_UNUSED(name);
    NN_UNUSED(value);

    NN_AUDIO_COMMAND_DUMP_LOG(" %s (%u)\n", name, value);
}

// TODO: Workaround for unused function warning on 32 bit envirionment.
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
void DumpValue(const char* name, uint64_t value)
{
    NN_UNUSED(name);
    NN_UNUSED(value);

    NN_AUDIO_COMMAND_DUMP_LOG(" %s (%llu)\n", name, value);
}
#endif

void DumpValue(const char* name, float value)
{
    NN_UNUSED(name);
    NN_UNUSED(value);

    NN_AUDIO_COMMAND_DUMP_LOG(" %s (%f)\n", name, value);
}

void DumpValue(const char* name, const char* value)
{
    NN_UNUSED(name);
    NN_UNUSED(value);

    NN_AUDIO_COMMAND_DUMP_LOG(" %s (%s)\n", name, value);
}

void DumpValue(const char* name, const nn::audio::EffectParameterStatus value)
{
    NN_UNUSED(name);
    NN_UNUSED(value);

    switch(value)
    {
    case nn::audio::EffectParameterStatus_Init:
        NN_AUDIO_COMMAND_DUMP_LOG(" %s (Init)\n", name);
        break;
    case nn::audio::EffectParameterStatus_UpdateParam:
        NN_AUDIO_COMMAND_DUMP_LOG(" %s (UpdateParam)\n", name);
        break;
    case nn::audio::EffectParameterStatus_Updated:
        NN_AUDIO_COMMAND_DUMP_LOG(" %s (Updated)\n", name);
        break;
    default:
        NN_AUDIO_COMMAND_DUMP_LOG(" %s (Invalid)\n", name);
        break;
    }
}

} // anonymous namespace

#define NN_AUDIO_DUMP_COMMAND_VALUE(command, value) DumpValue( #value, command->value)

namespace nn { namespace audio { namespace common {

void DumpCommandStructSizeAndAligns() NN_NOEXCEPT
{
#define NN_AUDIO_DUMP_SIZE_AND_ALIGN(name) NN_AUDIO_COMMAND_DUMP_LOG( #name ": size(%zu) align(%zu)\n", sizeof(name), NN_ALIGNOF(name))

    NN_AUDIO_DUMP_SIZE_AND_ALIGN(CommandListHeader);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(CommandHeader);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(PcmInt16DataSourceCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(AdpcmDataSourceCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(VolumeCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(VolumeRampCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(BiquadFilterCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(MixCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(MixRampCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(MixRampGroupedCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(DepopPrepareCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(DepopForMixBuffersCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(DelayCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(ReverbCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(DeviceSinkCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(CircularBufferSinkCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(UpsampleCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(DownMix6chTo2chCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(I3dl2ReverbCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(AuxCommand);
    NN_AUDIO_DUMP_SIZE_AND_ALIGN(PerformanceCommand);

#undef NN_AUDIO_DUMP_SIZE_AND_ALIGN
}

void DumpCommandListHeader(const CommandListHeader* pCommandListHeader) NN_NOEXCEPT
{
    NN_UNUSED(pCommandListHeader);

    NN_AUDIO_COMMAND_DUMP_LOG("CommandListHeader: size(%d) commandCount(%d) buffer(%d) bufferCount(%d) sampleCount(%d) sampleRate(%d)\n"
            , pCommandListHeader->size
            , pCommandListHeader->commandCount
            , pCommandListHeader->buffer
            , pCommandListHeader->bufferCount
            , pCommandListHeader->sampleCount
            , pCommandListHeader->sampleRate);
}

void DumpCommandHeader(const CommandHeader* pCommandHeader) NN_NOEXCEPT
{
    NN_UNUSED(pCommandHeader);

    NN_AUDIO_COMMAND_DUMP_LOG("magic(%u) size(%d) id(%d) estimatedProcessingTime(%u) nodeId(%u) isEnabled(%c)\n", pCommandHeader->magic, pCommandHeader->size, pCommandHeader->id, pCommandHeader->estimatedProcessingTime, pCommandHeader->nodeId, pCommandHeader->isEnabled ? 't' : 'f');
}

void DumpCommand(const PcmInt16DataSourceCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("PcmInt16DataSource: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, sampleRate);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, pitch);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, channel);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, channelCount);
    for(auto i = 0u; i < sizeof(pCommand->waveBuffer) / sizeof(pCommand->waveBuffer[0]); ++i)
    {
        const auto pWaveBuffer = &pCommand->waveBuffer[i];
        NN_AUDIO_COMMAND_DUMP_LOG("waveBuffer[%d]\n", i);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, buffer);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, size);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, startSampleOffset);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, endSampleOffset);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, loop);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, isEndOfStream);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, pContext);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, contextSize);
    }
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, state);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const AdpcmDataSourceCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("AdpcmDataSource: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, sampleRate);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, pitch);
    // NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, channel);
    // NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, channelCount);
    for(auto i = 0u; i < sizeof(pCommand->waveBuffer) / sizeof(pCommand->waveBuffer[0]); ++i)
    {
        const auto pWaveBuffer = &pCommand->waveBuffer[i];
        NN_AUDIO_COMMAND_DUMP_LOG("waveBuffer[%d]\n", i);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, buffer);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, size);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, startSampleOffset);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, endSampleOffset);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, loop);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, isEndOfStream);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, pContext);
        NN_AUDIO_DUMP_COMMAND_VALUE(pWaveBuffer, contextSize);
    }
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, state);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, adpcmParameter);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const VolumeCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("Volume: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, volume);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const VolumeRampCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("VolumeRamp: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, volume0);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, volume1);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const BiquadFilterCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("BiquadFilterCommand: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, numerator[0]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, numerator[1]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, numerator[2]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, denominator[0]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, denominator[1]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, state);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, needInitialization);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const MixCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("Mix: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, volume);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const MixRampCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("MixRamp: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, volume0);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, volume1);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, state);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const MixRampGroupedCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("MixRampGrouped: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, mixBufferCount);
    for (auto i = 0; i < pCommand->mixBufferCount; ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" inputBufferIndex[%d] (%d)\n", i, pCommand->inputBufferIndices[i]);
        NN_AUDIO_COMMAND_DUMP_LOG(" inputBufferIndex[%d] (%d)\n", i, pCommand->inputBufferIndices[i]);
        NN_AUDIO_COMMAND_DUMP_LOG(" volume0[%d] (%d)\n", i, pCommand->volume0[i]);
        NN_AUDIO_COMMAND_DUMP_LOG(" volume1[%d] (%d)\n", i, pCommand->volume1[i]);
    }
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, state);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const DepopPrepareCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("DepopPrepare: ");
    DumpCommandHeader(&pCommand->header);
    for (auto i = 0; i < sizeof(pCommand->outputBufferIndices) / sizeof(pCommand->outputBufferIndices[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" outputBufferIndex[%d] (%d)\n", i, pCommand->outputBufferIndices[i]);
    }
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, state);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, mixBufferCount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, depopBuffer);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const DepopForMixBuffersCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("DepopForMixBuffer: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, mixBufferOffset);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, mixBufferCount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, decay);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, depopBuffer);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const DelayCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("Delay: ");
    DumpCommandHeader(&pCommand->header);
    for(auto i = 0u; i < sizeof(pCommand->inputBufferIndex) / sizeof(pCommand->inputBufferIndex[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" inputBufferIndex[%d] (%d)\n", i, pCommand->inputBufferIndex[i]);
    }
    for(auto i = 0u; i < sizeof(pCommand->outputBufferIndex) / sizeof(pCommand->outputBufferIndex[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" outputBufferIndex[%d] (%d)\n", i, pCommand->outputBufferIndex[i]);
    }

    const auto* pDelayConfig = &pCommand->delayConfig;
    for(auto i = 0u; i < sizeof(pDelayConfig->_input) / sizeof(pDelayConfig->_input[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" _input[%d] (%d)\n", i, pDelayConfig->_input[i]);
    }
    for(auto i = 0u; i < sizeof(pDelayConfig->_output) / sizeof(pDelayConfig->_output[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" _output[%d] (%d)\n", i, pDelayConfig->_output[i]);
    }
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _numChannelCountMax);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _numChannels);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _delayTimeMax);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _delayTime);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _sampleRate);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _inGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _feedbackGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _outGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _dryGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _channelSpread);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _lowPassAmount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pDelayConfig, _parameterStatus);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, delayState);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, workBuffer);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, enabled);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const UpsampleCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("Upsample: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, pDestBuffers);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, input);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, mixBufferCount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputSampleCount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputSampleRate);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, pInfo);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const DownMix6chTo2chCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("DownMix6chTo2chCommand: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndices[0]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndices[1]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndices[2]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndices[3]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndices[4]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndices[5]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndices[0]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndices[1]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndices[2]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndices[3]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndices[4]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndices[5]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, DownMixMatrixCoeff[0]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, DownMixMatrixCoeff[1]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, DownMixMatrixCoeff[2]);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, DownMixMatrixCoeff[3]);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const AuxCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("Aux: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputAuxInfo);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputAuxInfo);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBuffer);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBuffer);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, countMax);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, writeOffset);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, updateCount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, enabled);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const DeviceSinkCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("DeviceSink: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, name);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, sessionId);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, pMixBuffers);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputCount);
    for(auto i = 0u; i < sizeof(pCommand->input) / sizeof(pCommand->input[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" input[%d] (%d)\n", i, pCommand->input[i]);
    }
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const CircularBufferSinkCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("CircularBufferSink: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputCount);
    for(auto i = 0u; i < sizeof(pCommand->input) / sizeof(pCommand->input[0]); ++i)
    {
        NN_AUDIO_COMMAND_DUMP_LOG(" input[%d] (%d)\n", i, pCommand->input[i]);
    }
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, circularBuffer);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, circularBufferSize);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, currentOffset);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, format);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const ReverbCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("Reverb: ");
    DumpCommandHeader(&pCommand->header);
    for(auto i = 0u; i < sizeof(pCommand->inputBufferIndex) / sizeof(pCommand->inputBufferIndex[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex[i]);
    }
    for(auto i = 0u; i < sizeof(pCommand->outputBufferIndex) / sizeof(pCommand->outputBufferIndex[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex[i]);
    }

    const auto* pReverbConfig = &pCommand->reverbConfig;

    for(auto i = 0u; i < sizeof(pReverbConfig->_input) / sizeof(pReverbConfig->_input[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _input[i]);
    }
    for(auto i = 0u; i < sizeof(pReverbConfig->_output) / sizeof(pReverbConfig->_output[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _output[i]);
    }

    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _numChannelCountMax);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _numChannels);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _sampleRate);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _earlyMode);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _earlyGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _predelayTime);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _lateMode);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _lateGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _decayTime);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _highFrequencyDecayRatio);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _coloration);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _reverbGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _outGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _dryGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _parameterStatus);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, reverbState);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, workBuffer);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, enabled);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, longSizePreDelaySupported);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const I3dl2ReverbCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);

    NN_AUDIO_COMMAND_DUMP_LOG("I3dl2Reverb: ");
    DumpCommandHeader(&pCommand->header);
    for (auto i = 0u; i < sizeof(pCommand->inputBufferIndex) / sizeof(pCommand->inputBufferIndex[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, inputBufferIndex[i]);
    }
    for (auto i = 0u; i < sizeof(pCommand->outputBufferIndex) / sizeof(pCommand->outputBufferIndex[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, outputBufferIndex[i]);
    }

    const auto* pReverbConfig = &pCommand->i3dl2ReverbConfig;

    for (auto i = 0u; i < sizeof(pReverbConfig->_input) / sizeof(pReverbConfig->_input[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _input[i]);
    }
    for (auto i = 0u; i < sizeof(pReverbConfig->_output) / sizeof(pReverbConfig->_output[0]); ++i)
    {
        NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _output[i]);
    }

    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _numChannelCountMax);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _numChannels);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _sampleRate);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _roomHf);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _hfReference);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _decayTime);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _hfDecayRatio);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _room);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _reflections);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _reverb);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _diffusion);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _reflectionsDelay);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _density);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _dryGain);
    NN_AUDIO_DUMP_COMMAND_VALUE(pReverbConfig, _parameterStatus);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, i3dl2ReverbState);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, workBuffer);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, enabled);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const PerformanceCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);
    NN_AUDIO_COMMAND_DUMP_LOG("Performance: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, type);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, baseAddress);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, offsetToStartTime);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, offsetToEntryCount);
    NN_AUDIO_DUMP_COMMAND_VALUE(pCommand, offsetToProcessingTime);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommand(const ClearMixBufferCommand* pCommand) NN_NOEXCEPT
{
    NN_UNUSED(pCommand);
    NN_AUDIO_COMMAND_DUMP_LOG("ClearMixBuffer: ");
    DumpCommandHeader(&pCommand->header);
    NN_AUDIO_COMMAND_DUMP_LOG("\n");
}

void DumpCommandList(const void* pCommandList, size_t commandListSize) NN_NOEXCEPT
{
    NN_UNUSED(pCommandList);
    NN_UNUSED(commandListSize);

    auto p = static_cast<const int8_t*>(pCommandList);
    size_t offset = 0;

    NN_AUDIO_COMMAND_DUMP_LOG("DumpCommand:\n");
    NN_AUDIO_COMMAND_DUMP_LOG(" pCommandList(0x%p) size(%zu) sizeof(CommandListHeader)(%zu) \n", pCommandList, commandListSize, sizeof(CommandListHeader));
    auto pListHeader = reinterpret_cast<const CommandListHeader*>(p + offset);
    DumpCommandListHeader(pListHeader);

    if(offset >= commandListSize)
    {
        NN_AUDIO_COMMAND_DUMP_LOG("[ERROR] offset >= commandListSize at step1\n");
        return;
    }

    offset += sizeof(CommandListHeader);
    if(offset >= commandListSize)
    {
        NN_AUDIO_COMMAND_DUMP_LOG("[ERROR] offset >= commandListSize at step2\n");
        return;
    }

    for (int n = 0; n < pListHeader->commandCount; ++n)
    {
        auto header = reinterpret_cast<const CommandHeader*>(p + offset);
        auto command = reinterpret_cast<const void*>(p + offset);

        offset += header->size;
        if(offset > commandListSize)
        {
            NN_AUDIO_COMMAND_DUMP_LOG("[ERROR] offset >= commandListSize at step4\n");
            return;
        }

        switch (header->id)
        {
            case CommandId_PcmInt16DataSource:
            {
                DumpCommand(reinterpret_cast<const PcmInt16DataSourceCommand*>(command));
                break;
            }
            case CommandId_AdpcmDataSource:
            {
                DumpCommand(reinterpret_cast<const AdpcmDataSourceCommand*>(command));
                break;
            }
            case CommandId_Volume:
            {
                DumpCommand(reinterpret_cast<const VolumeCommand*>(command));
                break;
            }
            case CommandId_VolumeRamp:
            {
                DumpCommand(reinterpret_cast<const VolumeRampCommand*>(command));
                break;
            }
            case CommandId_BiquadFilter:
            {
                DumpCommand(reinterpret_cast<const BiquadFilterCommand*>(command));
                break;
            }
            case CommandId_Mix:
            {
                DumpCommand(reinterpret_cast<const MixCommand*>(command));
                break;
            }
            case CommandId_MixRamp:
            {
                DumpCommand(reinterpret_cast<const MixRampCommand*>(command));
                break;
            }
            case CommandId_MixRampGrouped:
            {
                DumpCommand(reinterpret_cast<const MixRampGroupedCommand*>(command));
                break;
            }
            case CommandId_DepopPrepare:
            {
                DumpCommand(reinterpret_cast<const DepopPrepareCommand*>(command));
                break;
            }
            case CommandId_DepopForMixBuffers:
            {
                DumpCommand(reinterpret_cast<const DepopForMixBuffersCommand*>(command));
                break;
            }
            case CommandId_Delay:
            {
                DumpCommand(reinterpret_cast<const DelayCommand*>(command));
                break;
            }
            case CommandId_Reverb:
            {
                DumpCommand(reinterpret_cast<const ReverbCommand*>(command));
                break;
            }
            case CommandId_I3dl2Reverb:
            {
                DumpCommand(reinterpret_cast<const I3dl2ReverbCommand*>(command));
                break;
            }
            case CommandId_Upsample:
            {
                DumpCommand(reinterpret_cast<const UpsampleCommand*>(command));
                break;
            }
            case CommandId_DownMix6chTo2ch:
            {
                DumpCommand(reinterpret_cast<const DownMix6chTo2chCommand*>(command));
                break;
            }
            case CommandId_DeviceSink:
            {
                DumpCommand(reinterpret_cast<const DeviceSinkCommand*>(command));
                break;
            }
            case CommandId_CircularBufferSink:
            {
                DumpCommand(reinterpret_cast<const CircularBufferSinkCommand*>(command));
                break;
            }
            case CommandId_Aux:
            {
                DumpCommand(reinterpret_cast<const AuxCommand*>(command));
                break;
            }
            case CommandId_Performance:
            {
                DumpCommand(reinterpret_cast<const PerformanceCommand*>(command));
                break;
            }
            case CommandId_ClearMixBuffer:
            {
                DumpCommand(reinterpret_cast<const ClearMixBufferCommand*>(command));
                break;
            }
            default:
            {
                NN_AUDIO_COMMAND_DUMP_LOG("Unknown CommandId:%08x\n", header->id);
                break;
            }
        }
    }
}  // NOLINT(impl/function_size)

}}} // nn::audio::common
