﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/audio_Common.h>

#include "../audio_AddrTypes.h"
#include "../audio_AuxTypes.h"
#include "../audio_EffectPrivateTypes.h"
#include "../common/audio_Util.h"
#include "../dsp/audio_Upsampler.h"
#include "../server/audio_VoiceState.h"
#include "audio_CommonWaveBuffer.h"
#include "../common/audio_BuildDefinition.h"

namespace {
enum CommandAddressAlign
{
    CommandAddressAlign_Default = NN_AUDIO_DSP_ADDR_SIZE
};
} // namespace
#define NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(typeName, typeSizeA32, typeSizeA64) \
    NN_STATIC_ASSERT(sizeof(typeName) == NN_AUDIO_DSP_ADDRESS_SELECT(typeSizeA32, typeSizeA64) && \
                     CommandAddressAlign_Default % alignof(typeName) == 0)

namespace nn {
namespace audio {

enum DataSourceBehaviorFlag : uint16_t
{
    DataSourceBehaviorFlag_IsPlayedSampleCountResetAtLoopPoint = 1u << 0,
    DataSourceBehaviorFlag_IsPitchAndSrcSkipped = 1u << 1,
};

enum CommandId : uint32_t
{
    CommandId_Invalid = 0,
    CommandId_PcmInt16DataSource,
    CommandId_AdpcmDataSource,
    CommandId_OpusDataSource,
    CommandId_Volume,
    CommandId_VolumeRamp,
    CommandId_BiquadFilter,
    CommandId_Mix,
    CommandId_MixRamp,
    CommandId_MixRampGrouped,
    CommandId_DepopPrepare,
    CommandId_DepopForMixBuffers,
    CommandId_Delay,
    CommandId_Upsample,
    CommandId_DownMix6chTo2ch,
    CommandId_Aux,
    CommandId_DeviceSink,
    CommandId_CircularBufferSink,
    CommandId_Reverb,
    CommandId_I3dl2Reverb,
    CommandId_Performance,
    CommandId_ClearMixBuffer,
    CommandId_CopyMixBuffer,
};

NN_PRAGMA_PUSH_WARNINGS

#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
// Warning for struct padding
#pragma clang diagnostic warning "-Wpadded"
#endif

struct CommandListHeader
{
    int32_t size;
    int32_t commandCount;
    DspAddr buffer;
    int32_t bufferCount; // User count + pipeline working buffer
    int32_t sampleCount;
    int32_t sampleRate;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(CommandListHeader, 24, 32);


const uint32_t CommandHeaderMagicNumber = 0xcafebabe;

struct CommandHeader
{
    uint32_t magic;
    int16_t size;
    int16_t id;
    uint32_t estimatedProcessingTime;
    NodeId nodeId;
    bool isEnabled;
#if NN_AUDIO_DSP_ADDR_SIZE == 4
    int8_t _padding0[3];
#elif NN_AUDIO_DSP_ADDR_SIZE == 8
    int8_t _padding0[7];
    int64_t _padding1; // to align 8 byte
#else
#error Unsupported NN_AUDIO_DSP_ADDR_SIZE.
#endif
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(CommandHeader, 20, 32);


struct PcmInt16DataSourceCommand
{
    CommandHeader header;
    int16_t outputBufferIndex;
    uint16_t behaviorFlags;
    int32_t sampleRate;
    int32_t pitch;
    int32_t channel;
    int32_t channelCount;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding1[4]);
    common::WaveBuffer waveBuffer[4];
    DspAddr state;

    static const CommandId Id = CommandId_PcmInt16DataSource;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(PcmInt16DataSourceCommand, 156, 224);

struct AdpcmDataSourceCommand
{
    CommandHeader header;
    int16_t outputBufferIndex;
    uint16_t behaviorFlags;
    int32_t sampleRate;
    int32_t pitch;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding1[4]);
    common::WaveBuffer waveBuffer[4];
    DspAddr state;
    DspAddr adpcmParameter;

    static const CommandId Id = CommandId_AdpcmDataSource;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(AdpcmDataSourceCommand, 152, 224);

struct OpusDataSourceCommand
{
    CommandHeader header;
    int16_t outputBufferIndex;
    uint16_t behaviorFlags;
    int32_t sampleRate;
    int32_t pitch;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding1[4]);
    common::WaveBuffer waveBuffer[4];
    DspAddr state;
    DspAddr opusParameter;

    static const CommandId Id = CommandId_OpusDataSource;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(OpusDataSourceCommand, 152, 224);

struct VolumeCommand
{
    CommandHeader header;
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    int32_t volume;

    static const CommandId Id = CommandId_Volume;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(VolumeCommand, 28, 40);

struct VolumeRampCommand
{
    CommandHeader header;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    int32_t volume0;
    int32_t volume1;

    static const CommandId Id = CommandId_VolumeRamp;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(VolumeRampCommand, 32, 48);

struct BiquadFilterCommand
{
    CommandHeader header;
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    int16_t numerator[3];
    int16_t denominator[2];
    int8_t _padding0[NN_AUDIO_DSP_ADDRESS_SELECT(2, 2)];
    DspAddr state;
    bool needInitialization;
    int8_t _padding1[NN_AUDIO_DSP_ADDRESS_SELECT(3, 7)];

    static const CommandId Id = CommandId_BiquadFilter;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(BiquadFilterCommand, 44, 64);

struct MixCommand
{
    CommandHeader header;
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    int32_t volume;

    static const CommandId Id = CommandId_Mix;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(MixCommand, 28, 40);

struct MixRampCommand
{
    CommandHeader header;
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    int32_t volume0;
    int32_t volume1;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);
    DspAddr state;

    static const CommandId Id = CommandId_MixRamp;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(MixRampCommand, 36, 56);

struct MixRampGroupedCommand
{
    CommandHeader header;
    int32_t mixBufferCount;
    int16_t inputBufferIndices[MixBufferCountMax];
    int16_t outputBufferIndices[MixBufferCountMax];
    int32_t volume0[MixBufferCountMax];
    int32_t volume1[MixBufferCountMax];
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);
    DspAddr state;

    static const CommandId Id = CommandId_MixRampGrouped;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(MixRampGroupedCommand, 316, 336);

struct DepopPrepareCommand
{
    CommandHeader header;
    int16_t outputBufferIndices[MixBufferCountMax];
    DspAddr state;
    int32_t mixBufferCount;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);
    DspAddr depopBuffer;

    static const CommandId Id = CommandId_DepopPrepare;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(DepopPrepareCommand, 80, 104);

struct DepopForMixBuffersCommand
{
    CommandHeader header;
    int32_t mixBufferOffset;
    int32_t mixBufferCount;
    int32_t decay;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);
    DspAddr depopBuffer;

    static const CommandId Id = CommandId_DepopForMixBuffers;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(DepopForMixBuffersCommand, 36, 56);

struct DelayCommand
{
    CommandHeader header;
    int16_t inputBufferIndex[nn::audio::DelayParameter::ChannelCountMax];
    int16_t outputBufferIndex[nn::audio::DelayParameter::ChannelCountMax];
    DelayParameter delayConfig;
    DspAddr delayState;
    DspAddr workBuffer;
    bool enabled;
    int8_t _padding[NN_AUDIO_DSP_ADDRESS_SELECT(3, 7)];

    static const CommandId Id = CommandId_Delay;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(DelayCommand, 112, 136);

struct ReverbCommand
{
    CommandHeader header;
    int16_t inputBufferIndex[nn::audio::ReverbParameter::ChannelCountMax];
    int16_t outputBufferIndex[nn::audio::ReverbParameter::ChannelCountMax];
    ReverbParameter reverbConfig;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding0[4]);
    DspAddr reverbState;
    DspAddr workBuffer;
    bool enabled;
    bool longSizePreDelaySupported;
    int8_t _padding1[NN_AUDIO_DSP_ADDRESS_SELECT(6, 6)];

    static const CommandId Id = CommandId_Reverb;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(ReverbCommand, 128, 152);

struct I3dl2ReverbCommand
{
    CommandHeader header;
    int16_t inputBufferIndex[nn::audio::I3dl2ReverbParameter::ChannelCountMax];
    int16_t outputBufferIndex[nn::audio::I3dl2ReverbParameter::ChannelCountMax];
    I3dl2ReverbParameter i3dl2ReverbConfig;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding0[4]);
    DspAddr i3dl2ReverbState;
    DspAddr workBuffer;
    bool enabled;
    int8_t _padding1[NN_AUDIO_DSP_ADDRESS_SELECT(3, 7)];

    static const CommandId Id = CommandId_I3dl2Reverb;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(I3dl2ReverbCommand, 132, 160);

const int SinkInputChannelCountMax = 6;

struct DeviceSinkCommand
{
    CommandHeader header;
    char name[256];   // TODO: サイズを適切な値にする
    uint32_t sessionId;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding0[4]);
    DspAddr pMixBuffers;
    int32_t inputCount;
    int16_t input[SinkInputChannelCountMax];
    int8_t _padding1[NN_AUDIO_DSP_ADDRESS_SELECT(4, 8)];

    static const CommandId Id = CommandId_DeviceSink;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(DeviceSinkCommand, 304, 328);

struct CircularBufferSinkCommand
{
    CommandHeader header;

    int32_t inputCount;
    int16_t input[SinkInputChannelCountMax];

    // Destination buffer
    DspAddr circularBuffer;
    int32_t circularBufferSize;
    int32_t currentOffset; // [byte]
    int32_t format;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding[4]);

    static const CommandId Id = CommandId_CircularBufferSink;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(CircularBufferSinkCommand, 52, 72);

struct UpsampleCommand
{
    CommandHeader header;
    DspAddr pDestBuffers;
    DspAddr input;
    int32_t mixBufferCount;
    int32_t inputBufferIndex;
    int32_t inputSampleCount;
    int32_t inputSampleRate;
    DspAddr pInfo;

    static const CommandId Id = CommandId_Upsample;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(UpsampleCommand, 48, 72);

struct DownMix6chTo2chCommand
{
    static const int CoeffShift = 16;
    static const int CoeffCount = 4;

    CommandHeader header;
    int16_t inputBufferIndices[6];
    int16_t outputBufferIndices[6];
    int32_t DownMixMatrixCoeff[4];
    int8_t _padding[NN_AUDIO_DSP_ADDRESS_SELECT(4, 8)];

    static const CommandId Id = CommandId_DownMix6chTo2ch;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(DownMix6chTo2chCommand, 64, 80);

struct AuxCommand
{
    CommandHeader header;
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding0[4]);
    DspAddr inputAuxInfo;
    DspAddr outputAuxInfo;
    DspAddr inputBuffer;
    DspAddr outputBuffer;
    uint32_t countMax;
    int32_t writeOffset;
    int32_t updateCount;
    bool enabled;
    int8_t _padding1[3];

    static const CommandId Id = CommandId_Aux;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(AuxCommand, 56, 88);

enum PerformanceCommandType
{
    PerformanceCommandType_Invalid = 0,
    PerformanceCommandType_Begin,
    PerformanceCommandType_End,
};

struct PerformanceCommand
{
    CommandHeader header;
    PerformanceCommandType type;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding0[4]);
    DspAddr baseAddress;
    int32_t offsetToStartTime;
    int32_t offsetToEntryCount;
    int32_t offsetToProcessingTime;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding1[4]);

    static const CommandId Id = CommandId_Performance;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(PerformanceCommand, 40, 64);

struct ClearMixBufferCommand
{
    CommandHeader header;

    static const CommandId Id = CommandId_ClearMixBuffer;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(ClearMixBufferCommand, 20, 32);

struct CopyMixBufferCommand
{
    CommandHeader header;
    int16_t inputBufferIndex;
    int16_t outputBufferIndex;
    NN_AUDIO_ENABLE_IF_DSP_ADDRESS_64(int8_t _padding0[4]);

    static const CommandId Id = CommandId_CopyMixBuffer;
};
NN_AUDIO_COMMAND_SIZE_AND_ALIGN_CHECK(CopyMixBufferCommand, 24, 40);

NN_PRAGMA_POP_WARNINGS

}  // namespace audio
}  // namespace nn
