﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_BitPack.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/audio/audio_AudioRendererTypes.h> // nn::audio::MixBufferCountMax
#include <nn/audio/audio_VoiceTypes.h>

#include "audio_SplitterInfoManager.h"
#include "audio_AddrTypes.h"
#include "common/audio_Util.h"
#include "common/audio_VoiceCommon.h"

namespace nn {
namespace audio {

class VoiceChannelResource;
typedef int32_t VoiceChannelResourceId;

typedef nn::util::IntrusiveList<VoiceInfo, nn::util::IntrusiveListBaseNodeTraits<VoiceInfo>> VoiceInfoList;
typedef nn::util::IntrusiveList<VoiceChannelResource, nn::util::IntrusiveListBaseNodeTraits<VoiceChannelResource>> VoiceChannelResourceList;

class VoiceChannelResource : public nn::util::IntrusiveListBaseNode<VoiceChannelResource>
{
public:
    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameter
    {
        VoiceChannelResourceId _id;
        float _mixVolume[MixBufferCountMax];
        bool _isInUse;
        int8_t _padding[11];
    };
    NN_AUDIO_INFOTYPE_CHECK(InParameter, 112);

protected:
    VoiceChannelResourceId m_Id;
    float m_MixVolume[MixBufferCountMax];
    bool m_IsInUse;
public:
    NN_FORCEINLINE VoiceChannelResource(int index)
        : m_Id(index), m_IsInUse(false)
    {
        ClearMixVolumes();
    }


    NN_FORCEINLINE VoiceChannelResourceId GetId() const NN_NOEXCEPT
    {
        return m_Id;
    };

    NN_FORCEINLINE bool IsInUse() const NN_NOEXCEPT
    {
        return m_IsInUse;
    };

    NN_FORCEINLINE void SetInUse(bool isInUse) NN_NOEXCEPT
    {
        m_IsInUse = isInUse;
    };

    NN_FORCEINLINE void ClearMixVolumes() NN_NOEXCEPT
    {
        memset(m_MixVolume, 0, sizeof(m_MixVolume));
    };

    NN_FORCEINLINE float GetMixVolume(int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
        return m_MixVolume[index];
    };

    NN_FORCEINLINE void SetMixVolume(float volume, int index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_MINMAX(volume, VoiceType::GetVolumeMin(), VoiceType::GetVolumeMax());
        NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
        m_MixVolume[index] = volume;
    };

    NN_FORCEINLINE void Update(InParameter* pInParameter) NN_NOEXCEPT
    {
        if(IsInUse())
        {
            pInParameter->_id = m_Id;
            memcpy(pInParameter->_mixVolume, m_MixVolume, sizeof(m_MixVolume));
            pInParameter->_isInUse = true;
        }
        else
        {
            pInParameter->_isInUse = false;
        }
    };
};

static const VoiceChannelResourceId InvalidVoiceChannelResouceId = -1;

struct VoiceInfo : nn::util::IntrusiveListBaseNode<VoiceInfo>
{
    enum ParameterStatus
    {
        ParameterStatus_Latest,
        ParameterStatus_New,
        ParameterStatus_Update,
    };

    const WaveBuffer* _waveBufferAddress[VoiceType::WaveBufferCountMax];  // TODO
    int32_t _waveBufferReleased;

    VoiceInfo()
    {
        // 他のパラメータは AcquireVoiceSlot で初期化される
        _inParameter._isInUse = false;
    }

    struct WaveBufferInternal
    {
        CpuAddr buffer;
        uint64_t size;
        int32_t startSampleOffset;
        int32_t endSampleOffset;
        bool loop;
        bool isEndOfStream;
        bool _isSentToServer;
        int8_t _padding0[5];
        CpuAddr pContext;
        uint64_t contextSize;
        int8_t _padding1[8];
    };
    NN_STATIC_ASSERT(sizeof(WaveBufferInternal) == 56);

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameter
    {
        int32_t _id;
        NodeId _nodeId;

        bool _isNew;
        bool _isInUse;
        uint8_t _playState;
        uint8_t _sampleFormat;
        int32_t _sampleRate;

        int32_t _priority;
        int32_t _sortingOrder;
        int32_t _channelCount;

        float _pitch;
        float _volume;

         BiquadFilterParameter _biquadFilterParameter[VoiceType::BiquadFilterCountMax];
        int32_t _waveBufferCount;
        int16_t _waveBufferHead;
        int8_t _padding0[6];
        CpuAddr _pAdditionalParameter;
        uint64_t _additionalParameterSize;
        MixId _destinationMixId;
        common::SplitterInfoId _splitterInfoId;

        WaveBufferInternal _waveBuffer[VoiceType::WaveBufferCountMax];
        VoiceChannelResourceId _voiceChannelResourceIds[VoiceType::ChannelCountMax];
        bool _isVoiceDroppedFlagClearRequested;
        uint8_t _waveBufferFlushRequestedCount;
        int8_t _padding1[2];
        VoiceBehaviorFlagSet _behaviorFlags;
        uint32_t _audioCodecBufferOffset;
        uint32_t _audioCodecBufferSize;
        int8_t _padding2[8];
    };
    NN_AUDIO_INFOTYPE_CHECK(InParameter,  368);

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN OutStatus
    {
        int64_t _playedSampleCount;
        int32_t _waveBufferConsumed;
        bool _voiceDroppedFlag;
        int8_t _padding[3];
    };
    NN_AUDIO_INFOTYPE_CHECK(OutStatus, 16);

    InParameter _inParameter;
    OutStatus _outStatus;

    int nodeIdBase;
    int nodeIdVariation;
    int32_t waveBufferAppended;
    VoiceChannelResource* _pVoiceChannelResources[VoiceType::ChannelCountMax];
    int8_t _reserved[8];
};

class VoiceInfoManager
{
private:
    int m_VoiceInfoCount;
    int m_VoiceInfoUsedCount;
    VoiceInfo* m_VoiceInfos;
    VoiceInfoList m_VoiceInfoUsedList;
    VoiceInfoList m_VoiceInfoFreeList;

    int m_VoiceChannelResourceCount;
    int m_VoiceChannelResourceUsedCount;
    VoiceChannelResource* m_VoiceChannelResources;
    VoiceChannelResourceList m_VoiceChannelResourceFreeList;
    nn::os::Mutex m_Mutex;
private:
    VoiceInfo* GetFreeVoiceInfo() NN_NOEXCEPT;
    void InsertToUsedList(VoiceInfo* entry) NN_NOEXCEPT;
    size_t UpdateVoiceChannelResourceInParameter(VoiceChannelResource::InParameter* pParameter) NN_NOEXCEPT;
    size_t UpdateVoiceInParameter(VoiceInfo::InParameter* pParameter) NN_NOEXCEPT;

public:
    VoiceInfoManager() NN_NOEXCEPT;
    static size_t GetWorkBufferSize(int voiceCount) NN_NOEXCEPT;
    void Initialize(int voiceCount, void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;
    VoiceInfo* AcquireVoiceInfo(int priority, int channelCount) NN_NOEXCEPT;
    void ReleaseVoiceInfo(VoiceInfo* entry) NN_NOEXCEPT;

    void UpdateVoiceRelatedParameters(void* pOutParameter, size_t* pOutInParamDataSize,  size_t* pOutChParamDataSize ) NN_NOEXCEPT;
    size_t UpdateVoiceOutStatus(VoiceInfo::OutStatus* pStatus) NN_NOEXCEPT;
    bool IsVoiceInfoValid(const VoiceInfo* pVoiceInfo) const NN_NOEXCEPT;
    void FlushVoiceDataCache(const VoiceInfo* entry) NN_NOEXCEPT;
};

}  // namespace audio
}  // namespace nn
