﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Cache.h>

#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_SinkTypes.h>
#include <nn/audio/audio_SinkApi.h>
#include <nn/audio/audio_FinalMixTypes.h>
#include <nn/audio/audio_FinalMixApi.h>

#include "audio_SinkManager.h"
#include "audio_ResourceExclusionChecker.h"

#define NN_AUDIO_DETAIL_SINKINFO_EXCLUSION_SCOPED_CHEKCER(pSinkInfo) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessCheckerForSinkInfo(detail::FindResourceExclusionCheckerFromRegionInConfig(pSinkInfo))

#define NN_AUDIO_DETAIL_SINK_EXCLUSION_SCOPED_CHEKCER(pSink) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessCheckerForSink(detail::FindResourceExclusionCheckerFromRegionInConfig(pSink->_handle))

namespace nn {
namespace audio {

NN_DEFINE_STATIC_CONSTANT(const int DeviceSinkType::DownMixParameter::CoeffCount);

Result AddDeviceSink(AudioRendererConfig* pConfig, DeviceSinkType* pOutSink, FinalMixType* pFinalMix, const int8_t* input, int inputCount, const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutSink);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_GREATER_EQUAL(GetFinalMixBufferCount(pFinalMix), inputCount);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_GREATER(inputCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_UNUSED(pFinalMix);

    SinkInfo* pInfo = pConfig->_pSinkManager->Allocate(common::SinkType_Device);
    NN_RESULT_THROW_UNLESS(pInfo != nullptr, ResultOutOfResource());
    NN_AUDIO_DETAIL_SINKINFO_EXCLUSION_SCOPED_CHEKCER(pInfo);

    pInfo->deviceInParam.inputCount = inputCount;
    std::memcpy(pInfo->deviceInParam.input, input, std::min(pInfo->InputCountMax, inputCount));
    memset(pInfo->deviceInParam.id, 0, sizeof(pInfo->deviceInParam.id)); // 終端の null を保証
    std::strncpy(pInfo->deviceInParam.id, name, sizeof(pInfo->deviceInParam.id) - 1);
    pOutSink->_handle = pInfo;
    pInfo->deviceInParam.useDownMixMatrix = false;
    memset(pInfo->deviceInParam.downMixMatrixCoeff, 0, sizeof(float) * common::DeviceParameter::CoeffCount);

    NN_RESULT_SUCCESS;
}

void RemoveDeviceSink(AudioRendererConfig* pConfig, DeviceSinkType* pSink, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES(pConfig->_pSinkManager->IsValidSinkInfo(pSink->_handle), "pSink is not initialized");
    NN_AUDIO_DETAIL_SINK_EXCLUSION_SCOPED_CHEKCER(pSink);
    NN_UNUSED(pFinalMix);

    pConfig->_pSinkManager->Free(pSink->_handle);
}

void SetDownMixParameter(DeviceSinkType* pSink, const DeviceSinkType::DownMixParameter* pParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES(pSink->_handle != nullptr, "pSink is not nn::audio::AddDeviceSink()");
    NN_SDK_REQUIRES(pSink->_handle->_type == common::SinkType_Device, "pSink is not nn::audio::AddDeviceSink()");
    NN_SDK_REQUIRES_NOT_NULL(pParameter);
    NN_AUDIO_DETAIL_SINK_EXCLUSION_SCOPED_CHEKCER(pSink);

    for (auto i = 0; i < common::DeviceParameter::CoeffCount; ++i)
    {
        // copy only first 4 parameter. rest 16 - 4 = 12  are not used.
        pSink->_handle->deviceInParam.downMixMatrixCoeff[i] = pParameter->coeff[i];
    }
}

void GetDownMixParameter(DeviceSinkType::DownMixParameter* pOutParameter, const DeviceSinkType* pSink) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES(pSink->_handle != nullptr, "pSink is not nn::audio::AddDeviceSink()");
    NN_SDK_REQUIRES(pSink->_handle->_type == common::SinkType_Device, "pSink is not nn::audio::AddDeviceSink()");
    NN_SDK_REQUIRES_NOT_NULL(pOutParameter);

    memset(pOutParameter, 0, sizeof(DeviceSinkType::DownMixParameter));
    for (auto i = 0; i < common::DeviceParameter::CoeffCount; ++i)
    {
        pOutParameter->coeff[i] = pSink->_handle->deviceInParam.downMixMatrixCoeff[i];
    }
}

void SetDownMixParameterEnabled(DeviceSinkType* pSink, bool enabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES(pSink->_handle != nullptr, "pSink is not nn::audio::AddDeviceSink()");
    NN_SDK_REQUIRES(pSink->_handle->_type == common::SinkType_Device, "pSink is not nn::audio::AddDeviceSink()");
    NN_AUDIO_DETAIL_SINK_EXCLUSION_SCOPED_CHEKCER(pSink);

    pSink->_handle->deviceInParam.useDownMixMatrix = enabled;
}

bool IsDownMixParameterEnabled(const DeviceSinkType* pSink) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES(pSink->_handle != nullptr, "pSink is not nn::audio::AddDeviceSink()");
    NN_SDK_REQUIRES(pSink->_handle->_type == common::SinkType_Device, "pSink is not nn::audio::AddDeviceSink()");

    return pSink->_handle->deviceInParam.useDownMixMatrix;
}

Result AddCircularBufferSink(AudioRendererConfig* pConfig, CircularBufferSinkType* pOutSink, FinalMixType* pFinalMix, const int8_t* input, int inputCount, void* buffer, size_t size, SampleFormat sampleFormat) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutSink);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_GREATER_EQUAL(GetFinalMixBufferCount(pFinalMix), inputCount);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(inputCount, 0);
    NN_UNUSED(pFinalMix);

    NN_SDK_REQUIRES_EQUAL(sampleFormat, SampleFormat_PcmInt16); // TODO: need to support other formats.
    NN_SDK_REQUIRES_ALIGNED(reinterpret_cast<uintptr_t>(buffer), nn::audio::BufferAlignSize);
    NN_SDK_ASSERT_NOT_NULL(pConfig->_pSinkManager);
    NN_SDK_REQUIRES(size % (inputCount * sizeof(int16_t) * pConfig->_pSinkManager->GetSampleCount()) == 0);

    memset(buffer, 0, size);
    nn::os::FlushDataCache(buffer, size);

    SinkInfo* pInfo = pConfig->_pSinkManager->Allocate(common::SinkType_CircularBuffer);
    NN_RESULT_THROW_UNLESS(pInfo != nullptr, ResultOutOfResource());
    NN_AUDIO_DETAIL_SINKINFO_EXCLUSION_SCOPED_CHEKCER(pInfo);

    pInfo->circularInParam._inputCount = inputCount;
    std::memcpy(pInfo->circularInParam._input, input, std::min(pInfo->InputCountMax, inputCount));

    pInfo->circularInParam._bufferAddress = reinterpret_cast<uintptr_t>(buffer);
    NN_SDK_ASSERT_LESS(size, static_cast<size_t>(std::numeric_limits<int32_t>::max()));
    pInfo->circularInParam._size = static_cast<int32_t>(size);
    pInfo->circularInParam._sampleFormat = sampleFormat;
    pInfo->circularInParam._sampleCount = pConfig->_pSinkManager->GetSampleCount();
    pInfo->circularOutStatus._currentPosition = 0;

    pOutSink->_handle = pInfo;

    NN_RESULT_SUCCESS;
}

void RemoveCircularBufferSink(AudioRendererConfig* pConfig, CircularBufferSinkType* pSink, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES(pConfig->_pSinkManager->IsValidSinkInfo(pSink->_handle));
    NN_AUDIO_DETAIL_SINK_EXCLUSION_SCOPED_CHEKCER(pSink);
    NN_UNUSED(pFinalMix);

    pConfig->_pSinkManager->Free(pSink->_handle);
}

size_t GetRequiredBufferSizeForCircularBufferSink(const AudioRendererParameter* pParameter, int inputCount, int frameCount, SampleFormat sampleFormat) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParameter);
    NN_SDK_REQUIRES_GREATER(inputCount, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(inputCount, nn::audio::MixBufferCountMax);
    NN_SDK_REQUIRES_GREATER(frameCount, 0);
    NN_SDK_REQUIRES_EQUAL(sampleFormat, nn::audio::SampleFormat_PcmInt16);
    return pParameter->sampleCount * inputCount * nn::audio::GetSampleByteSize(sampleFormat) * frameCount;
}

size_t ReadCircularBufferSink(CircularBufferSinkType* pSink, void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES_NOT_NULL(pSink->_handle);
    auto pInfo = pSink->_handle;
    NN_SDK_REQUIRES_EQUAL(pInfo->_type, common::SinkType_CircularBuffer);
    NN_AUDIO_DETAIL_SINK_EXCLUSION_SCOPED_CHEKCER(pSink);

    size_t circularBufferSize = pSink->_handle->circularInParam._size;
    size_t writePosition = pInfo->circularOutStatus._currentPosition;
    size_t readPosition = pInfo->circularInParam._previousPosition;

    size_t totalReadSize = std::min((readPosition > writePosition) ? (circularBufferSize + writePosition) - readPosition : writePosition - readPosition,
                                     bufferSize);

    // copy buffer
    auto base = reinterpret_cast<const int8_t*>(pSink->_handle->circularInParam._bufferAddress);
    auto dest = reinterpret_cast<int8_t*>(pOutBuffer);
    size_t curPos = readPosition;
    size_t left = totalReadSize;
    while (left)
    {
        size_t readSize = (curPos + left > circularBufferSize) ? (circularBufferSize - curPos) : left;
        auto src = base + curPos;
        // Using "flush" instead of "invalidate" because game applications cannot call nn::os::InvalidateDataCache()
        // In this use case flush works as "invalidate" because game applications never write to the src buffer
        os::FlushDataCache(src, readSize);
        memcpy(dest, src, readSize);

        left -= readSize;
        dest += readSize;
        curPos += readSize;
        curPos %= circularBufferSize;
    }
    pInfo->circularInParam._previousPosition = static_cast<int32_t>(curPos);

    return totalReadSize;
}

nn::audio::NodeId GetSinkNodeId(const DeviceSinkType* pSink) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES_NOT_NULL(pSink->_handle);
    return pSink->_handle->_nodeId;
}

nn::audio::NodeId GetSinkNodeId(const CircularBufferSinkType* pSink) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSink);
    NN_SDK_REQUIRES_NOT_NULL(pSink->_handle);
    return pSink->_handle->_nodeId;
}

}  // namespace audio
}  // namespace nn

