﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/audio_Result.h>
#include <nn/nn_Abort.h>
#include "audio_MixManager.h"
#include "common/audio_NodeIdManager.h"
#include "common/audio_SplitterParameters.h"

namespace nn {
namespace audio {

MixManager::MixManager(MixInfo* mixInfos, int mixInfoCount, int finalMixSampleRate, int mixBufferCount, int mixBufferSampleCount) NN_NOEXCEPT
    : m_MixInfos(mixInfos)
    , m_MixInfosCount(mixInfoCount)
    , m_MixBufferCount(mixBufferCount)
    , m_MixBufferSampleCount(mixBufferSampleCount)
    , m_UsedMixBufferCount(0)
    , m_Mutex(true)
{
    NN_SDK_ASSERT_NOT_NULL(mixInfos);
    NN_SDK_ASSERT(mixInfoCount > 0);
    NN_SDK_ASSERT(mixBufferCount > 0);

    for(auto i = 0; i < m_MixInfosCount; ++i)
    {
        m_MixInfos[i].isInUse = false;
    }

    m_MixInfos[MixInfo::FinalMixId].sampleRate = finalMixSampleRate;
}

bool MixManager::Acquire(FinalMixType* pFinalMix, int bufferCount) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    auto pFinalMixInfo = &m_MixInfos[MixInfo::FinalMixId];

    if(pFinalMixInfo->isInUse)
    {
        return false;
    }

    if(m_UsedMixBufferCount + bufferCount > m_MixBufferCount)
    {
        return false;
    }

    m_UsedMixBufferCount += bufferCount;
    pFinalMixInfo->isInUse = true;
    pFinalMixInfo->bufferCount = bufferCount;
    pFinalMixInfo->mixId = MixInfo::FinalMixId;
    pFinalMixInfo->nodeId = common::NodeIdManager::GetNodeId(common::NodeIdManager::NodeIdType::Mix, pFinalMixInfo->mixId, 0);
    pFinalMixInfo->destinationMixId = Invalid_MixId;
    pFinalMixInfo->splitterInfoId = common::InvalidSplitterInfoId;
    pFinalMix->_pMixInfo = pFinalMixInfo;
    return true;
}

void MixManager::Release(FinalMixType* pFinalMix) NN_NOEXCEPT
{
    // TODO: 紐づくエフェクトが解放済み Mix を触らないようにする対処が必要？
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    auto pFinalMixInfo = &m_MixInfos[MixInfo::FinalMixId];

    NN_ABORT_UNLESS(pFinalMix->_pMixInfo == pFinalMixInfo);
    NN_ABORT_UNLESS(m_UsedMixBufferCount >= 0);


    m_UsedMixBufferCount -= pFinalMixInfo->bufferCount;

    pFinalMixInfo->isInUse = false;
    pFinalMix->_pMixInfo = nullptr;
}

int MixManager::GetFinalMixSampleRate() const NN_NOEXCEPT
{
    return m_MixInfos[MixInfo::FinalMixId].sampleRate;
}

int MixManager::GetMixBufferSampleCount() const NN_NOEXCEPT
{
    return m_MixBufferSampleCount;
}

bool MixManager::Acquire(SubMixType* pSubMix, int bufferCount) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    MixInfo* p = nullptr;

    if(m_UsedMixBufferCount + bufferCount > m_MixBufferCount)
    {
        return false;
    }

    for (auto i = MixInfo::SubMixIdFirst; i < m_MixInfosCount; ++i)
    {
        if (!m_MixInfos[i].isInUse)
        {
            p = &m_MixInfos[i];
            p->isInUse = true;
            p->bufferCount = bufferCount;
            m_UsedMixBufferCount += bufferCount;
            p->mixId = i;
            p->nodeId = common::NodeIdManager::GetNodeId(common::NodeIdManager::NodeIdType::Mix, p->mixId, 0);
            p->destinationMixId = Invalid_MixId;
            p->splitterInfoId = common::InvalidSplitterInfoId;
            pSubMix->_pMixInfo = p;
            break;
        }
    }

    return p != nullptr;
}

void MixManager::Release(SubMixType* pSubMix) NN_NOEXCEPT
{
    // TODO: 紐づくエフェクトが解放済み Mix を触らないようにする対処が必要？
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    for (auto i = MixInfo::SubMixIdFirst; i < m_MixInfosCount; ++i)
    {
        if (&m_MixInfos[i] == pSubMix->_pMixInfo)
        {
            m_UsedMixBufferCount -= pSubMix->_pMixInfo->bufferCount;
            NN_ABORT_UNLESS(m_UsedMixBufferCount >= 0);
            m_MixInfos[i].isInUse = false;
            pSubMix->_pMixInfo = nullptr;
            return;
        }
    }
}

bool MixManager::IsValidMix(const SubMixType* pSubMix) const NN_NOEXCEPT
{
    // SubMix 用に確保したメモリ範囲内かチェックした後、ポインタとして適切なアドレスかを確認
    return (m_MixInfos + MixInfo::SubMixIdFirst) <= pSubMix->_pMixInfo
        && pSubMix->_pMixInfo <= (m_MixInfos + m_MixInfosCount)
        && (reinterpret_cast<uintptr_t>(pSubMix->_pMixInfo + MixInfo::SubMixIdFirst) - reinterpret_cast<uintptr_t>(m_MixInfos)) % sizeof(MixInfo) == 0;
}

bool MixManager::IsValidMix(const FinalMixType* pFinalMix) const NN_NOEXCEPT
{
    return (m_MixInfos + MixInfo::FinalMixId) == pFinalMix->_pMixInfo;
}

size_t MixManager::UpdateMixInParameter(void* pInParameter) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    MixInfo* pInMixInfo = reinterpret_cast<MixInfo*>(pInParameter);

    for (auto i = 0; i < m_MixInfosCount; ++i)
    {
        pInMixInfo[i] = m_MixInfos[i];
    }

    return m_MixInfosCount * sizeof(MixInfo);
}


}  // namespace audio
}  // namespace nn
