﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/audio_MemoryPool.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include "audio_MemoryPoolInfo.h"
#include "common/audio_BitArray.h"

#include <nn/os/os_Mutex.h>



namespace nn { namespace audio {

namespace common {
class WorkBufferAllocator;
}

class MemoryPoolManager
{
public:
    static size_t GetWorkBufferSize(const AudioRendererParameter& param) NN_NOEXCEPT;
    static MemoryPoolManager* CreateManager(common::WorkBufferAllocator& allocator, const AudioRendererParameter& param) NN_NOEXCEPT;

private:
    MemoryPoolManager(MemoryPoolInfo* pInfos, int count, void* activeFlagBuf, size_t bitArrayBufSize) NN_NOEXCEPT;

public:
    /**
     * @brief           Return # of MemoryPool which can be acquired.
     */
    int GetFreeCount() const NN_NOEXCEPT;

    /**
     * @brief           Acquire MemroyPool from MemoryPoolInfo array; m_pInfos.
     * @param[in]       address     Address for new memory pool
     * @param[in]       size        Buffer size of "address"
     * @return          pointer to MemoryPoolInfo in array. If MemoryPoolInfo array is full, return nullptr
     */
    MemoryPoolInfo* Acquire(void* address, size_t size) NN_NOEXCEPT;

    /**
     * @brief           Release a MemoryPool to MemoryPoolInfo array
     * @param[in]       MemoryPoolInfo * pPool
     */
    void Release(MemoryPoolInfo* pPool) NN_NOEXCEPT;

    /**
     * @brief           Check if "address" is in MemoryPools which is in m_pInfos;
     */
    bool IsOverlapped(void* address, size_t size) const NN_NOEXCEPT;

    size_t UpdateInParameter(void* pInParameter) NN_NOEXCEPT;
    size_t UpdateOutStatus(void* pOutOutStatus) NN_NOEXCEPT;

private:
    using InfoFlag = common::BitArray<uint32_t>;

    nn::os::Mutex m_Mutex;
    int m_PoolCountMax;
    int m_FreePoolCount;
    MemoryPoolInfo* m_pInfos;
    InfoFlag m_ActiveFlag;
};

}} // namespace nn::audio
