﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_FinalMixTypes.h>
#include "audio_MixManager.h"

#include "audio_ResourceExclusionChecker.h"
#define NN_AUDIO_DETAIL_FINALMIX_EXCLUSION_SCOPED_CHEKCER(pFinalMix) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessChecker(detail::FindResourceExclusionCheckerFromRegionInConfig(pFinalMix->_pMixInfo))

namespace nn {
namespace audio {

bool AcquireFinalMix(AudioRendererConfig* pConfig, FinalMixType* pOutFinalMix, int bufferCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutFinalMix);
    NN_SDK_REQUIRES_GREATER(bufferCount, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(bufferCount, MixBufferCountMax);
    auto pMixManager = pConfig->_pMixManager;

    if(!pMixManager->Acquire(pOutFinalMix, bufferCount))
    {
        return false;
    }

    NN_AUDIO_DETAIL_FINALMIX_EXCLUSION_SCOPED_CHEKCER(pOutFinalMix);

    pOutFinalMix->_pMixInfo->bufferCount = bufferCount;
    pOutFinalMix->_pMixInfo->sampleRate = pMixManager->GetFinalMixSampleRate();
    pOutFinalMix->_pMixInfo->volume = 1.0f;
    return true;
}

void ReleaseFinalMix(AudioRendererConfig* pConfig, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);
    NN_AUDIO_DETAIL_FINALMIX_EXCLUSION_SCOPED_CHEKCER(pFinalMix);
    auto pMixManager = pConfig->_pMixManager;
    NN_SDK_REQUIRES(pMixManager->IsValidMix(pFinalMix));

    pMixManager->Release(pFinalMix);
}

int GetFinalMixBufferCount(const FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);
    return pFinalMix->_pMixInfo->bufferCount;
}

float GetFinalMixVolume(const FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);
    return pFinalMix->_pMixInfo->volume;
}

void SetFinalMixVolume(FinalMixType* pFinalMix, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::FinalMixType::GetVolumeMin(), nn::audio::FinalMixType::GetVolumeMax());
    NN_AUDIO_DETAIL_FINALMIX_EXCLUSION_SCOPED_CHEKCER(pFinalMix);
    pFinalMix->_pMixInfo->volume = volume;
}

NodeId GetFinalMixNodeId(const FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);
    return pFinalMix->_pMixInfo->nodeId;
}

int GetFinalMixSampleRate(const FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);
    return pFinalMix->_pMixInfo->sampleRate;
}

}  // namespace audio
}  // namespace nn

