﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/audio/audio_Common.h>
#include <nn/audio/audio_AudioOutApi.h>
#include <nn/audio/audio_AudioOutApi.private.h>
#include <nn/audio/audio_AudioOutTypes.private.h>
#include <cstring>

#include "common/audio_Util.h"
namespace nn {
namespace audio {

int g_AudioOutCountMax = nn::audio::AudioOutCountMax;

NN_DEFINE_STATIC_CONSTANT( const size_t AudioOutBuffer::AddressAlignment );
NN_DEFINE_STATIC_CONSTANT( const size_t AudioOutBuffer::SizeGranularity );

template <>
void AudioOutBufferList::clear() NN_NOEXCEPT
{
    pHead = pTail = nullptr;
}

template <>
void AudioOutBufferList::push_back(AudioOutBuffer* p) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(p);
    NN_SDK_REQUIRES_NOT_NULL(p->buffer);
    if (!pHead)
    {
        NN_SDK_ASSERT(pTail == nullptr);
        pHead = pTail = p;
    }
    else
    {
        NN_SDK_ASSERT_NOT_NULL(pTail);
        pTail->pNext = p;
        pTail = p;
    }
    p->pNext = nullptr;
    p->offset = 0;
}

template <>
void AudioOutBufferList::pop_front() NN_NOEXCEPT
{
    AudioOutBuffer* p = pHead;
    if (p)
    {
        pHead = p->pNext;
        if (pHead == nullptr)
        {
            pTail = nullptr;
        }
    }
}

template <>
AudioOutBuffer* AudioOutBufferList::front() const NN_NOEXCEPT
{
    return pHead;
}

template <>
AudioOutBuffer* AudioOutBufferList::back() const NN_NOEXCEPT
{
    return pTail;
}

void InitializeAudioOutParameter(AudioOutParameter* pOutParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParameter);

    pOutParameter->sampleRate = 0;
    pOutParameter->channelCount = 0;
    pOutParameter->_reserved = common::AudioOutInitializedMagic;
}

void SetAudioOutBufferInfo(AudioOutBuffer* pOutAudioOutBuffer, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAudioOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    pOutAudioOutBuffer->_bufferU64 = 0;
    pOutAudioOutBuffer->_sizeU64 = 0;
#endif
    pOutAudioOutBuffer->buffer = buffer;
    pOutAudioOutBuffer->size = size;
}

void SetAudioOutBufferInfo(AudioOutBuffer* pOutAudioOutBuffer, void* buffer, size_t bufferSize, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAudioOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, AudioOutBuffer::AddressAlignment);
    NN_SDK_REQUIRES_ALIGNED(bufferSize, AudioOutBuffer::SizeGranularity);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_LESS_EQUAL(dataSize, bufferSize);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    pOutAudioOutBuffer->_bufferU64 = 0;
    pOutAudioOutBuffer->_bufferSizeU64 = 0;
    pOutAudioOutBuffer->_sizeU64 = 0;
#endif
    pOutAudioOutBuffer->buffer = buffer;
    pOutAudioOutBuffer->bufferSize = bufferSize;
    pOutAudioOutBuffer->size = dataSize;
}

void* GetAudioOutBufferDataPointer(const AudioOutBuffer* pAudioOutBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOutBuffer);
    return pAudioOutBuffer->buffer;
}

size_t GetAudioOutBufferDataSize(const AudioOutBuffer* pAudioOutBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOutBuffer);
    return pAudioOutBuffer->size;
}

size_t GetAudioOutBufferBufferSize(const AudioOutBuffer* pAudioOutBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOutBuffer);
    return pAudioOutBuffer->bufferSize;
}

void SetAudioOutCountMaxForLibraryApplet(int countMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(countMax, 0, nn::audio::AudioOutCountMaxForLibraryApplet);
    g_AudioOutCountMax = countMax;
}

int GetAudioOutCountMaxForLibraryApplet() NN_NOEXCEPT
{
    return g_AudioOutCountMax;
}

}  // namespace audio
}  // namespace nn

