﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include "common/audio_Util.h" // NN_AUDIO_INFOTYPE_FILED_ALIGN
#include "common/audio_IndexAllocator.h"

namespace nn { namespace audio {

class ExtraManager
{
public:
    static size_t GetRequiredMemorySize(size_t managedBufferSize) NN_NOEXCEPT
    {
        return Allocator::CalcBufferSize(managedBufferSize / BlockSize);
    }

    void Initialize(void* buffer, size_t size) NN_NOEXCEPT
    {
        const auto count = nn::util::align_down(size, BlockSize) / BlockSize;
        m_Allocator.Initialize(count, buffer, size);
    }

    bool AcquireBuffer(size_t* pOffset, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOffset);
        const auto count = static_cast<int>(nn::util::align_up(size, BlockSize) / BlockSize);
        uint16_t index;
        auto ret = m_Allocator.Alloc(&index, count);
        if (ret)
        {
            *pOffset = index * BlockSize;
        }
        return ret;
    }

    void Release(size_t offset, size_t size) NN_NOEXCEPT
    {
        const auto count = static_cast<int>(nn::util::align_up(size, BlockSize) / BlockSize);
        const auto index = static_cast<int>(offset / BlockSize);
        m_Allocator.Free(index, count);
    }

private:
    static const int BlockSize = 16 * 1024;

    using Allocator = common::IndexAllocator<uint16_t>;
    Allocator m_Allocator;
};

}} // namespace nn::audio
