﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/applet/applet_LibraryApplet.h>
#include <nn/applet/applet_LibraryAppletSelf.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/applet/applet_Result.h>

#include <mutex>
#include <utility>
#include <nn/am/am_Result.h>
#include <nn/am/am_Shim.h>
#include <nn/am/service/am_CaptureBufferIndex.h>
#include <nn/applet/applet_Storage.h>
#include <nn/applet/applet_Apis.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_ConditionVariable.h>
#include <nn/util/util_Optional.h>
#include "applet_StorageUtility.h"

namespace nn { namespace applet {

NN_STATIC_ASSERT(static_cast<int>(CaptureBufferIndex_LastApplication) == static_cast<int>(am::service::CaptureBufferIndex_LastApplication));
NN_STATIC_ASSERT(static_cast<int>(CaptureBufferIndex_LastForeground) == static_cast<int>(am::service::CaptureBufferIndex_LastForeground));
NN_STATIC_ASSERT(static_cast<int>(CaptureBufferIndex_CallerApplet) == static_cast<int>(am::service::CaptureBufferIndex_CallerApplet));

using detail::GetPointer;
using detail::GetShared;
using detail::PopStorage;

struct LibraryAppletHandleObject
{
    struct Meta
    {
        bool used;
        bool starting;
        uint64_t number;
    };
    Meta meta;
    am::service::ILibraryAppletAccessor* p;
    AppletId appletId;
    LibraryAppletMode libraryAppletMode;
    os::SystemEventType exitEvent;
    bool isInteractiveOutEventValid;
    os::SystemEventType interactiveOutEvent;
    bool isJoined;
    bool needsJoin;
    LibraryAppletExitReason exitReason;
    vi::IndirectConsumerHandleType indirectLayerConsumerHandle;
};

namespace {

    void (*g_OnLibraryAppletStarted)(LibraryAppletHandle handle);
    void (*g_OnLibraryAppletJoined)(LibraryAppletHandle handle);

    StartLibraryAppletHook g_StartLibraryAppletHook;
    bool g_IsCreatingSelfLibraryApplet;

    const int LibraryAppletCountMax = 8;

    class LaManager
    {
    private:

        os::Mutex m_Mutex{false};
        LibraryAppletHandleObject m_Objects[LibraryAppletCountMax];

        os::ConditionVariable m_Condition;
        uint64_t m_CurrentNumber{0};
        LibraryAppletHandleObject* m_pCurrent{nullptr};

        util::optional<os::SystemEvent> m_pLibraryAppletToBeLaunchable;

        static int GetPriority(const LibraryAppletHandleObject& x) NN_NOEXCEPT
        {
            switch (x.libraryAppletMode)
            {
                case applet::LibraryAppletMode_PartialForeground: return 1;
                case applet::LibraryAppletMode_PartialForegroundWithIndirectDisplay: return 1;
                case applet::LibraryAppletMode_NoUi: return 0;
                default: return 0;
            }
        }

        static bool IsPriorTo(const LibraryAppletHandleObject& x, const LibraryAppletHandleObject& y) NN_NOEXCEPT
        {
            auto px = GetPriority(x);
            auto py = GetPriority(y);
            return px < py;
        }

        static bool IsPreviousTo(const LibraryAppletHandleObject& x, const LibraryAppletHandleObject& y) NN_NOEXCEPT
        {
            auto px = GetPriority(x);
            auto py = GetPriority(y);
            if (px != py)
            {
                return px < py;
            }
            return x.meta.number < y.meta.number;
        }

        LibraryAppletHandleObject* SelectNextImpl() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
            auto p = static_cast<LibraryAppletHandleObject*>(nullptr);
            for (auto&& e : m_Objects)
            {
                if (e.meta.used && e.meta.starting)
                {
                    if (!p || IsPreviousTo(e, *p))
                    {
                        p = &e;
                    }
                }
            }
            return p;
        }

    public:

        decltype(m_Mutex)& GetMutex() NN_NOEXCEPT
        {
            return m_Mutex;
        }

        LibraryAppletHandleObject* Allocate() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            for (auto&& e : m_Objects)
            {
                if (!e.meta.used)
                {
                    e.meta.used = true;
                    e.meta.starting = false;
                    return &e;
                }
            }
            NN_ABORT("too many library applets(max %d)", LibraryAppletCountMax);
        }

        void Deallocate(LibraryAppletHandleObject* p) NN_NOEXCEPT
        {
            auto&& e = *p;
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            e.meta.used = false;
        }

        void LockToStart(LibraryAppletHandleObject* p) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            p->meta.starting = true;
            p->meta.number = ++this->m_CurrentNumber;
            while (!(m_pCurrent == nullptr && SelectNextImpl() == p))
            {
                if (m_pCurrent && IsPriorTo(*p, *m_pCurrent))
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pCurrent->p->RequestExit());
                }
                m_Condition.Wait(m_Mutex);
            }
            this->m_pCurrent = p;
            p->meta.starting = false;
        }

        bool TryLockToStart(LibraryAppletHandleObject* p) NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            p->meta.starting = true;
            p->meta.number = ++this->m_CurrentNumber;
            if (!(m_pCurrent == nullptr && SelectNextImpl() == p))
            {
                p->meta.starting = false;
                return false;
            }
            this->m_pCurrent = p;
            p->meta.starting = false;
            return true;
        }

        void OnStart(LibraryAppletHandleObject* p) NN_NOEXCEPT
        {
            if (g_OnLibraryAppletStarted)
            {
                g_OnLibraryAppletStarted({p});
            }
        }

        void UnlockToStart() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            this->m_pCurrent = nullptr;
            m_Condition.Broadcast();
        }

        void OnJoin(LibraryAppletHandleObject* p) NN_NOEXCEPT
        {
            if (g_OnLibraryAppletJoined)
            {
                g_OnLibraryAppletJoined({p});
            }
        }

        os::SystemEvent& GetLibraryAppletToBeLaunchable() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            if (!m_pLibraryAppletToBeLaunchable)
            {
                sf::NativeHandle sfHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->GetLibraryAppletLaunchableEvent(&sfHandle));
                m_pLibraryAppletToBeLaunchable.emplace();
                m_pLibraryAppletToBeLaunchable->AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), os::EventClearMode_ManualClear);
                sfHandle.Detach();
            }
            return *m_pLibraryAppletToBeLaunchable;
        }

    };

    LaManager g_LaManager;

    void WaitForLibraryAppletToBeLaunchable() NN_NOEXCEPT
    {
        if (g_IsCreatingSelfLibraryApplet)
        {
            return;
        }
        // LA を起動できる状態になるまで待機
        g_LaManager.GetLibraryAppletToBeLaunchable().Wait();
    }

    LibraryAppletHandleObject* CreateObjectDirectly(sf::SharedPointer<am::service::ILibraryAppletAccessor> pAccessor, AppletId appletId, LibraryAppletMode libraryAppletMode) NN_NOEXCEPT
    {
        auto&& e = *g_LaManager.Allocate();
        e.appletId = appletId;
        e.libraryAppletMode = libraryAppletMode;
        sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pAccessor->GetAppletStateChangedEvent(&eventHandle));
        os::AttachReadableHandleToSystemEvent(&e.exitEvent, eventHandle.GetOsHandle(), eventHandle.IsManaged(), os::EventClearMode_ManualClear);
        eventHandle.Detach();
        if (libraryAppletMode == LibraryAppletMode_PartialForegroundWithIndirectDisplay)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(pAccessor->GetIndirectLayerConsumerHandle(&e.indirectLayerConsumerHandle, applet::GetAppletResourceUserId()));
        }
        else
        {
            e.indirectLayerConsumerHandle = 0;
        }
        e.p = pAccessor.Detach();
        e.isInteractiveOutEventValid = false;
        e.isJoined = false;
        e.needsJoin = false;
        e.exitReason = LibraryAppletExitReason_Unexpected;
        return &e;
    }

    LibraryAppletHandleObject* CreateObject(AppletId appletId, LibraryAppletMode libraryAppletMode) NN_NOEXCEPT
    {
        WaitForLibraryAppletToBeLaunchable();
        sf::SharedPointer<am::service::ILibraryAppletAccessor> p;
        NN_RESULT_DO(am::GetLibraryAppletCreator()->CreateLibraryApplet(&p, appletId, libraryAppletMode));
        return CreateObjectDirectly(std::move(p), appletId, libraryAppletMode);
    }

    void DeleteObject(LibraryAppletHandleObject* p) NN_NOEXCEPT
    {
        auto&& e = *p;
        if (e.isInteractiveOutEventValid)
        {
            os::DestroySystemEvent(&e.interactiveOutEvent);
        }
        os::DestroySystemEvent(&e.exitEvent);
        sf::ReleaseSharedObject(e.p);
        e.p = nullptr;
        g_LaManager.Deallocate(p);
    }

    LibraryAppletHandleObject* GetObject(LibraryAppletHandle handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(handle != InvalidLibraryAppletHandle);
        auto ret = static_cast<LibraryAppletHandleObject*>(handle._p);
        NN_SDK_REQUIRES(ret->p, "specified LibraryAppletHandle is invalid");
        return ret;
    }

    am::service::ILibraryAppletAccessor* GetPointer(LibraryAppletHandle handle) NN_NOEXCEPT
    {
        return GetObject(handle)->p;
    }

} // anon

void SetOnAddLibraryApplet(void (*f)(LibraryAppletHandle handle)) NN_NOEXCEPT
{
    g_OnLibraryAppletStarted = f;
}

void SetOnRemoveLibraryApplet(void (*f)(LibraryAppletHandle handle)) NN_NOEXCEPT
{
    g_OnLibraryAppletJoined = f;
}

void SetCreatingSelfLibraryApplet(bool isCreatingSelfLibraryApplet) NN_NOEXCEPT
{
    g_IsCreatingSelfLibraryApplet = isCreatingSelfLibraryApplet;
}

Result CreateLibraryApplet(LibraryAppletHandle* pOut, AppletId appletId, LibraryAppletMode libraryAppletMode) NN_NOEXCEPT
{
    pOut->_p = CreateObject(appletId, libraryAppletMode);
    NN_RESULT_SUCCESS;
}

LibraryAppletHandle CreateLibraryAppletHandleOnUnwinding(sf::SharedPointer<am::service::ILibraryAppletAccessor> pAccessor) NN_NOEXCEPT
{
    am::service::LibraryAppletInfo libraryAppletInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pAccessor->GetLibraryAppletInfo(&libraryAppletInfo));
    auto p = CreateObjectDirectly(std::move(pAccessor), static_cast<AppletId>(libraryAppletInfo.appletId), static_cast<LibraryAppletMode>(libraryAppletInfo.mode));
    g_LaManager.LockToStart(p);
    g_LaManager.OnStart(p);
    p->needsJoin = true;
    return {p};
}

AppletId GetLibraryAppletId(LibraryAppletHandle handle) NN_NOEXCEPT
{
    return GetObject(handle)->appletId;
}

LibraryAppletMode GetLibraryAppletMode(LibraryAppletHandle handle) NN_NOEXCEPT
{
    return GetObject(handle)->libraryAppletMode;
}

void SetStartLibraryAppletHook(StartLibraryAppletHook hook) NN_NOEXCEPT
{
    g_StartLibraryAppletHook = hook;
}

Result StartLibraryApplet(LibraryAppletHandle handle, void* userArgument) NN_NOEXCEPT
{
    auto p = GetObject(handle);

    WaitForLibraryAppletToBeLaunchable();

    // もし起動に失敗する場合は、別途 Release() 処理が必要
    g_LaManager.LockToStart(p);
    g_LaManager.OnStart(p);
    p->needsJoin = true;

    if (g_StartLibraryAppletHook)
    {
        if (g_StartLibraryAppletHook(sf::SharedPointer<am::service::ILibraryAppletAccessor>(p->p, true), p->appletId, p->libraryAppletMode, userArgument))
        {
            NN_RESULT_SUCCESS;
        }
    }

    // 起動前にもう一度 FG チェック
    WaitForLibraryAppletToBeLaunchable();
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(handle)->Start());
    NN_RESULT_SUCCESS;
}

Result StartLibraryAppletImmediately(LibraryAppletHandle handle, void* userArgument) NN_NOEXCEPT
{
    auto p = GetObject(handle);

    NN_RESULT_THROW_UNLESS(g_LaManager.GetLibraryAppletToBeLaunchable().TryWait(), applet::ResultCannotStartImmediately());

    // もし起動に失敗する場合は、別途 Release() 処理が必要
    NN_RESULT_THROW_UNLESS(g_LaManager.TryLockToStart(p), applet::ResultCannotStartImmediately());
    g_LaManager.OnStart(p);
    p->needsJoin = true;

    if (g_StartLibraryAppletHook)
    {
        if (g_StartLibraryAppletHook(sf::SharedPointer<am::service::ILibraryAppletAccessor>(p->p, true), p->appletId, p->libraryAppletMode, userArgument))
        {
            NN_RESULT_SUCCESS;
        }
    }

    // 「起動前にもう一度 FG チェック」は省略
    // WaitForLibraryAppletToBeLaunchable();
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(handle)->Start());
    NN_RESULT_SUCCESS;
}

NN_NORETURN void JumpLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT
{
    auto p = GetObject(handle);

    WaitForLibraryAppletToBeLaunchable();

    // もし起動に失敗する場合は、別途 Release() 処理が必要
    g_LaManager.LockToStart(p);
    g_LaManager.OnStart(p);
    p->needsJoin = true;

    // 起動前にもう一度 FG チェック
    WaitForLibraryAppletToBeLaunchable();
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetProcessWindingController()->ReserveToStartAndWait(sf::SharedPointer<am::service::ILibraryAppletAccessor>(p->p, true)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetProcessWindingController()->WindAndDoReserved());
    for (;;)
    {
        os::SleepThread( TimeSpan::FromDays(1) );
    }
}

os::SystemEventType* GetLibraryAppletExitEvent(LibraryAppletHandle handle) NN_NOEXCEPT
{
    return &GetObject(handle)->exitEvent;
}

void RequestExitLibraryApplet(applet::LibraryAppletHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle != InvalidLibraryAppletHandle);
    am::service::IAppletAccessor* laAccessor = GetPointer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(laAccessor->RequestExit());
}

bool RequestExitLibraryAppletOrTerminate(am::service::IAppletAccessor* laAccessor, TimeSpan timeout) NN_NOEXCEPT
{
    if (!laAccessor)
    {
        return true;
    }
    sf::NativeHandle eventHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(laAccessor->GetAppletStateChangedEvent(&eventHandle));
    os::SystemEvent e;
    e.AttachReadableHandle(eventHandle.GetOsHandle(), false, os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(laAccessor->RequestExit());
    auto success = true;
    if (timeout >= 0)
    {
        success = e.TimedWait(timeout);
        if (!success)
        {
            laAccessor->Terminate();
        }
    }
    e.Wait();
    return success;
}

bool RequestExitLibraryAppletAndWaitOrTerminate(applet::LibraryAppletHandle handle, TimeSpan timeout) NN_NOEXCEPT
{
    return RequestExitLibraryAppletOrTerminate(GetPointer(handle), timeout);
}

void RequestExitLibraryAppletAndWait(applet::LibraryAppletHandle handle) NN_NOEXCEPT
{
    auto success = RequestExitLibraryAppletAndWaitOrTerminate(handle, TimeSpan::FromSeconds(-1));
    NN_UNUSED(success);
    NN_SDK_ASSERT(success);
}

Result JoinLibraryAppletImpl(LibraryAppletHandle handle) NN_NOEXCEPT
{
    os::WaitSystemEvent(&GetObject(handle)->exitEvent);
    GetObject(handle)->isJoined = true;
    auto exitReason = LibraryAppletExitReason_Normal;
    auto result = GetPointer(handle)->GetResult();
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(am::ResultAppletCancelled)
        {
            exitReason = LibraryAppletExitReason_Canceled;
        }
        NN_RESULT_CATCH(am::ResultAppletTerminated)
        {
            exitReason = LibraryAppletExitReason_Abnormal;
        }
        NN_RESULT_CATCH(am::ResultAppletExitedNotSuccess)
        {
            exitReason = LibraryAppletExitReason_Abnormal;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_SDK_LOG("nn::applet::JoinLibraryApplet() detects unexpected error: result %08x\n", result.GetInnerValueForDebug());
            exitReason = LibraryAppletExitReason_Unexpected;
        }
    NN_RESULT_END_TRY
    GetObject(handle)->exitReason = exitReason;
    NN_RESULT_SUCCESS;
}


LibraryAppletExitReason GetLibraryAppletExitReason(LibraryAppletHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(GetObject(handle)->isJoined, "specified LibraryAppletHandle is not joined");
    return GetObject(handle)->exitReason;
}

void JoinLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(JoinLibraryAppletImpl(handle));
    GetObject(handle)->needsJoin = false;
    g_LaManager.OnJoin(GetObject(handle));
    g_LaManager.UnlockToStart();
}

void CloseLibraryApplet(LibraryAppletHandle handle) NN_NOEXCEPT
{
    if (GetObject(handle)->needsJoin)
    {
        NN_SDK_ASSERT(GetObject(handle)->isJoined);
    }
    DeleteObject(GetObject(handle));
}

void PushToInChannel(LibraryAppletHandle handle, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(handle)->PushInData(GetShared(storageHandle)));
    ReleaseStorage(storageHandle);
}

void PushToInteractiveInChannel(LibraryAppletHandle handle, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(handle)->PushInteractiveInData(GetShared(storageHandle)));
    ReleaseStorage(storageHandle);
}

bool TryPopFromOutChannel(StorageHandle* pOut, LibraryAppletHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(GetObject(handle)->isJoined, "specified LibraryAppletHandle is not joined");
    NN_SDK_ASSERT(GetObject(handle)->exitReason == LibraryAppletExitReason_Normal, "specified LibraryAppletHandle is not exited successfully");
    return PopStorage(pOut, [&] (sf::SharedPointer<am::service::IStorage>* pOut) -> Result
    {
        return GetPointer(handle)->PopOutData(pOut);
    });
}

bool TryPopFromInteractiveOutChannel(StorageHandle* pOut, LibraryAppletHandle handle) NN_NOEXCEPT
{
    return PopStorage(pOut, [&] (sf::SharedPointer<am::service::IStorage>* pOut) -> Result
    {
        return GetPointer(handle)->PopInteractiveOutData(pOut);
    });
}

os::SystemEventType* GetPopFromInteractiveOutChannelEvent(LibraryAppletHandle handle) NN_NOEXCEPT
{
    auto&& e = *GetObject(handle);
    std::lock_guard<decltype(g_LaManager.GetMutex())> lk(g_LaManager.GetMutex());
    if (!e.isInteractiveOutEventValid)
    {
        sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(handle)->GetPopInteractiveOutDataEvent(&eventHandle));
        os::AttachReadableHandleToSystemEvent(&e.interactiveOutEvent, eventHandle.GetOsHandle(), eventHandle.IsManaged(), os::EventClearMode_ManualClear);
        eventHandle.Detach();
        e.isInteractiveOutEventValid = true;
    }
    return &e.interactiveOutEvent;
}

bool WaitPopFromInteractiveOutChannelEvent(LibraryAppletHandle handle) NN_NOEXCEPT
{
    return os::WaitAny(GetPopFromInteractiveOutChannelEvent(handle), GetLibraryAppletExitEvent(handle)) == 0;
}

Result TakeScreenShotOfCallerApplet() NN_NOEXCEPT
{
    return am::GetDisplayController()->UpdateCallerAppletCaptureImage();
}

Result TakeScreenShotOfOwnLayerIntoLastApplicationBuffer(bool isScreenShotPermitted) NN_NOEXCEPT
{
    return am::GetDisplayController()->TakeScreenShotOfOwnLayer(CaptureBufferIndex_LastApplication, isScreenShotPermitted);
}

Result TakeScreenShotOfOwnLayerIntoLastForegroundBuffer(bool isScreenShotPermitted) NN_NOEXCEPT
{
    return am::GetDisplayController()->TakeScreenShotOfOwnLayer(CaptureBufferIndex_LastForeground, isScreenShotPermitted);
}

Result TakeScreenShotOfOwnLayerIntoCallerAppletBuffer(bool isScreenShotPermitted) NN_NOEXCEPT
{
    return am::GetDisplayController()->TakeScreenShotOfOwnLayer(CaptureBufferIndex_CallerApplet, isScreenShotPermitted);
}

Result TakeScreenShotOfOwnLayerIntoCallerAppletBufferImmediately(bool isScreenShotPermitted) NN_NOEXCEPT
{
    bool isImmediate = true;
    return am::GetDisplayController()->TakeScreenShotOfOwnLayerEx(CaptureBufferIndex_CallerApplet, isScreenShotPermitted, isImmediate);
}

vi::IndirectConsumerHandleType GetIndirectLayerConsumerHandle(LibraryAppletHandle handle) NN_NOEXCEPT
{
    auto&& e = *GetObject(handle);
    return e.indirectLayerConsumerHandle;
}

Result SetOutOfFocusApplicationSuspendingEnabled(LibraryAppletHandle handle, bool isEnabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(am::IsInitializedAsApplication(), "Caller is not application.");
    return GetPointer(handle)->SetOutOfFocusApplicationSuspendingEnabled(isEnabled);
}

// キーボード配列取得
bool GetDesirableKeyboardLayout(Bit32* pOutLayout) NN_NOEXCEPT
{
    return am::GetLibraryAppletSelfAccessor()->GetDesirableKeyboardLayout(pOutLayout).IsSuccess();
}

}}
