﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/apm/detail/apm_Log.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>

#include "apm_SettingsHolder-spec.nx.h"

namespace nn { namespace apm { namespace server {

SettingsHolder::SettingsHolder() NN_NOEXCEPT
    : m_PerformanceModePolicy(PerformanceModePolicy::Auto)
    , m_SdevThrottlingEnabled(true)
    , m_SdevThrottlingAdditionalDelayUs(DefaultSdevThrottlingAdditionalDelayUs)
    , m_BatteryDrainingEnabled(false)
    , m_SdevCpuOverclockEnabled(false)
    , m_ThrottlingForUndockEnabled(false)
    , m_ThrottlingForSmpdEnabled(false)
    , m_BoardTypeIsSdev(false)
    , m_FirmwareDebugSettingsAccessor()
{
    // 何も処理をしない。
}

void SettingsHolder::CheckBoardType() NN_NOEXCEPT
{
    settings::factory::ConfigurationId1 configurationId1;
    settings::factory::GetConfigurationId1(&configurationId1);

    NN_DETAIL_APM_TRACE("BoardType: %s\n", configurationId1.string);

    if ( nn::util::string_view(configurationId1.string).substr(0, 4) == "SDEV" )
    {
        m_BoardTypeIsSdev = true;
    }
    else
    {
        m_BoardTypeIsSdev = false;
    }
}

void SettingsHolder::ReloadSettings() NN_NOEXCEPT
{
    const size_t PerformanceModePolicyBufferSize = 32;
    char performanceModePolicy[PerformanceModePolicyBufferSize] = "\0";

    m_FirmwareDebugSettingsAccessor.ReadSettingString(performanceModePolicy, PerformanceModePolicyBufferSize, "performance_mode_policy");
    if ( nn::util::Strncmp<char>(performanceModePolicy, "normal", PerformanceModePolicyBufferSize) == 0 )
    {
        m_PerformanceModePolicy = PerformanceModePolicy::Normal;
    }
    else if ( nn::util::Strncmp<char>(performanceModePolicy, "boost", PerformanceModePolicyBufferSize) == 0 )
    {
        m_PerformanceModePolicy = PerformanceModePolicy::Boost;
    }
    else
    {
        m_PerformanceModePolicy = PerformanceModePolicy::Auto;
    }
    NN_DETAIL_APM_TRACE("performance_mode_policy            : %s(%d)\n", performanceModePolicy, m_PerformanceModePolicy);

    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_SdevCpuOverclockEnabled, "sdev_cpu_overclock_enabled");
    NN_DETAIL_APM_TRACE("sdev_cpu_overclock_enabled         : %s\n", m_SdevCpuOverclockEnabled ? "true" : "false");
}

void SettingsHolder::LoadSettings() NN_NOEXCEPT
{
    CheckBoardType();

    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_SdevThrottlingEnabled, "sdev_throttling_enabled");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_SdevThrottlingAdditionalDelayUs, "sdev_throttling_additional_delay_us");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_BatteryDrainingEnabled, "battery_draining_enabled");

    NN_DETAIL_APM_TRACE("sdev_throttling_enabled            : %s\n", m_SdevThrottlingEnabled ? "true" : "false");
    NN_DETAIL_APM_TRACE("sdev_throttling_additional_delay_us: %d\n", m_SdevThrottlingAdditionalDelayUs);
    NN_DETAIL_APM_TRACE("battery_draining_enabled           : %s\n", m_BatteryDrainingEnabled ? "true" : "false");

    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_ThrottlingForUndockEnabled, "throttling_for_undock_enabled");
    m_FirmwareDebugSettingsAccessor.ReadSetting(&m_ThrottlingForSmpdEnabled, "throttling_for_smpd_enabled");

    NN_DETAIL_APM_TRACE("throttling_for_undock_enabled      : %s\n", m_ThrottlingForUndockEnabled ? "true" : "false");
    NN_DETAIL_APM_TRACE("throttling_for_smpd_enabled        : %s\n", m_ThrottlingForSmpdEnabled ? "true" : "false");

    ReloadSettings();
}

SettingsHolder::PerformanceModePolicy SettingsHolder::GetPerformanceModePolicy() NN_NOEXCEPT
{
    auto performanceModePolicy = m_PerformanceModePolicy;

    if ( performanceModePolicy == SettingsHolder::PerformanceModePolicy::Boost
        && !m_BoardTypeIsSdev )
    {
        performanceModePolicy = SettingsHolder::PerformanceModePolicy::Auto;
    }

    return performanceModePolicy;
}

bool SettingsHolder::IsSdevThrottlingEnabled() NN_NOEXCEPT
{
    return m_SdevThrottlingEnabled && m_BoardTypeIsSdev;
}

int SettingsHolder::GetSdevThrottlingAdditionalDelayUs() NN_NOEXCEPT
{
    return m_SdevThrottlingAdditionalDelayUs;
}

bool SettingsHolder::IsBatteryDrainingEnabled() NN_NOEXCEPT
{
    return m_BatteryDrainingEnabled;
}

bool SettingsHolder::IsSdevCpuOverclockEnabled() NN_NOEXCEPT
{
    return m_SdevCpuOverclockEnabled && m_BoardTypeIsSdev;
}

bool SettingsHolder::IsThrottlingForUndockEnabled() NN_NOEXCEPT
{
    return m_ThrottlingForUndockEnabled;
}

bool SettingsHolder::IsThrottlingForSmpdEnabled() NN_NOEXCEPT
{
    return m_ThrottlingForSmpdEnabled;
}

bool SettingsHolder::IsEnoughPowerChargeEmulationRequired() NN_NOEXCEPT
{
    return (GetPerformanceModePolicy() == PerformanceModePolicy::Boost)
        || IsSdevCpuOverclockEnabled();
}

}}} // namespace nn::apm::server
