﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/apm/server/apm.h>
#include <nn/apm/apm_ManagerImpl.h>
#include <nn/apm/apm_SystemManagerImpl.h>

#include "apm_PerformanceServer.h"
#include "apm_SessionImpl.h"

namespace nn { namespace apm {

ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化。
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);

    // アロケータにヒープハンドルをアタッチ。
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::apm::ISession>> pOutSession) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;

    auto p = Factory::CreateSharedEmplaced<ISession, server::SessionImpl>(&m_Allocator, this);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *pOutSession = std::move(p);

    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::GetPerformanceMode(nn::sf::Out<nn::apm::PerformanceMode> pOutPerformanceMode) NN_NOEXCEPT
{
    *pOutPerformanceMode = server::GetPerformanceMode();

    NN_RESULT_SUCCESS;
}

nn::Result SystemManagerImpl::RequestPerformanceMode(nn::apm::PerformanceMode performanceMode) NN_NOEXCEPT
{
    server::RequestPerformanceMode(performanceMode);
    NN_RESULT_SUCCESS;
}

nn::Result SystemManagerImpl::GetPerformanceEvent(nn::sf::Out<nn::sf::NativeHandle> pOutNativeHandle, nn::apm::EventTarget eventTarget) NN_NOEXCEPT
{
    nn::os::SystemEventType* pEvent;

    server::GetEventPtr(&pEvent, eventTarget);

    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(pEvent);

    *pOutNativeHandle = nn::sf::NativeHandle(internalHandle, false);

    NN_RESULT_SUCCESS;
}

nn::Result SystemManagerImpl::GetThrottlingState(nn::sf::Out<nn::apm::ThrottlingState> pOutThrottlingState) NN_NOEXCEPT
{
    ThrottlingState throttlingState;
    server::GetThrottlingState(&throttlingState);
    *pOutThrottlingState = throttlingState;
    NN_RESULT_SUCCESS;
}

nn::Result SystemManagerImpl::GetLastThrottlingState(nn::sf::Out<nn::apm::ThrottlingState> pOutThrottlingState) NN_NOEXCEPT
{
    ThrottlingState throttlingState;
    server::GetLastThrottlingState(&throttlingState);
    *pOutThrottlingState = throttlingState;
    NN_RESULT_SUCCESS;
}

nn::Result SystemManagerImpl::ClearLastThrottlingState() NN_NOEXCEPT
{
    server::ClearLastThrottlingState();
    NN_RESULT_SUCCESS;
}

nn::Result SystemManagerImpl::LoadAndApplySettings() NN_NOEXCEPT
{
    server::LoadAndApplySettings();
    NN_RESULT_SUCCESS;
}

ManagerPrivilegedImpl::ManagerPrivilegedImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化。
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);

    // アロケータにヒープハンドルをアタッチ。
    m_Allocator.Attach(m_HeapHandle);
}

ManagerPrivilegedImpl::~ManagerPrivilegedImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerPrivilegedImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::apm::ISession>> pOutSession) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;

    auto p = Factory::CreateSharedEmplaced<ISession, server::SessionPrivilegedImpl>(&m_Allocator, this);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *pOutSession = std::move(p);

    NN_RESULT_SUCCESS;
}

}} // namespace nn::apm
