﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nn;
using nn.sf;
using std;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;

namespace nn.am
{
    /// <summary>
    /// アプレットメッセージ
    /// </summary>
    /// <remarks>
    /// サイズ等は別途検討
    /// </remarks>
    [CppRefPath(Location.CommonTypesPath)]
    [ExternalStruct(4, 4)]
    public struct AppletMessage
    {
    }
}

namespace nn.am.service
{
    /// <summary>
    /// アプレットの種類
    /// </summary>
    /// <remarks>
    /// 必要性や詳細未定
    ///
    /// 必要かもしれない場面
    /// - LA Shim でスクラッチにするかどうかを動的に判定する必要がある場合
    ///   - 少なくとも MA 上でスクラッチにはできない
    /// </remarks>
    [CppRefPath(Location.FunctionsSfdlPath)]
    [ExternalStruct(8, 4)]
    public struct AppletKind
    {
    }

    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface ICommonStateGetter : IServiceObject
    {
        /// <summary>
        /// メッセージ通知検知用のイベントを取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// 各アプレットへの FG/BG 遷移要求などのメッセージ通知を
        /// 受信するための SystemEvent を受け取ります。
        /// </remarks>
        [MethodId(0)]
        Result GetEventHandle(Out<NativeHandle> pOut);

        /// <summary>
        /// メッセージ通知を受信する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// 各アプレットへの通知メッセージを受信します。
        /// メッセージが全て受信されると上記 SystemEvent がクリアされます。
        /// 通知メッセージのキューイング段数は実装依存です。
        /// </remarks>
        [MethodId(1)]
        Result ReceiveMessage(Out<AppletMessage> pOut);

        /// <summary>
        /// アプレット種別を取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(2)]
        Result GetThisAppletKind(Out<AppletKind> pOut);

        #region 暫定: スリープ要求

        /// <summary>
        /// スリープ要求を許可する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// TODO: スリープ関連（近いうちに I/F が変わる予定）
        ///
        /// 自アプレットへの Message_ChangeIntoSleep 通知に対して、
        /// スリープ許可を返答するためのメソッドです。
        /// </remarks>
        [MethodId(3)]
        Result AllowToEnterSleep();

        /// <summary>
        /// スリープ要求を却下する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// TODO: スリープ関連（近いうちに I/F が変わる予定）
        ///
        /// 自アプレットへの Message_ChangeIntoSleep 通知に対して、
        /// スリープ拒否を返答するためのメソッドです。
        /// </remarks>
        [MethodId(4)]
        Result DisallowToEnterSleep();

        /// <summary>
        /// 現在の動作モードを取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(5)]
        Result GetOperationMode(Out<Bit8> pOut);

        /// <summary>
        /// 現在の性能モードを取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(6)]
        Result GetPerformanceMode(Out<Bit32> pOut);

        /// <summary>
        /// 現在のクレードル状態を取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(7)]
        Result GetCradleStatus(Out<Bit8> pOut);

        [MethodId(8)]
        Result GetBootMode(Out<Bit8> pOut);

        /// <summary>
        /// 現在のフォーカス状態を取得
        /// </summary>
        /// <param name="pOutFocusState">フォーカス状態の格納先</param>
        /// <remarks>
        /// </remarks>
        [MethodId(9)]
        Result GetCurrentFocusState(Out<Bit8> pOutFocusState);

        #endregion

        #region スリープ遅延 API

        /// <summary>
        /// スリープロックをリクエストする。(非同期)
        /// </summary>
        /// <remarks>
        /// このメソッドを呼んだだけでは、スリープロックは実際には取得されない。
        /// 実際のロック取得は、スリープシーケンスの実行中などにおいて遅延される可能性があり、
        /// GetAcquiredSleepLockEvent で取得できるイベントにて取得が可能である。
        ///
        /// スリープロックが取得されている間はスリープが遅延される。
        /// スリープを間に挟んではならない動作や、スリープ前の準備動作は、
        /// GetAcquiredSleepLockEvent で取得されるイベントのシグナルから ReleaseSleepLock(Transiently) の呼び出しまでの間に行う必要がある。
        ///
        /// スリープロック取得中にスリープシーケンスが実行された場合には Message_RequestToReleaseSleepLock が通知される。
        /// この際、ユーザは、必要な処理を行った後に速やかに ReleaseSleepLock(Transiently) を呼んでロックを解除する必要がある。
        ///
        /// 以下のようなシナリオが想定されている。
        /// - プログラムの初期化時に RequestToAcquireSleepLock を呼んだ後、イベントを待機する
        /// - Message_RequestToReleaseSleepLock が通知されたら必要な処理を行った後 ReleaseSleepLockTransiently を呼び出す
        /// - 再度イベントを待機する
        /// - Message_RequestToReleaseSleepLock が通知されたら必要な処理を行った後 ReleaseSleepLockTransiently を呼び出す
        /// - ...
        /// </remarks>
        /// <returns></returns>
        [MethodId(10)]
        Result RequestToAcquireSleepLock();

        /// <summary>
        /// スリープロックを解除する。
        /// </summary>
        /// <remarks>
        /// 通常はあまり使用されることを想定していない。
        /// 多くの場合は ReleaseSleepLockTransiently を使用する。
        /// </remarks>
        /// <returns></returns>
        [MethodId(11)]
        Result ReleaseSleepLock();

        /// <summary>
        /// スリープロックを一時的に解除し、スリープを行ったのを待機した後、スリープロックを再度自動で取得する。(非同期)
        /// </summary>
        /// <remarks>
        /// 再度のスリープロックの取得は、スリープ後に自動で行われ、ロックの取得完了は、Message_AcquiredSleepLock によって通知される。
        /// </remarks>
        /// <returns></returns>
        [MethodId(12)]
        Result ReleaseSleepLockTransiently();

        /// <summary>
        /// スリープロックが取得されているときにシグナルされているイベントを取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        [MethodId(13)]
        Result GetAcquiredSleepLockEvent(Out<sf.NativeHandle> pOut);

        #endregion

        [MethodId(20)]
        Result PushToGeneralChannel(IStorage storage);

        /// <summary>
        ///
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        /// <remarks>TODO: 後でシステム用の場所に移す</remarks>
        [MethodId(30)]
        Result GetHomeButtonReaderLockAccessor(Out<ILockAccessor> pOut);

        [MethodId(31)]
        Result GetReaderLockAccessorEx(Out<ILockAccessor> pOut, int index);

        [MethodId(40)]
        Result GetCradleFwVersion(Out<uint32_t> pOutPdcH, Out<uint32_t> pOutPdcA, Out<uint32_t> pOutMcu, Out<uint32_t> pOutDp2Hdmi);

        /// <summary>
        /// 現在、VR モードが有効か否かを返します。
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(50)]
        Result IsVrModeEnabled(Out<bool> pOut);

        /// <summary>
        /// 現在、VR モードが有効か否かを設定します。
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// VR モードはグローバルに一つだけあるフラグであり、
        /// どのアプレットから呼ばれてもそのフラグを変更します。
        /// </remarks>
        [MethodId(51)]
        Result SetVrModeEnabled(bool isEnabled);

        /// <summary>
        /// LCD バックライト OFF 要求を設定します。
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// </remarks>
        [MethodId(52)]
        Result SetLcdBacklighOffEnabled(bool isLcdBacklightOff);

        /// <summary>
        /// コントローラ F/W 更新期間中か否かを返す
        /// </summary>
        /// <returns>
        /// </returns>
        [MethodId(55)]
        Result IsInControllerFirmwareUpdateSection(Out<bool> pOut);

        [MethodId(60)]
        Result GetDefaultDisplayResolution(Out<int32_t> pOutWidth, Out<int32_t> pOutHeight);

        [MethodId(61)]
        Result GetDefaultDisplayResolutionChangeEvent(Out<NativeHandle> outValue);

        [MethodId(62)]
        Result GetHdcpAuthenticationState(Out<int32_t> outValue);

        [MethodId(63)]
        Result GetHdcpAuthenticationStateChangeEvent(Out<NativeHandle> outValue);

        /// <summary>
        /// アプリからのテレビ電源連動モード変更
        /// </summary>
        /// <param name="mode">テレビとの電源連動モード</param>
        /// <remarks>
        /// CROSS2 専用、アプリがインフォーカス状態時のみ効果あり
        /// </remarks>
        [MethodId(64)]
        Result SetTvPowerStateMatchingMode(int mode);

        [MethodId(65)]
        Result GetApplicationIdByContentActionName(Out<Bit64> pOut, [BufferTransferMode(BufferTransferMode.MapAlias)] InArray<char_t> name);

    }
}
