﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_SystemSharedLowLayer.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>

namespace nn{ namespace am{ namespace service{ namespace display{
    namespace detail{

        //---------------------------------

        SystemSharedLowLayer::SystemSharedLowLayer() NN_NOEXCEPT
            : m_LayerId(0)
            , m_SynchronizedEvent()
            , m_BufferHandle()
        {
        }

        bool SystemSharedLowLayer::IsInitialized() const NN_NOEXCEPT
        {
            return m_LayerId != 0;
        }

        nn::Result SystemSharedLowLayer::Initialize(nn::vi::fbshare::SharedBufferHandle hBuffer, bool isOpaque, uint32_t zOrder) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(!IsInitialized());
            NN_AM_DISPCTRL_LOG_TRACE("initializing system-shared-low-layer\n");

            nn::vi::LayerSettings settings = {};
            settings.Set<vi::LayerFlags::Fullscreen>(1);
            settings.Set<vi::LayerFlags::Opaque>(isOpaque);
            NN_AM_DISPCTRL_LOG_TRACE("  creating system-shared-layer\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateDisplayLayer(&m_LayerId, settings, {0}));
            NN_AM_DISPCTRL_LOG_TRACE("  binding system-shared-layer\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::BindSharedLowLevelLayerToDisplayLayer(m_LayerId, {0}));
            NN_AM_DISPCTRL_LOG_TRACE("  creating system-shared-layer synchronized event\n");
            DisplayPrimitiveOperation::InitializeSharedLowLevelLayerSynchronizedEvent(&m_SynchronizedEvent, m_LayerId);
            NN_AM_DISPCTRL_LOG_TRACE("  connecting system-shared-layer\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::ConnectSharedLowLevelLayerToDisplayLayer(m_LayerId, hBuffer));

            NN_AM_DISPCTRL_LOG_TRACE("  setting system-shared-layer z-order\n");
            DisplayPrimitiveOperation::SetDisplayLayerZOrder(m_LayerId, zOrder);

            m_BufferHandle = hBuffer;
            NN_RESULT_SUCCESS;
        }

        void SystemSharedLowLayer::Finalize() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(IsInitialized());

            DisplayPrimitiveOperation::DisconnectSharedLowLevelLayerFromDisplayLayer(m_LayerId);
            DisplayPrimitiveOperation::UnbindSharedLowLevelLayerFromDisplayLayer(m_LayerId);
            DisplayPrimitiveOperation::FinalizeSharedLowLevelLayerSynchronizedEvent(&m_SynchronizedEvent, m_LayerId);
            DisplayPrimitiveOperation::DestroyDisplayLayer(m_LayerId);

            m_LayerId = {};
            m_SynchronizedEvent = {};
            m_BufferHandle = {};
        }

        nn::vi::LayerId SystemSharedLowLayer::GetLayerId() const NN_NOEXCEPT
        {
            return m_LayerId;
        }

        nn::os::SystemEventType* SystemSharedLowLayer::GetSynchronizedEvent() NN_NOEXCEPT
        {
            return &m_SynchronizedEvent;
        }

    }   // namespace detail
}}}}
