﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_MyDisplayManagerService.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    MyDisplayManagerService::MyDisplayManagerService() NN_NOEXCEPT
    {
        vi::ProxyName proxyName = { "dispmgr" };
        this->InitializeMinimum(nn::vi::PolicyLevel_Composition, &proxyName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(this->GetApplicationDisplayService()->OpenDefaultDisplay(&m_DisplayId));
        std::memset(m_DisplayTable, 0, sizeof(m_DisplayTable));
    }

    MyDisplayManagerService::~MyDisplayManagerService() NN_NOEXCEPT
    {
        this->GetApplicationDisplayService()->CloseDisplay(m_DisplayId);
        this->FinalizeMinimum();
    }

    vi::DisplayId MyDisplayManagerService::GetDisplayId() NN_NOEXCEPT
    {
        return m_DisplayId;
    }
    uint64_t MyDisplayManagerService::MakeName64Value(const char* name) NN_NOEXCEPT
    {
        uint64_t name64 = 0;
        std::strncpy(reinterpret_cast<char*>(&name64), name, sizeof(name64));
        return name64;
    }

    bool MyDisplayManagerService::OpenByName(vi::DisplayId* pOutDisplayId, const char* name) NN_NOEXCEPT
    {
        const uint64_t name64 = MakeName64Value(name);

        std::lock_guard<os::Mutex> lk(m_DisplayTableMutex);
        for(auto& itr : m_DisplayTable)
        {
            if (itr.m_Name64 == name64)
            {
                *pOutDisplayId = itr.m_DisplayId;
                ++itr.m_OpenCount;
                return true;
            }
        }
        return false;
    }

    void MyDisplayManagerService::RegisterDisplayIdAndName(const char* name, vi::DisplayId id) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lk(m_DisplayTableMutex);
        for(auto& itr : m_DisplayTable)
        {
            if (itr.m_DisplayId == 0)
            {
                itr.m_Name64 = MakeName64Value(name);
                itr.m_DisplayId = id;
                itr.m_OpenCount = 1;
                return;
            }
        }
        NN_ABORT("AM: Cannot register displayId.");
    }

    uint64_t MyDisplayManagerService::CloseByDisplayId(vi::DisplayId id) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lk(m_DisplayTableMutex);

        uint64_t count = 0;
        for(auto& itr : m_DisplayTable)
        {
            if (itr.m_DisplayId == id)
            {
                count = itr.m_OpenCount;
                itr.m_OpenCount = count - 1;
                if (count == 1)
                {
                    itr.m_Name64 = 0;
                    itr.m_DisplayId = 0;
                }
                break;
            }
        }
        return count;
    }

    Result MyDisplayManagerService::OpenDisplayWrapper(vi::DisplayId* pOutDisplayId, const char* displayName) NN_NOEXCEPT
    {
        if (!OpenByName(pOutDisplayId, displayName))
        {
            vi::DisplayName name = {};
            util::Strlcpy(name.value, displayName, static_cast<int>(sizeof(name)));
            NN_RESULT_DO(GetApplicationDisplayService()->OpenDisplay(pOutDisplayId, name));
            RegisterDisplayIdAndName(displayName, *pOutDisplayId);
        }
        NN_RESULT_SUCCESS;
    }

    Result MyDisplayManagerService::CloseDisplayWrapper(vi::DisplayId displayId) NN_NOEXCEPT
    {
        auto count = CloseByDisplayId(displayId);
        NN_ABORT_UNLESS(count > 0);
        if (count == 1)
        {
            NN_RESULT_DO(GetApplicationDisplayService()->CloseDisplay(displayId));
        }
        NN_RESULT_SUCCESS;
    }

}}}}

