﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_DirectLayerVisibilityManager.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>
#include <nn/am/service/display/am_DisplayControlLog.h>

namespace nn{ namespace am{ namespace service{ namespace display{ namespace detail{

    namespace {

        nn::vi::LayerStackSet GetLayerStackMask() NN_NOEXCEPT
        {
            nn::vi::LayerStackSet value = {};
            value
                .Set<nn::vi::LayerStackFlags::Default>()
                .Set<nn::vi::LayerStackFlags::Lcd>()
                .Set<nn::vi::LayerStackFlags::Screenshot>()
                .Set<nn::vi::LayerStackFlags::Recording>()
                .Set<nn::vi::LayerStackFlags::LastFrame>()
                .Set<nn::vi::LayerStackFlags::Arbitrary>()
                .Set<nn::vi::LayerStackFlags::Null>();

            if(!IsProdMode())
            {
                value.Set<nn::vi::LayerStackFlags::ApplicationForDebug>();
            }

            return value;
        }

        nn::vi::LayerStackSet GetPhysicalLayerStackMask() NN_NOEXCEPT
        {
            nn::vi::LayerStackSet value = {};
            value
                .Set<nn::vi::LayerStackFlags::Default>()
                .Set<nn::vi::LayerStackFlags::Lcd>();
            return value;
        }

    };

    void DirectLayerVisibilityManager::Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());

        m_LayerId = 0;
        m_ZOrder = 0;
        m_IsActive = false;
        m_ForegroundStacks = {};
        m_InstantCaptureStacks = {};
        m_MovieRecordingStacks = {};
        m_DebugScreenshotStacks = {};

        m_CurrentLayerStacks = {};
        m_CurrentZOrder = 0;

        m_DumpAccessor.Initialize(this);

        m_IsInitialized = true;
    }

    void DirectLayerVisibilityManager::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        HideCurrentLayerImpl();

        m_LayerId = 0;
        m_ZOrder = 0;
        m_IsActive = false;
        m_ForegroundStacks = {};
        m_InstantCaptureStacks = {};
        m_MovieRecordingStacks = {};
        m_DebugScreenshotStacks = {};

        m_CurrentLayerStacks = {};
        m_CurrentZOrder = 0;

        m_DumpAccessor.Finalize();

        m_IsInitialized = false;
    }

    void DirectLayerVisibilityManager::Activate() NN_NOEXCEPT
    {
        m_IsActive = true;
        UpdateCurrentLayerConfigImpl(false);
    }

    void DirectLayerVisibilityManager::Deactivate() NN_NOEXCEPT
    {
        m_IsActive = false;
        UpdateCurrentLayerConfigImpl(false);
    }

    void DirectLayerVisibilityManager::SetLayerId(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        m_LayerId = layerId;
        UpdateCurrentLayerConfigImpl(true);
    }

    void DirectLayerVisibilityManager::SetForegroundVisibility(bool isVisible, int zOrder, bool isCaptureTarget) NN_NOEXCEPT
    {
        nn::vi::LayerStackSet stacks = {};

        if(isVisible)
        {
            stacks.Set<nn::vi::LayerStackFlags::Default>();
            if(isCaptureTarget)
            {
                stacks.Set<nn::vi::LayerStackFlags::LastFrame>();
                stacks.Set<nn::vi::LayerStackFlags::Screenshot>();
            }
        }

        m_ForegroundStacks = stacks;
        m_ZOrder = isVisible ? zOrder : 0;
        UpdateCurrentLayerConfigImpl(false);
    }

    void DirectLayerVisibilityManager::SetInstantCaptureVisibility(bool isVisible) NN_NOEXCEPT
    {
        nn::vi::LayerStackSet value = {};

        if(isVisible)
        {
            value.Set<nn::vi::LayerStackFlags::Arbitrary>();
        }

        m_InstantCaptureStacks = value;
        UpdateCurrentLayerConfigImpl(false);
    }

    void DirectLayerVisibilityManager::SetMovieRecordingVisibility(bool isVisible) NN_NOEXCEPT
    {
        nn::vi::LayerStackSet value = {};

        if(isVisible)
        {
            value.Set<nn::vi::LayerStackFlags::Recording>();
        }

        m_MovieRecordingStacks = value;
        UpdateCurrentLayerConfigImpl(false);
    }

    void DirectLayerVisibilityManager::SetDebugScreenshotVisibility(bool isVisible) NN_NOEXCEPT
    {
        nn::vi::LayerStackSet value = {};

        if(!IsProdMode() && isVisible)
        {
            value.Set<nn::vi::LayerStackFlags::ApplicationForDebug>();
        }

        m_DebugScreenshotStacks = value;
        UpdateCurrentLayerConfigImpl(false);
    }

    void DirectLayerVisibilityManager::UpdateCurrentLayerConfigImpl(bool isForced) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        if(m_LayerId == 0)
        {
            return;
        }

        nn::vi::LayerConfig config = {};
        config.stacks = {};
        config.zOrder = 0;
        config.mask = GetLayerStackMask();

        if(IsActive())
        {
            nn::vi::LayerStackSet stacks = {};
            stacks |= m_ForegroundStacks;
            stacks |= m_InstantCaptureStacks;
            stacks |= m_MovieRecordingStacks;
            stacks |= m_DebugScreenshotStacks;
            config.stacks = stacks;
        }

        if(config.stacks.IsAnyOn())
        {
            config.zOrder = m_ZOrder;
        }

        // 物理ディスプレイに出画しないなら NullDisplay に入れる
        if((config.stacks & GetPhysicalLayerStackMask()).IsAllOff())
        {
            config.stacks.Set<nn::vi::LayerStackFlags::Null>();
        }

        // レイヤスタックが変化したら更新
        if(isForced || config.zOrder != m_CurrentZOrder || m_CurrentLayerStacks != config.stacks)
        {
            NN_AM_DISPCTRL_LOG_PLVM("update: layerId=%llu z=%d stacks=%X\n", m_LayerId, config.zOrder, config.stacks);
            DisplayPrimitiveOperation::SetDisplayLayerConfig(m_LayerId, config);
            m_CurrentLayerStacks = config.stacks;
            m_CurrentZOrder = config.zOrder;
        }
    }

    void DirectLayerVisibilityManager::HideCurrentLayerImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        if(m_LayerId == 0)
        {
            return;
        }

        nn::vi::LayerConfig config = {};
        config.stacks = {};
        config.zOrder = 0;
        config.mask = GetLayerStackMask();

        config.stacks.Set<nn::vi::LayerStackFlags::Null>();

        NN_AM_DISPCTRL_LOG_PLVM("hide: layerId=%llu z=%d stacks=%X\n", m_LayerId, m_ZOrder, config.stacks);
        DisplayPrimitiveOperation::SetDisplayLayerConfig(m_LayerId, config);
        m_CurrentLayerStacks = config.stacks;
        m_CurrentZOrder = config.zOrder;
    }


}}}}}
