﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_StuckChecker.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_TimeSpan.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_Monitoring.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_ThreadLocalStorage.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/util/util_Exchange.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/err/err_SystemApi.h>

namespace nn { namespace am { namespace service {

namespace detail {

namespace {

nn::Result AddErrorCode(erpt::Context* pContext, const err::ErrorCode& errorCode) NN_NOEXCEPT
{
    char errorString[nn::err::ErrorCode::StringLengthMax];
    err::GetErrorCodeString(errorString, sizeof(errorString), errorCode);
    auto errorStringLength = static_cast<uint32_t>(strnlen(errorString, sizeof(errorString)));
    return pContext->Add(erpt::FieldId::ErrorCode, errorString, errorStringLength);
}

}

class StuckMonitorThreadInfo
{
private:

    typedef detail::StuckMonitorHandler Item;

    os::ThreadType* const m_pThread;
    uint64_t m_SequenceNumber = 0;

    os::Mutex m_Mutex{false};
    bool m_IsValid = true;
    util::IntrusiveList<Item, util::IntrusiveListBaseNodeTraits<Item>> m_List;

    int32_t m_TraceBuffer[16];

public:

    explicit StuckMonitorThreadInfo(os::ThreadType* pThread = os::GetCurrentThread()) NN_NOEXCEPT
        : m_pThread(pThread)
    {
    }

    uint64_t PublishSequenceNumber() NN_NOEXCEPT
    {
        return ++m_SequenceNumber;
    }

    void Add(Item* p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        m_List.push_back(*p);
    }

    void Remove(Item* p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        m_List.erase(m_List.iterator_to(*p));
    }

    void OnExpired() NN_NOEXCEPT
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        if (!m_IsValid)
        {
            return;
        }
        this->m_IsValid = false;
        NN_RESULT_IGNORING_BLOCK
        {
            erpt::Context c(erpt::ErrorInfo);

            // スレッド名
            NN_RESULT_DO(c.AddThreadName(m_pThread));

            // トレース
            auto p = static_cast<Item*>(nullptr);
            auto maxSequenceNumber = static_cast<decltype(m_SequenceNumber)>(0);
            auto n = 0u;
            for (auto&& e : m_List)
            {
                if (!(e.m_SequenceNumber < maxSequenceNumber))
                {
                    p = &e;
                    maxSequenceNumber = e.m_SequenceNumber;
                }
                if (n < sizeof(m_TraceBuffer) / sizeof(*m_TraceBuffer))
                {
                    m_TraceBuffer[n++] = e.m_Kind;
                }
            }
            NN_RESULT_DO(c.Add(erpt::AppletManagerContextTrace, m_TraceBuffer, n));
            auto kind = p ? p->m_Kind : StuckCheckKind_am_Unknown;

            // ログ
            auto getText = [](decltype(p) q)
            {
                return (q && q->m_Text) ? q->m_Text : "unknown";
            };
            NN_UNUSED(getText);
            NN_AM_SERVICE_LOG(error, "Stuck detected: kind = %d (%s)\n", kind, getText(p));
            NN_AM_SERVICE_LOG(error, "  on thread '%s'\n", os::GetThreadNamePointer(m_pThread));
            auto i = 0;
            for (auto&& e : m_List)
            {
                NN_UNUSED(e);
                NN_UNUSED(i);
                NN_AM_SERVICE_LOG(error, "  [%d] kind = %d (%s)\n", i, e.m_Kind, getText(&e));
                ++i;
            }

            // エラーレポート
            err::ErrorCode ec = {};
            ec.category = 2521;
            ec.number = kind;
            NN_RESULT_DO(AddErrorCode(&c, ec));

            // stuck 原因が高優先度スレッドの暴走の場合、
            // CreateReport が返らない可能性があり、
            // これによって m_Mutex が開放されないことで他スレッドが stuck することをさけるため、
            // CreateReport はロック外で行う。
            lk.unlock();
            return c.CreateReport(erpt::ReportType_Invisible);
        };
    }

};

namespace {

template <typename T>
class Tls
{
private:
    os::TlsSlot m_TlsSlot;
public:
    Tls() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::AllocateTlsSlot(&m_TlsSlot, [](uintptr_t p)
        {
            if (p)
            {
                delete reinterpret_cast<T*>(p);
            }
        }));
    }
    T* Get() const NN_NOEXCEPT
    {
        auto p = reinterpret_cast<T*>(os::GetTlsValue(m_TlsSlot));
        if (!p)
        {
            p = new T();
            os::SetTlsValue(m_TlsSlot, reinterpret_cast<uintptr_t>(p));
        }
        return p;
    }
};

class StuckMonitorManager
{
private:

    Tls<StuckMonitorThreadInfo> m_pTlsThreadInfo;

public:

    StuckMonitorThreadInfo* GetCurrentThreadInfo() NN_NOEXCEPT
    {
        return m_pTlsThreadInfo.Get();
    }

    void Add(StuckMonitorHandler* p) NN_NOEXCEPT
    {
        GetCurrentThreadInfo()->Add(p);
    }

    void Remove(StuckMonitorHandler* p) NN_NOEXCEPT
    {
        GetCurrentThreadInfo()->Remove(p);
    }

};

StuckMonitorManager& GetStuckMonitorManager() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(StuckMonitorManager, g_StuckMonitorManager);
    return g_StuckMonitorManager;
}

} // <unnamed>

StuckMonitorHandler::StuckMonitorHandler(int32_t kind, const char* text) NN_NOEXCEPT
    : m_Kind(kind)
    , m_Text(text)
    , m_pThreadInfo(GetStuckMonitorManager().GetCurrentThreadInfo())
    , m_SequenceNumber(m_pThreadInfo->PublishSequenceNumber())
{
    NN_UNUSED(m_Text);
    m_pThreadInfo->Add(this);
}

StuckMonitorHandler::~StuckMonitorHandler() NN_NOEXCEPT
{
    m_pThreadInfo->Remove(this);
}

void StuckMonitorHandler::OnExpired() NN_NOEXCEPT
{
    m_pThreadInfo->OnExpired();
}

ScopedStuckMonitorHandler::ScopedStuckMonitorHandler(int kind, TimeSpan expireTime, const char* text) NN_NOEXCEPT
    : m_Handler(kind, text)
    , m_pHolder(BeginMonitoring(expireTime, &this->m_Handler))
{
}

ScopedStuckMonitorHandler::~ScopedStuckMonitorHandler() NN_NOEXCEPT
{
    EndMonitoring(m_pHolder);
}

} // detail

}}}
