﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_Monitoring.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/os/os_Tick.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ConditionVariable.h>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include "am_CommonUtility.h"

namespace nn { namespace am { namespace service {

struct MonitorHolder
{
    os::Tick baseExpireTime;
    os::Tick expiredTick;
    MonitorHandler* pHandler;
};

namespace {

class Monitor
{
private:

    os::Mutex m_Mutex{false};
    os::ConditionVariable m_Cv;
    bool m_IsSuspended{false};
    util::optional<os::Tick> m_NextTick;

    MonitorHolder* m_Holders;
    size_t m_HolderCount;
    Bit64 m_Availables;
    Bit64 m_InUses;

    template <typename F>
    void ForeachInUses(F&& f) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());
        auto inUses = m_InUses;
        while (inUses)
        {
            auto i = cntt0(inUses);
            auto mask = static_cast<decltype(inUses)>(1) << i;
            inUses &= ~mask;
            f(&m_Holders[i]);
        }
    }

public:

    Monitor(MonitorHolder* holders, size_t holderCount) NN_NOEXCEPT
        : m_Holders(holders)
        , m_HolderCount(holderCount)
        , m_InUses(0)
    {
        NN_SDK_REQUIRES(holderCount <= NN_BITSIZEOF(decltype(m_Availables)));
        this->m_Availables = 0;
        auto mask = static_cast<decltype(m_Availables)>(1);
        for (auto i = 0u; i < holderCount; ++i)
        {
            m_Availables |= mask;
            mask <<= 1;
        }
    }

    void Run() NN_NOEXCEPT
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        for (;;)
        {
            auto currentTick = os::GetSystemTick();
            util::optional<os::Tick> minTick;
            if (!m_IsSuspended)
            {
                ForeachInUses([&](MonitorHolder* p)
                {
                    auto&& h = *p;
                    if (h.expiredTick < currentTick)
                    {
                        // expire してたら一度だけ実行
                        if (h.pHandler)
                        {
                            h.pHandler->OnExpired();
                            h.pHandler = nullptr;
                        }
                    }
                    else
                    {
                        // expire してないものの最小値をとる
                        if (!(static_cast<bool>(minTick) && minTick.value() <= h.expiredTick))
                        {
                            minTick = h.expiredTick;
                        }
                    }
                });
            }
            this->m_NextTick = minTick;
            if (minTick)
            {
                m_Cv.TimedWait(m_Mutex, (minTick.value() - currentTick).ToTimeSpan());
            }
            else
            {
                m_Cv.Wait(m_Mutex);
            }
        }
    }

    void Suspend() NN_NOEXCEPT
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        this->m_IsSuspended = true;
    }

    void Resume(os::Tick extraTime) NN_NOEXCEPT
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        auto currentTick = os::GetSystemTick();
        ForeachInUses([&](MonitorHolder* p)
        {
            auto&& h = *p;
            h.expiredTick = currentTick + h.baseExpireTime + extraTime;
        });
        this->m_IsSuspended = false;
        m_Cv.Signal();
    }

    MonitorHolder* Begin(os::Tick tick, MonitorHandler* pHandler) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(tick.GetInt64Value() >= 0);
        NN_SDK_REQUIRES_NOT_NULL(pHandler);
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        auto&& availables = m_Availables;
        if (!availables)
        {
            return nullptr;
        }
        auto i = cntt0(availables);
        auto mask = static_cast<decltype(m_Availables)>(1) << i;
        availables &= ~mask;
        m_InUses |= mask;
        auto&& h = m_Holders[i];
        h.pHandler = pHandler;
        h.baseExpireTime = tick;
        h.expiredTick = os::GetSystemTick() + tick;
        if (!(static_cast<bool>(m_NextTick) && m_NextTick.value() <= h.expiredTick))
        {
            m_Cv.Signal();
        }
        return &h;
    }

    bool End(MonitorHolder* pHolder) NN_NOEXCEPT
    {
        if (!pHolder)
        {
            return false;
        }
        auto i = pHolder - m_Holders;
        NN_SDK_REQUIRES(static_cast<size_t>(i) < m_HolderCount);
        NN_UNUSED(m_HolderCount);
        auto&& availables = m_Availables;
        auto mask = static_cast<decltype(m_Availables)>(1) << i;
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        NN_SDK_REQUIRES(!(mask & availables));
        availables |= mask;
        m_InUses &= ~mask;
        // スレッドを起こす時間が早まることはないので、シグナルの必要はない
        return pHolder->pHandler == nullptr;
    }

};

Monitor& GetMonitor() NN_NOEXCEPT
{
    // VC++2013 でのビルドエラー回避
    struct Aggregate
    {
        MonitorHolder holders[64];
        Monitor monitor;
        Aggregate() NN_NOEXCEPT
            : monitor(holders, sizeof(holders) / sizeof(*holders))
        {
        }
    };
    NN_FUNCTION_LOCAL_STATIC(Aggregate, g_Aggregate);
    return g_Aggregate.monitor;

}

} // <unnamed>

void RunMonitor() NN_NOEXCEPT
{
    GetMonitor().Run();
}

void SuspendMonitoring() NN_NOEXCEPT
{
    GetMonitor().Suspend();
}

void ResumeMonitoring(TimeSpan extraTime) NN_NOEXCEPT
{
    GetMonitor().Resume(os::Tick(extraTime));
}

MonitorHolder* BeginMonitoring(os::Tick expireTime, MonitorHandler* pHandler) NN_NOEXCEPT
{
    return GetMonitor().Begin(expireTime, pHandler);
}

bool EndMonitoring(MonitorHolder* pHolder) NN_NOEXCEPT
{
    return GetMonitor().End(pHolder);
}

}}}
