﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_IntegratedSystemApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <utility>
#include <memory>
#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/am/service/am_IntegratedApplication.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/util/util_Span.h>
#include <nn/util/util_UniqueLock.h>

namespace nn { namespace am { namespace service {

IntegratedSystemApplet::IntegratedSystemApplet(AppletSystem* pAppletSystem, applet::AppletId appletId) NN_NOEXCEPT
    : Derived(pAppletSystem)
    , m_AppletId(appletId)
{
}

applet::AppletIdentityInfo IntegratedSystemApplet::GetAppletIdentityInfo() NN_NOEXCEPT
{
    applet::AppletIdentityInfo ret = {};
    ret.appletId = m_AppletId;
    return ret;
}

Result IntegratedSystemApplet::CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT
{
    return CreateAppletProcessImpl(pOut, m_AppletId);
}

std::shared_ptr<process::NsProcess> IntegratedSystemApplet::MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT
{
    return GetAppletSystem()->GetNsProcessManager()->MakeLibraryAppletProcess(processId);
}

class IntegratedSystemApplet::SystemAppletSelfProxyImpl
    : public MainAppletSelfProxyImpl
{
public:

    explicit SystemAppletSelfProxyImpl(std::shared_ptr<IntegratedSystemApplet> p, os::ProcessId processId) NN_NOEXCEPT
        : MainAppletSelfProxyImpl(p, processId)
    {
    }

    std::shared_ptr<SystemAppletSelfProxyImpl> GetHomeMenuFunctions() NN_NOEXCEPT
    {
        return std::static_pointer_cast<SystemAppletSelfProxyImpl>(shared_from_this());
    }

    std::shared_ptr<SystemAppletSelfProxyImpl> GetGlobalStateController() NN_NOEXCEPT
    {
        return std::static_pointer_cast<SystemAppletSelfProxyImpl>(shared_from_this());
    }

    Result LockForeground() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(seq, "LockForeground()\n");
        this->GetApplet()->MoveWindowToScreenLock();
        NN_RESULT_SUCCESS;
    }

    Result UnlockForeground() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(seq, "UnlockForeground()\n");
        this->GetApplet()->MoveWindowToNormal();
        NN_RESULT_SUCCESS;
    }

    Result RequestToGetForeground() NN_NOEXCEPT
    {
        return GetApplet()->MoveWindowToTop();
    }

    Result RequestToEnterSleep() NN_NOEXCEPT
    {
        NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedSystemApplet::SystemAppletSelfProxyImpl::RequestToEnterSleep()");
    }

    Result EnterSleep() NN_NOEXCEPT
    {
        NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: IntegratedSystemApplet::SystemAppletSelfProxyImpl::EnterSleep()");
    }

    Result StartSleepSequence(bool needsFinishedMssage) NN_NOEXCEPT
    {
        auto pGlobalStateManager = GetApplet()->GetAppletSystem()->GetGlobalStateManager();
        if (needsFinishedMssage)
        {
            // TODO: TORIAEZU: 本来は通知先は Applet ではなく SystemAppletSelfProxyImpl とするべき
            pGlobalStateManager->SetNotifier(GetApplet()->shared_from_this());
        }
        return pGlobalStateManager->StartSleepSequence();
    }

    Result StartShutdownSequence() NN_NOEXCEPT
    {
        return GetApplet()->GetAppletSystem()->GetGlobalStateManager()->StartShutdownSequence();
    }

    Result StartRebootSequence() NN_NOEXCEPT
    {
        return GetApplet()->GetAppletSystem()->GetGlobalStateManager()->StartRebootSequence();
    }

    Result LoadAndApplyIdlePolicySettings() NN_NOEXCEPT
    {
        return GetApplet()->GetAppletSystem()->GetGlobalStateManager()->LoadAndApplyIdlePolicySettings();
    }

    Result NotifyCecSettingsChanged() NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(omm_NotifyCecSettingsChanged, 60);
        omm::NotifyCecSettingsChanged();
        NN_RESULT_SUCCESS;
    }

    Result UpdateDefaultDisplayResolution() NN_NOEXCEPT
    {
        omm::UpdateDefaultDisplayResolution();
        NN_RESULT_SUCCESS;
    }

    Result ShouldSleepOnBoot(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        *pOut = omm::ShouldSleepOnBoot();
        NN_RESULT_SUCCESS;
    }

    Result GetHdcpAuthenticationFailedEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        os::SystemEvent event;
        nn::omm::GetHdcpAuthenticationFailedEvent(&event);
        // SIGLO-69853:
        //  IntegratedApplet::AppletSelfProxyImpl::GetDefaultDisplayResolutionChangeEvent() のコメントを参照
        *outValue = sf::NativeHandle(event.DetachReadableHandle(), true);
        NN_RESULT_SUCCESS;
    }

    Result PopFromGeneralChannel(sf::Out<sf::SharedPointer<IStorage>> pOut) NN_NOEXCEPT
    {
        auto ret = GetApplet()->GetAppletSystem()->GetGeneralChannel()->Pop();
        NN_RESULT_THROW_UNLESS(ret, am::ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s", NN_CURRENT_FUNCTION_NAME);
        NN_RESULT_SUCCESS;
    }

    Result GetPopFromGeneralChannelEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->GetAppletSystem()->GetGeneralChannel()->GetPopEventHandle();
        NN_RESULT_SUCCESS;
    }

    Result CreateApplication(sf::Out<sf::SharedPointer<am::service::IApplicationAccessor>> pOut, ncm::ApplicationId id) NN_NOEXCEPT
    {
        // アプリ起動要求元の情報
        applet::ApplicationLaunchRequestInfo launchRequestInfo = {};
        launchRequestInfo.requestingAppletId     = applet::AppletId_SystemAppletMenu;
        launchRequestInfo.requestingMainAppletId = applet::AppletId_SystemAppletMenu;
        launchRequestInfo.applicationId          = ncm::ApplicationId::GetInvalidId();

        sf::SharedPointer<am::service::IApplicationAccessor> p;
        NN_RESULT_DO(GetApplet()->GetAppletSystem()->CreateApplication(&p, id, nullptr, launchRequestInfo));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result CreateSystemApplication(sf::Out<sf::SharedPointer<am::service::IApplicationAccessor>> pOut, ncm::SystemApplicationId id) NN_NOEXCEPT
    {
        sf::SharedPointer<am::service::IApplicationAccessor> p;
        NN_RESULT_DO(GetApplet()->GetAppletSystem()->CreateSystemApplication(&p, id, GetApplet()->SharedFromThis()));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result PopLaunchRequestedApplication(sf::Out<sf::SharedPointer<am::service::IApplicationAccessor>> pOut) NN_NOEXCEPT
    {
        auto p = GetApplet()->PopLaunchRequestedApplication();
        NN_RESULT_THROW_UNLESS(p, am::ResultNoStorage()); // TODO: 適切な Result
        *pOut = p->CreateAccessor();
        NN_RESULT_SUCCESS;
    }

    Result PopFloatingApplicationForDevelopment(sf::Out<sf::SharedPointer<am::service::IApplicationAccessor>> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->GetAppletSystem()->PopFloatingApplication();
        NN_RESULT_SUCCESS;
    }

    Result SetDefaultHomeButtonLongPressTime(int64_t longPressTimeInNanoSeconds)
    {
        GetApplet()->GetAppletSystem()->SetDefaultHomeButtonLongPressTime(TimeSpan::FromNanoSeconds(longPressTimeInNanoSeconds));
        NN_RESULT_SUCCESS;
    }

    std::shared_ptr<SystemAppletSelfProxyImpl> GetApplicationCreator() NN_NOEXCEPT
    {
        return std::static_pointer_cast<SystemAppletSelfProxyImpl>(shared_from_this());
    }

    Result PopRequestLaunchApplicationForDebug(sf::Out<ncm::ApplicationId> pApplicationId, sf::Out<int> pOutUidCount, util::Span<account::Uid> outUids) NN_NOEXCEPT
    {
        return GetApplet()->m_ControllerForDebug.PopRequestLaunchApplication(pApplicationId.GetPointer(), pOutUidCount.GetPointer(), outUids);
    }

private:

    IntegratedSystemApplet* GetApplet() NN_NOEXCEPT
    {
        return static_cast<IntegratedSystemApplet*>(AppletSelfProxyImpl::GetApplet());
    }

    os::SystemEvent m_SleepSequenceEvent{os::EventClearMode_ManualClear, true};

};

IntegratedSystemApplet::AppletProxyInfo IntegratedSystemApplet::DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT
{
    return MakeAppletProxyInfo<SystemAppletProxy>(MakeShared<SystemAppletSelfProxyImpl>(SharedFromThis(), processId));
}

void IntegratedSystemApplet::NotifyFloatingApplicationExistence() NN_NOEXCEPT
{
    AppletMessage message = {};
    message.message[0] = applet::Message_FloatingApplicationDetected;
    PushMessageToProcess(message);
}

void IntegratedSystemApplet::NotifyApplicationExit() NN_NOEXCEPT
{
    AppletMessage message = {};
    message.message[0] = applet::Message_ApplicationExited;
    PushMessageToProcess(message);
}

void IntegratedSystemApplet::PushLaunchRequestedApplication(std::shared_ptr<IntegratedApplication> pApplication) NN_NOEXCEPT
{
    {
        std::lock_guard<decltype(m_LaunchRequestedApplicationListMutex)> lk(m_LaunchRequestedApplicationListMutex);
        m_LaunchRequestedApplicationList.push_back(std::move(pApplication));
    }
    AppletMessage message = {};
    message.message[0] = applet::Message_LaunchApplicationRequested;
    PushMessageToProcess(message);
}

std::shared_ptr<IntegratedApplication> IntegratedSystemApplet::PopLaunchRequestedApplication() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_LaunchRequestedApplicationListMutex)> lk(m_LaunchRequestedApplicationListMutex);
    if (m_LaunchRequestedApplicationList.empty())
    {
        return nullptr;
    }
    auto ret = m_LaunchRequestedApplicationList.front();
    m_LaunchRequestedApplicationList.pop_front();
    return ret;
}

void IntegratedSystemApplet::OnHomeButtonLongPressed() NN_NOEXCEPT
{
    GetAppletSystem()->NotifyHomeButtonLongPressedToOverlayApplet();
}

inline auto IntegratedSystemApplet::ControllerForDebug::Lock() NN_NOEXCEPT
{
    return util::MakeUniqueLock(m_Mutex);
}

Result IntegratedSystemApplet::ControllerForDebug::RequestLaunchApplication(ncm::ApplicationId applicationId, util::Span<const account::Uid> uids) NN_NOEXCEPT
{
    {
        auto lk = Lock();
        LaunchApplicationRequestInfo info = {};
        info.applicationId = applicationId;
        auto dest = util::MakeSpan(info.uids);
        NN_RESULT_THROW_UNLESS(uids.size() <= dest.size(), ResultInvalidParameter());
        NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<int>(uids.size()), ResultInvalidParameter());
        std::copy(uids.begin(), uids.end(), dest.begin());
        info.uidCount = static_cast<int>(uids.size());
        this->m_pLaunchApplicationRequestInfo = info;
    }
    {
        m_pParent->PushMessageToProcess({{applet::Message_RequestToLaunchApplicationForDebug}});
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedSystemApplet::ControllerForDebug::PopRequestLaunchApplication(ncm::ApplicationId* pApplicationId, int* pOutUidCount, util::Span<account::Uid> outUids) NN_NOEXCEPT
{
    auto lk = Lock();
    NN_RESULT_THROW_UNLESS(m_pLaunchApplicationRequestInfo, am::ResultNoData());
    const auto& info = *m_pLaunchApplicationRequestInfo;

    auto src = util::MakeSpan(info.uids.data(), info.uidCount);
    NN_RESULT_THROW_UNLESS(src.size() <= outUids.size(), ResultInvalidParameter());
    NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<int>(src.size()), ResultInvalidParameter());

    *pApplicationId = info.applicationId;
    auto it = std::copy(src.begin(), src.end(), outUids.begin());
    std::fill(it, outUids.end(), account::Uid{});
    *pOutUidCount = static_cast<int>(src.size());

    this->m_pLaunchApplicationRequestInfo = util::nullopt;
    NN_RESULT_SUCCESS;
}

}}}
