﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_IntegratedOverlayApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <utility>
#include <memory>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <mutex>

namespace nn { namespace am { namespace service {

IntegratedOverlayApplet::IntegratedOverlayApplet(AppletSystem* pAppletSystem, applet::AppletId appletId) NN_NOEXCEPT
    : IntegratedApplet(pAppletSystem)
    , m_AppletId(appletId)
{
}

applet::AppletIdentityInfo IntegratedOverlayApplet::GetAppletIdentityInfo() NN_NOEXCEPT
{
    applet::AppletIdentityInfo ret = {};
    ret.appletId = m_AppletId;
    return ret;
}

Result IntegratedOverlayApplet::CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT
{
    return CreateAppletProcessImpl(pOut, m_AppletId);
}

std::shared_ptr<process::NsProcess> IntegratedOverlayApplet::MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT
{
    return GetAppletSystem()->GetNsProcessManager()->MakeLibraryAppletProcess(processId);
}

void IntegratedOverlayApplet::NotifyApplicationStarted(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    if (applicationId == ncm::ApplicationId::GetInvalidId())
    {
        return;
    }
    {
        std::lock_guard<decltype(m_ApplicationNotificationMutex)> lk(m_ApplicationNotificationMutex);
        this->m_CurrentApplicationId = applicationId;
    }
    this->PushMessageToProcess({{applet::Message_ShowApplicationLogo}});
}

void IntegratedOverlayApplet::NotifyApplicationRunning(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    if (applicationId == ncm::ApplicationId::GetInvalidId())
    {
        return;
    }
    this->PushMessageToProcess({{applet::Message_HideApplicationLogo}});
}

void IntegratedOverlayApplet::NotifyApplicationExited(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_UNUSED(applicationId);
    {
        std::lock_guard<decltype(m_ApplicationNotificationMutex)> lk(m_ApplicationNotificationMutex);
        this->m_CurrentApplicationId = ncm::ApplicationId::GetInvalidId();
    }
    this->PushMessageToProcess({{applet::Message_ForceHideApplicationLogo}});
}

void IntegratedOverlayApplet::SetVrModeFlag(bool isEnabled, bool fromOe) NN_NOEXCEPT
{
    auto updater = this->GetAppletSystem()->GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(this->GetWindow()).isVrMode = isEnabled;
    updater.RefWindowProperty(this->GetWindow()).isVrModeChangedByApplication = fromOe;
}

ncm::ApplicationId IntegratedOverlayApplet::GetCurrentApplicationId() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ApplicationNotificationMutex)> lk(m_ApplicationNotificationMutex);
    return m_CurrentApplicationId;
}

class IntegratedOverlayApplet::OverlayAppletSelfProxyImpl
    : public AppletSelfProxyImpl
{
public:

    explicit OverlayAppletSelfProxyImpl(std::shared_ptr<IntegratedOverlayApplet> p, os::ProcessId processId) NN_NOEXCEPT
        : AppletSelfProxyImpl(p, processId)
    {
        AppletSelfProxyImpl::SetNotifiesFocusStateChanged(false);
    }

    Result BeginToWatchShortHomeButtonMessage() NN_NOEXCEPT
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = true;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonLongPressed = true;
        updater.RefWindowProperty(GetApplet()->GetWindow()).hidInputTakingMode = window::HidWindowController::HidInputTakingMode_TakeForcedExclusive;
        NN_RESULT_SUCCESS;
    }

    Result EndToWatchShortHomeButtonMessage() NN_NOEXCEPT
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = false;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonLongPressed = false;
        updater.RefWindowProperty(GetApplet()->GetWindow()).hidInputTakingMode = window::HidWindowController::HidInputTakingMode_Never;
        NN_RESULT_SUCCESS;
    }

    Result BeginToObserveHidInputForDevelop() NN_NOEXCEPT
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = false;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonLongPressed = false;
        updater.RefWindowProperty(GetApplet()->GetWindow()).hidInputTakingMode = window::HidWindowController::HidInputTakingMode_TakeForcedShared;
        NN_RESULT_SUCCESS;
    }

    Result SetAutoSleepTimeAndDimmingTimeEnabled(bool isEnabled)
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).needsSetHandlingContext = isEnabled;
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationIdForLogo(sf::Out<ncm::ApplicationId> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->GetCurrentApplicationId();
        NN_RESULT_SUCCESS;
    }

    Result TerminateApplicationAndSetReason(uint32_t resultInnerValue) NN_NOEXCEPT
    {
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "TerminateApplicationAndSetReason(result=0x%08x)", resultInnerValue);
        GetApplet()->GetAppletSystem()->TerminateCurrentApplicationAndSetResult(result::detail::ConstructResult(resultInnerValue));
        NN_RESULT_SUCCESS;
    }

    Result SetGpuTimeSliceBoost(uint64_t nanoSeconds) NN_NOEXCEPT
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).gpuTimeSliceBoost = TimeSpan::FromNanoSeconds(nanoSeconds);
        NN_RESULT_SUCCESS;
    }

    std::shared_ptr<OverlayAppletSelfProxyImpl> GetOverlayFunctions() NN_NOEXCEPT
    {
        return std::static_pointer_cast<OverlayAppletSelfProxyImpl>(shared_from_this());
    }

    Result SetScreenShotPermissionGlobally(bool isPermitted) NN_NOEXCEPT
    {
        // 撮影禁止区間中は OA が横取りして常に撮影禁止を通知する
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetScreenShotPermissionGlobally(%s)", isPermitted ? "true" : "false");
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesCaptureButtonShortPressed = !isPermitted;
        NN_RESULT_SUCCESS;
    }

private:

    IntegratedOverlayApplet* GetApplet() NN_NOEXCEPT
    {
        return static_cast<IntegratedOverlayApplet*>(AppletSelfProxyImpl::GetApplet());
    }

};

IntegratedOverlayApplet::AppletProxyInfo IntegratedOverlayApplet::DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT
{
    return MakeAppletProxyInfo<OverlayAppletProxy>(MakeShared<OverlayAppletSelfProxyImpl>(SharedFromThis(), processId));
}

void IntegratedOverlayApplet::OnHomeButtonLongPressed() NN_NOEXCEPT
{
    // 試遊台の場合は HOME ボタン長押しを常に無効化
    if ( IsQuestMode() )
    {
        return;
    }
    NN_AM_SERVICE_APPLET_LOG(seq, this, "OnHomeButtonLongPressed");
    PushMessageToProcess(applet::Message_DetectLongPressingHomeButton);
}

bool IntegratedOverlayApplet::IsScreenShotEnabled() NN_NOEXCEPT
{
    // OA は常に撮影禁止を返す
    return false;
}

}}}
