﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_IntegratedApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <utility>
#include <memory>
#include <cstring>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/am/service/display/am_IntegratedDisplayManager.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/util/util_Exchange.h>
#include <nn/util/util_Optional.h>
#include <nn/am/am_ResultPrivate.h>
#include <nn/oe/oe_GamePlayRecordingApi.private.h>
#include <nn/oe/oe_TvPowerStateMatchingControl.private.h>

#include <nn/audio/audio_Applet.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_ConvertScreenShotOrientation.h>
#include <nn/album/album_ImageOrientation.h>
#include <nn/nfc/am/nfc_Am.h>
#include <nn/hid/system/hid_Applet.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/idle/idle_SystemApi.h>
#include <nn/lbl/lbl.h>
#include <nn/lbl/lbl_Switch.h>
#include <nn/irsensor/system/irsensor_Applet.h>
#include <nn/hidbus/system/hidbus_Applet.h>
#include <nn/grc/grc_Api.h>
#include <nn/grc/grc_Result.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/am/service/am_OverlayNotificationControl.h>
#include <nn/am/service/am_GpuErrorControl.h>
#include <nn/am/service/am_GpuResourceControl.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/am_BtmWrapper.h>
#include <nn/applet/applet_InternalFunctions.h>
#include "am_AudioControl.h"
#include "am_GrcControl.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#endif

namespace nn { namespace am { namespace service {

inline window::WindowManager::Updater IntegratedApplet::BeginWindowUpdate() NN_NOEXCEPT
{
    return GetAppletSystem()->GetWindowManager()->BeginUpdate();
}

namespace {

class NullWindowTransiter : public window::IWindowTransiter
{
};

NullWindowTransiter g_NullWindowTransiter;


Result SuspendWindowImplUnsafe(os::NativeHandle handle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto result = svc::SetProcessActivity(svc::Handle(handle), svc::ProcessActivity_Paused);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(svc::ResultInvalidState)
        {
            // 対象プロセスが Terminating および Terminated の場合は成功扱い
        }
    NN_RESULT_END_TRY
#else
    NN_AM_SERVICE_LOG(seq, "SuspendWindowImplUnsafe() is no operation.\n");
    NN_UNUSED(handle);
#endif
    NN_RESULT_SUCCESS;
}

Result ResumeWindowImplUnsafe(os::NativeHandle handle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto result = svc::SetProcessActivity(svc::Handle(handle), svc::ProcessActivity_Runnable);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(svc::ResultInvalidState)
        {
            // 対象プロセスが Terminating および Terminated の場合は成功扱い
        }
    NN_RESULT_END_TRY
#else
    NN_AM_SERVICE_LOG(seq, "ResumeWindowImplUnsafe() is no operation.\n");
    NN_UNUSED(handle);
#endif
    NN_RESULT_SUCCESS;
}

}

void IntegratedApplet::CreateWindow() NN_NOEXCEPT
{
    const auto& windowProperty = GetInitialWindowProperty();
    this->m_pWindow = GetAppletSystem()->GetWindowManager()->CreateWindow(m_pWindowGroup, &g_NullWindowTransiter, windowProperty);
    std::unique_lock<decltype(m_WindowDeletionMutex)> lk(m_WindowDeletionMutex);
    auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
    m_pWindow->SetSubOrder(m_WindowSubOrder);
    if (m_WindowShouldCreatedAsActive)
    {
        updater.ActivateWindow(m_pWindow);
    }
    NN_AM_SERVICE_APPLET_LOG(seq, this, "CreateWindow(subOrder = %u)", m_WindowSubOrder);
    if (m_AutoMoveToTop)
    {
        NN_AM_SERVICE_APPLET_LOG(dev, this, "AutoMoveToTop");
        updater.MoveToTop(m_pWindowGroup.get());
        this->m_AutoMoveToTop = false;
    }
}

void IntegratedApplet::DestroyWindow() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_WindowDeletionMutex)> lk(m_WindowDeletionMutex);
    auto pWindow = util::Exchange(&this->m_pWindow, nullptr);
    lk.unlock();
    if (m_WindowShouldCreatedAsActive)
    {
        auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.DeactivateWindow(pWindow);
    }
    GetAppletSystem()->GetWindowManager()->RemoveWindow(pWindow);
}

void IntegratedApplet::SetWindowTansiter(window::IWindowTransiter* pTransiter) NN_NOEXCEPT
{
    if (m_WindowShouldCreatedAsActive)
    {
        NN_AM_SERVICE_APPLET_LOG(seq, this, "activate window");
        {
            auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
            updater.RefWindowProperty(m_pWindow).foregroundMode = this->GetForegroundMode();
            updater.RefWindowProperty(m_pWindow).pGpuResourceGroupId = this->GetGpuResourceGroupId();
        }
        GetAppletSystem()->GetWindowManager()->ChangeWindowTransiter(m_pWindow, pTransiter);
        {
            auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
            updater.RefWindowProperty(m_pWindow).handlesCaptureButtonShortPressed = this->IsCaptureTarget();
            updater.RefWindowProperty(m_pWindow).handlesCaptureButtonLongPressed = this->IsCaptureTarget();
        }
    }
}

void IntegratedApplet::ClearWindowTansiter() NN_NOEXCEPT
{
    if (m_WindowShouldCreatedAsActive)
    {
        NN_AM_SERVICE_APPLET_LOG(seq, this, "deactivate window");
        GetAppletSystem()->GetWindowManager()->ChangeWindowTransiter(m_pWindow, &g_NullWindowTransiter);
        {
            auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
            updater.RefWindowProperty(m_pWindow).displayIsValid = true;
        }
    }
}

void IntegratedApplet::NotifyControllerFirmwareUpdateSectionChanged() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    auto pSelfProxy = m_pSelfProxy.lock();
    lk.unlock();
    if (pSelfProxy)
    {
        pSelfProxy->NotifyControllerFirmwareUpdateSectionChangedImpl();
    }
}

void IntegratedApplet::NotifyVrModeCurtainRequired() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    auto pSelfProxy = m_pSelfProxy.lock();
    lk.unlock();
    if (pSelfProxy)
    {
        pSelfProxy->NotifyVrModeCurtainRequiredImpl();
    }
}

class IntegratedApplet::ProcessGateway
    : public window::IWindowTransiter
{
public:

    explicit ProcessGateway(std::shared_ptr<IntegratedApplet> pApplet, applet::AppletResourceUserId appletResourceUserId) NN_NOEXCEPT
        : m_pApplet(std::move(pApplet))
        , m_pWindow(m_pApplet->m_pWindow)
        , m_AppletResourceUserId(appletResourceUserId)
        , m_BootMode(pm::GetBootMode())
    {
        auto info = GetApplet()->GetAppletIdentityInfo();
        if (info.IsApplication())
        {
            NN_AM_SERVICE_STUCK_CHECKED(capsrv_RegisterAruid, 60, capsrv::RegisterAppletResourceUserId(m_AppletResourceUserId, info.applicationId));
        }
        NN_AM_SERVICE_STUCK_CHECKED(audio_RegisterAruid, 60, audio::RegisterAppletResourceUserId(m_AppletResourceUserId));
        RegisterAruidForAudioControl(m_AppletResourceUserId);
        NN_AM_SERVICE_STUCK_CHECKED(hid_RegisterAruid, 60, hid::system::RegisterAppletResourceUserId(m_AppletResourceUserId, false));
        NN_AM_SERVICE_STUCK_CHECKED(irsensor_RegisterAruid, 60, irsensor::system::RegisterAppletResourceUserId(m_AppletResourceUserId, false));
        NN_AM_SERVICE_STUCK_CHECKED(hidbus_RegisterAruid, 60, hidbus::system::RegisterAppletResourceUserId(m_AppletResourceUserId, info.appletId));
        BtmRegisterAppletResourceUserId(m_AppletResourceUserId, info.appletId);
        RegisterToGpuResourceControl(GetApplet()->GetGpuResourceGroupId(), m_AppletResourceUserId);
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ProcessGateway(): Registered ARUID = %lld", m_AppletResourceUserId);

        this->SetScreenShotAppletIdentityInfoImpl(GetApplet()->GetAppletIdentityInfo());
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateDisplayLayerImpl());

        ncm::ProgramId id;
        if( GetApplet()->MakeProgramIdFromAppletIdentityInfo(&id, info).IsSuccess() )
        {
            RegisterRunningApplet(id);
        }
        this->m_IsGamePlayRecordingForceDisabledByFwdbgSettings = IsContinuousRecordingForceDisabled();
    }

    void SetProcessHandle(sf::NativeHandle processHandle) NN_NOEXCEPT
    {
        m_ProcessHandle = std::move(processHandle);
    }

    void Activate(const ProxyOption& proxyOption) NN_NOEXCEPT
    {
        this->m_ProxyOption = proxyOption;
        if (proxyOption.forOe)
        {
            RegisterToGpuErrorControl(m_AppletResourceUserId);
            // 実アプリケーションの場合だけ、初期値を false へ
            this->m_NotifiesOperationModeChanged = false;
            this->m_NotifiesPerformanceModeChanged = false;

            if (!IsProdMode())
            {
                NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AddLayerToApplicationForDebugLayerStack()");
                EnableIntegratedDisplayLayerToDebugLayerStackForApplication(m_LayerHandle);
            }
        }

        // TORIAEZU: オーバレイの判定をこれで行うのはよくない
        auto recodingVolume = GetApplet()->GetInitialWindowProperty().isOverlayAppletWindow ? 0.f : 1.f;
        NN_ABORT_UNLESS_RESULT_SUCCESS(audio::SetAudioOutsProcessRecordVolume(m_AppletResourceUserId, recodingVolume, TimeSpan::FromMilliSeconds(10)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(audio::SetAudioRenderersProcessRecordVolume(m_AppletResourceUserId, recodingVolume, TimeSpan::FromMilliSeconds(10)));

        ActivateGamePlayRecording();
    }

    ~ProcessGateway() NN_NOEXCEPT
    {
        DeactivateGamePlayRecording();

        auto info = GetApplet()->GetAppletIdentityInfo();
        if (info.IsApplication())
        {
            NN_AM_SERVICE_STUCK_CHECKED(capsrv_UnregisterAruid, 60, capsrv::UnregisterAppletResourceUserId(m_AppletResourceUserId, info.applicationId));
        }
        BtmUnregisterAppletResourceUserId(m_AppletResourceUserId);
        NN_AM_SERVICE_STUCK_CHECKED(hidbus_UnregisterAruid, 60, hidbus::system::UnregisterAppletResourceUserId(m_AppletResourceUserId));
        NN_AM_SERVICE_STUCK_CHECKED(irsensor_UnregisterAruid, 60, irsensor::system::UnregisterAppletResourceUserId(m_AppletResourceUserId));
        NN_AM_SERVICE_STUCK_CHECKED(hid_UnregisterAruid, 60, hid::system::UnregisterAppletResourceUserId(m_AppletResourceUserId));
        UnregisterAruidForAudioControl(m_AppletResourceUserId);
        NN_AM_SERVICE_STUCK_CHECKED(audio_UnregisterAruid, 60, audio::UnregisterAppletResourceUserId(m_AppletResourceUserId));
        UnregisterFromGpuResourceControl(GetApplet()->GetGpuResourceGroupId(), m_AppletResourceUserId);
        if (m_ProxyOption.forOe)
        {
            UnregisterToGpuErrorControl(m_AppletResourceUserId);
        }

        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "~ProcessGateway(): Unregistered ARUID = %lld", m_AppletResourceUserId);

        ncm::ProgramId id;
        if( GetApplet()->MakeProgramIdFromAppletIdentityInfo(&id, info).IsSuccess() )
        {
            UnregisterRunningApplet(id);
        }

        if (m_ProxyOption.forOe && !IsProdMode())
        {
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "RemoveLayerFromApplicationForDebugLayerStack()");
            am::service::DisableIntegratedDisplayLayerToDebugLayerStackForApplication(m_LayerHandle);
        }
        DestroyDisplayLayerImpl();
        if (m_ControllerFirmwareUpdateSectionOwner)
        {
            this->SetControllerFirmwareUpdateSectionImpl(false);
        }
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        if (m_SleepLockCount > 0)
        {
            m_SleepLockCount = 0;
            ReleaseSleepLockImpl();
        }
        if (m_IsSleepLockedEventValid)
        {
            os::DestroySystemEvent(&m_SleepLockedEvent);
        }
    }

    applet::AppletResourceUserId GetAppletResourceUserId() const NN_NOEXCEPT
    {
        return m_AppletResourceUserId;
    }

    sf::NativeHandle GetPopEventHandle() NN_NOEXCEPT
    {
        return sf::NativeHandle(m_Event.GetReadableHandle(), false);
    }

    bool TryPopMessage(am::AppletMessage* pOut) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        if (!TryPopMessageImpl(pOut))
        {
            return false;
        }
        UpdateMessageSignaled();
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "PopMessage -> %d", static_cast<int>(pOut->message[0]));
        return true;
    }

    void PushMessage(const am::AppletMessage& message) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        m_Queue.push_back(message);
        UpdateMessageSignaled();
    }

    void RequestExit() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_Status.exitRequested = true;
        {
            // TODO: UpdateStatus() 内へ移動
            // LA 起動許可イベントをシグナル状態へ
            m_LibraryAppletLaunchableEvent.Signal();
        }
        UpdateStatus(true);
    }

    void SetNextVolumeChangeFadeTime(float volume, TimeSpan fadeTime) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_AudioVolumeMutex)> lk(m_AudioVolumeMutex);
        m_NextVolume    = volume;
        m_NextFadeTime  = fadeTime;
    }

    IntegratedApplet* GetApplet() const NN_NOEXCEPT
    {
        return m_pApplet.get();
    }

    // sleep
    Result RequestToAcquireSleepLock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        ++m_SleepLockCount;
        if (m_SleepLockCount == 1)
        {
            RequestToAcquireSleepLockImpl();
        }
        NN_RESULT_SUCCESS;
    }

    Result ReleaseSleepLock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        NN_RESULT_THROW_UNLESS(m_SleepLockCount > 0, am::ResultInvalidCall()); // TODO:(Result) 詳細な Result
        --m_SleepLockCount;
        if (m_SleepLockCount == 0)
        {
            ReleaseSleepLockImpl();
        }
        NN_RESULT_SUCCESS;
    }

    Result ReleaseSleepLockTransiently() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        NN_RESULT_THROW_UNLESS(m_ApprovingClockForSleep > 0, am::ResultInvalidCall()); // TODO:(Result) 詳細な Result
        ReleaseSleepLockTransientlyImpl();
        NN_RESULT_SUCCESS;
    }

    Result GetAcquiredSleepLockEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        if (!m_IsSleepLockedEventValid)
        {
            NN_RESULT_DO(os::CreateSystemEvent(&m_SleepLockedEvent, os::EventClearMode_ManualClear, true));
            if (m_SleepLocked)
            {
                os::SignalSystemEvent(&m_SleepLockedEvent);
            }
            this->m_IsSleepLockedEventValid = true;
        }
        *pOut = sf::NativeHandle(os::GetReadableHandleOfSystemEvent(&m_SleepLockedEvent), false);
        NN_RESULT_SUCCESS;
    }

    FocusState GetCurrentFocusStateImpl() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_FocusStateRecognized = m_FocusState;
        return m_FocusState;
    }

    void SetFocusStateSet(FocusStateSet focusStateSet) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_Status.focusStateSet = focusStateSet;
        UpdateStatus(false);
    }

    void SetNotifiesFocusStateChanged(bool notifiesFocusStateChanged) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_NotifiesFocusStateChanged = notifiesFocusStateChanged;
        UpdateStatus(true);
    }

    void SetFocusHandlingModeForOld(bool notifiesFocusMessage, bool notifiesBgMessage, bool backgroundAutoSuspendable) NN_NOEXCEPT
    {
        NN_UNUSED(notifiesBgMessage);
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        switch (m_Status.focusStateSet)
        {
            case FocusStateSet::InFocusOnly:
            case FocusStateSet::InFocusAndOutOfFocus:
            {
                if (backgroundAutoSuspendable == false)
                {
                    m_Status.focusStateSet = FocusStateSet::All;
                }
                break;
            }
            case FocusStateSet::All:
            {
                if (backgroundAutoSuspendable == true)
                {
                    m_Status.focusStateSet = FocusStateSet::InFocusAndOutOfFocus;
                }
                break;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
        m_NotifiesFocusStateChanged = notifiesFocusMessage;
        UpdateStatus(true);
    }

    void SetOutOfFocusSuspendingEnabledForOld(bool outFocusSuspending) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        switch (m_Status.focusStateSet)
        {
            case FocusStateSet::InFocusOnly:
            {
                if (outFocusSuspending == false)
                {
                    m_Status.focusStateSet = FocusStateSet::InFocusAndOutOfFocus;
                }
                break;
            }
            case FocusStateSet::InFocusAndOutOfFocus:
            case FocusStateSet::All:
            {
                if (outFocusSuspending == true)
                {
                    m_Status.focusStateSet = FocusStateSet::InFocusOnly;
                }
                break;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
        UpdateStatus(false);
    }

    void SetAlbumImageOrientationImpl(album::ImageOrientation orientation) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_AlbumImageOrientation = orientation;
    }

    void SetNotifiesOperationModeChanged(bool notifiesOperationModeChanged) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_NotifiesOperationModeChanged = notifiesOperationModeChanged;
        UpdateMessageSignaled();
    }

    void SetNotifiesPerformanceModeChanged(bool notifiesPerformanceModeChanged) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_NotifiesPerformanceModeChanged = notifiesPerformanceModeChanged;
        UpdateMessageSignaled();
    }

    void SetNotifiesRestartFromSuspend(bool notifiesRestartFromSuspend) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_NotifiesRestartFromSuspend = notifiesRestartFromSuspend;
        UpdateMessageSignaled();
    }

private:

    std::shared_ptr<IntegratedApplet> m_pApplet;
    window::Window* m_pWindow;
    applet::AppletResourceUserId m_AppletResourceUserId;
    sf::NativeHandle m_ProcessHandle;
    ProxyOption m_ProxyOption;

    ShimLibraryMode GetShimLibraryMode() const NN_NOEXCEPT
    {
        return this->m_ProxyOption.forOe
            ? ShimLibraryMode::Oe
            : ShimLibraryMode::Ae
        ;
    }

    mutable os::Mutex m_Mutex{false};
    os::SystemEvent m_Event{os::EventClearMode_ManualClear, true};
    os::SystemEvent m_LibraryAppletLaunchableEvent{os::EventClearMode_ManualClear, true};

    bool m_IsSignaled{false};
    std::list<am::AppletMessage, StaticAllocator<am::AppletMessage>> m_Queue;

    uint64_t m_SleepLockCount{0};

    struct Status
    {
        bool exitRequested = false;
        bool canUseApplicationCoreAsRequested = true;
        bool canRun = true;
        FocusStateSet focusStateSet = FocusStateSet::All;
        WindowFocusState windowFocusState = WindowFocusState::Background;
        Wakefulness wakefulness = Wakefulness::Awake;
        bool isFatal = false;

    private:

        enum class BaseFocusState
        {
            InFocus,
            OutOfFocus,
            Background,
            BackgroundOutOfFocus,
        };

        BaseFocusState GetBaseFocusState() const NN_NOEXCEPT
        {
            switch (this->wakefulness)
            {
                case Wakefulness::Awake:
                {
                    switch (this->windowFocusState)
                    {
                        case WindowFocusState::InFocus: return BaseFocusState::InFocus;
                        case WindowFocusState::OutOfFocus: return BaseFocusState::OutOfFocus;
                        case WindowFocusState::BackgroundOutOfFocus: return BaseFocusState::BackgroundOutOfFocus;
                        default: return BaseFocusState::Background;
                    }
                }
                default: return BaseFocusState::Background;
            }
        }

        bool IsProcessActiveOnNormal() const NN_NOEXCEPT
        {
            switch (this->GetBaseFocusState())
            {
                case BaseFocusState::InFocus: return true;
                case BaseFocusState::OutOfFocus:
                {
                    switch (this->focusStateSet)
                    {
                        case FocusStateSet::InFocusOnly: return false;
                        case FocusStateSet::InFocusAndOutOfFocus: return true;
                        case FocusStateSet::All: return true;
                        default: NN_UNEXPECTED_DEFAULT;
                    }
                }
                case BaseFocusState::Background:
                case BaseFocusState::BackgroundOutOfFocus:
                {
                    switch (this->focusStateSet)
                    {
                        case FocusStateSet::InFocusOnly: return false;
                        case FocusStateSet::InFocusAndOutOfFocus: return false;
                        case FocusStateSet::All: return true;
                        default: NN_UNEXPECTED_DEFAULT;
                    }
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }

    public:

        FocusState GetFocusState(ShimLibraryMode shimLibraryMode) const NN_NOEXCEPT
        {
            auto baseFocusState = this->GetBaseFocusState();
            switch (baseFocusState)
            {
                case BaseFocusState::InFocus: return FocusState_InFocus;
                case BaseFocusState::OutOfFocus:
                {
                    switch (this->focusStateSet)
                    {
                        case FocusStateSet::InFocusOnly: return FocusState_InFocus;
                        case FocusStateSet::InFocusAndOutOfFocus: return FocusState_OutOfFocus;
                        case FocusStateSet::All: return FocusState_OutOfFocus;
                        default: NN_UNEXPECTED_DEFAULT;
                    }
                }
                case BaseFocusState::Background:
                case BaseFocusState::BackgroundOutOfFocus:
                {
                    switch (this->focusStateSet)
                    {
                        case FocusStateSet::InFocusOnly: return FocusState_InFocus;
                        case FocusStateSet::InFocusAndOutOfFocus:
                        {
                            if (baseFocusState == BaseFocusState::BackgroundOutOfFocus)
                            {
                                return FocusState_OutOfFocus;
                            }
                            else
                            {
                                return FocusState_InFocus;
                            }
                        }
                        case FocusStateSet::All:
                        {
                            switch (shimLibraryMode)
                            {
                                case ShimLibraryMode::Ae: return FocusState_OutOfFocus;
                                case ShimLibraryMode::Oe: return FocusState_Background;
                                default: NN_UNEXPECTED_DEFAULT;
                            }
                        }
                        default: NN_UNEXPECTED_DEFAULT;
                    }
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }

        bool IsProcessActive() const NN_NOEXCEPT
        {
            if (isFatal)
            {
                return false;
            }
            else if (!canRun)
            {
                return false;
            }
            else
            {
                switch (this->wakefulness)
                {
                    case Wakefulness::Sleep: return false;
                    case Wakefulness::Waking: return exitRequested;
                    case Wakefulness::Awake: return exitRequested || (canUseApplicationCoreAsRequested && IsProcessActiveOnNormal());
                    default: NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    };
    Status m_Status{};
    bool m_IsProcessActive = true;

    void UpdateStatus(bool forceSignal) NN_NOEXCEPT;

    // focus 制御
    FocusState m_FocusState{FocusState_Background};
    FocusState m_FocusStateRecognized{FocusState_Background};

    // 通知
    bool m_NotifiesFocusStateChanged{true};
    bool m_NotifiesOperationModeChanged{true};
    bool m_NotifiesPerformanceModeChanged{true};
    bool m_NotifiesRestartFromSuspend{false};

    // window 同期
    bool m_ExitRequestedRecognized{false};
    bool m_ReleaseSleepLockRequested{false};
    bool m_ReleaseSleepLockRequestedRecognized{false};
    bool m_OperationModeChanged{false};
    bool m_PerformanceModeChanged{false};
    bool m_VrModeChanged{false};
    bool m_VrModeDisabledBySystem{false};
    bool m_LcdBackLightOnNotification{false};
    bool m_SdCardRemoved{false};
    bool m_SleepRequiredByHighTemperature{false};
    bool m_SleepRequiredByByLowBattery{false};
    bool m_AutoPowerDown{false};
    bool m_RestartFromSuspend{false};
    bool m_FocusStateChanged{false};
    bool m_NotifiesVrModeCurtainRequired{false};

    bool m_ControllerFirmwareUpdateSectionOwner{false};
    bool m_NotifiesControllerFirmwareUpdateSectionChanged{false};

    uint64_t m_ApprovingClockForSleep{0}; // 0 無効

    pm::BootMode m_BootMode;

    window::WindowManager* GetWindowManager() const NN_NOEXCEPT
    {
        return m_pApplet->GetAppletSystem()->GetWindowManager();
    }

    bool ShouldBeMessageSignaled() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());
        auto needsFocusNotify = [&]() -> bool
        {
            switch (GetShimLibraryMode())
            {
                case ShimLibraryMode::Oe: return m_NotifiesFocusStateChanged && m_FocusStateChanged;
                case ShimLibraryMode::Ae: return m_NotifiesFocusStateChanged && m_FocusState != m_FocusStateRecognized;
                default: NN_UNEXPECTED_DEFAULT;
            }
        }();
        return false
            || !m_Queue.empty()
            || needsFocusNotify
            || m_RestartFromSuspend
            || m_Status.exitRequested != m_ExitRequestedRecognized
            || m_ReleaseSleepLockRequested != m_ReleaseSleepLockRequestedRecognized
            || m_OperationModeChanged
            || m_PerformanceModeChanged
            || m_VrModeChanged
            || m_VrModeDisabledBySystem
            || m_NotifiesVrModeCurtainRequired
            || m_NotifiesControllerFirmwareUpdateSectionChanged
            || m_LcdBackLightOnNotification
            || m_SdCardRemoved
            || m_SleepRequiredByHighTemperature
            || m_SleepRequiredByByLowBattery
            || m_AutoPowerDown
            || m_ClockForRequestToDisplay != m_ClockForRequestToDisplayRecognized
        ;
    }

    void UpdateMessageSignaled() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());
        if (ShouldBeMessageSignaled() != m_IsSignaled)
        {
            if (m_IsSignaled)
            {
                m_Event.Clear();
                this->m_IsSignaled = false;
            }
            else
            {
                m_Event.Signal();
                this->m_IsSignaled = true;
            }
        }
    }

    static am::AppletMessage MakeMessage(applet::Message message)
    {
        am::AppletMessage ret = {};
        ret.message[0] = message;
        return ret;
    }

    static_assert(static_cast<Bit32>(FocusState_InFocus) ==
                  static_cast<Bit32>(applet::Message_InFocus), "");
    static_assert(static_cast<Bit32>(FocusState_OutOfFocus) ==
                  static_cast<Bit32>(applet::Message_OutOfFocus), "");
    bool TryPopMessageImpl(am::AppletMessage* pOut) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());
        if (m_RestartFromSuspend)
        {
            this->m_RestartFromSuspend = false;
            *pOut = MakeMessage(applet::Message_RestartFromSuspend);
            return true;
        }
        if (m_NotifiesVrModeCurtainRequired)
        {
            this->m_NotifiesVrModeCurtainRequired = false;
            *pOut = MakeMessage(applet::Message_VrModeCurtainRequired);
            return true;
        }
        if (m_LcdBackLightOnNotification)
        {
            this->m_LcdBackLightOnNotification = false;
            *pOut = MakeMessage(applet::Message_LcdBacklightSwitchedOn);
            return true;
        }
        // SIGLO-62988:
        // Message_Exit は Message_FocusStateChanged よりも優先的に返すべき
        if (m_Status.exitRequested != m_ExitRequestedRecognized)
        {
            NN_SDK_ASSERT(m_Status.exitRequested);
            this->m_ExitRequestedRecognized = true;
            *pOut = MakeMessage(applet::Message_Exit);
            return true;
        }
        if (m_NotifiesFocusStateChanged)
        {
            switch (GetShimLibraryMode())
            {
                case ShimLibraryMode::Oe:
                {
                    // アプリは nn::oe::GetCurrentFocusState() を呼んだ時に代入
                    if (m_FocusStateChanged)
                    {
                        *pOut = MakeMessage(applet::Message_FocusStateChanged);
                        this->m_FocusStateChanged = false;
                        return true;
                    }
                    break;
                }
                case ShimLibraryMode::Ae:
                {
                    if (m_FocusState != m_FocusStateRecognized)
                    {
                        auto message = [&]() -> applet::Message
                        {
                            switch (m_FocusState)
                            {
                                case FocusState_InFocus: return applet::Message_InFocus;
                                case FocusState_OutOfFocus: return applet::Message_OutOfFocus;
                                default: NN_UNEXPECTED_DEFAULT;
                            }
                        }();
                        *pOut = MakeMessage(message);
                        this->m_FocusStateRecognized = m_FocusState;
                        return true;
                    }
                    break;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
        if (m_ReleaseSleepLockRequested && !m_ReleaseSleepLockRequestedRecognized)
        {
            this->m_ReleaseSleepLockRequestedRecognized = m_ReleaseSleepLockRequested;
            *pOut = MakeMessage(applet::Message_RequestToReleaseSleepLock);
            return true;
        }
        if (m_ClockForRequestToDisplayRecognized != m_ClockForRequestToDisplay)
        {
            this->m_ClockForRequestToDisplayRecognized = m_ClockForRequestToDisplay;
            *pOut = MakeMessage(applet::Message_RequestToDisplay);
            return true;
        }
        if (m_OperationModeChanged)
        {
            this->m_OperationModeChanged = false;
            *pOut = MakeMessage(applet::Message_OperationModeChanged);
            return true;
        }
        if (m_PerformanceModeChanged)
        {
            this->m_PerformanceModeChanged = false;
            *pOut = MakeMessage(applet::Message_PerformanceModeChanged);
            return true;
        }
        if (m_VrModeChanged)
        {
            this->m_VrModeChanged = false;
            *pOut = MakeMessage(applet::Message_VrModeChanged);
            return true;
        }
        if (m_VrModeDisabledBySystem)
        {
            this->m_VrModeDisabledBySystem = false;
            *pOut = MakeMessage(applet::Message_VrModeDisabledBySystem);
            return true;
        }
        if (m_NotifiesControllerFirmwareUpdateSectionChanged)
        {
            this->m_NotifiesControllerFirmwareUpdateSectionChanged = false;
            *pOut = MakeMessage(applet::Message_ControllerFirmwareUpdateSectionChanged);
            return true;
        }
        if (m_SdCardRemoved)
        {
            this->m_SdCardRemoved = false;
            *pOut = MakeMessage(applet::Message_SdCardRemoved);
            return true;
        }
        if (m_SleepRequiredByHighTemperature)
        {
            this->m_SleepRequiredByHighTemperature = false;
            *pOut = MakeMessage(applet::Message_SleepRequiredByHighTemperature);
            return true;
        }
        if (m_SleepRequiredByByLowBattery)
        {
            this->m_SleepRequiredByByLowBattery = false;
            *pOut = MakeMessage(applet::Message_SleepRequiredByLowBattery);
            return true;
        }
        if (m_AutoPowerDown)
        {
            this->m_AutoPowerDown = false;
            *pOut = MakeMessage(applet::Message_AutoPowerDown);
            return true;
        }
        if (!m_Queue.empty())
        {
            *pOut = m_Queue.front();
            m_Queue.pop_front();
            return true;
        }
        return false;
    } // NOLINT(impl/function_size)

    void RequestToAcquireSleepLockImpl() NN_NOEXCEPT;
    void ReleaseSleepLockImpl() NN_NOEXCEPT;
    void ReleaseSleepLockTransientlyImpl() NN_NOEXCEPT;

    os::NativeHandle GetProcessHandle() const NN_NOEXCEPT
    {
        return m_ProcessHandle.GetOsHandle();
    }

public:

    // overrides IWindowTransiter
    virtual void SetEffectiveVolume(bool available, float volume) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetForeground(WindowFocusState state, bool canUseApplicationCoreAsRequested) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetCanRun(bool canRun) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnHomeButtonShortPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnHomeButtonLongPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnPowerButtonShortPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnPowerButtonMiddlePressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnPowerButtonLongPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnCaptureButtonShortPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnCaptureButtonLongPressed() NN_NOEXCEPT NN_OVERRIDE;
    virtual void RequestApprovalForSleep(uint64_t clock) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnSleepRequiredByHighTemperature() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnSleepRequiredByLowBattery() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnAutoPowerDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnCecSystemStandbyReceived() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetCecStateAndNotifyChanged(bool) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnSleepLockAcquired() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnSleepLockReleased() NN_NOEXCEPT NN_OVERRIDE;
    virtual void BeforeSleepSystem() NN_NOEXCEPT NN_OVERRIDE;
    virtual void AfterSleepSystem() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnFatal() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnProcessException(const os::ProcessId& processId) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetHandlingContextToIdle() NN_NOEXCEPT NN_OVERRIDE;
    virtual void RequestToSetHandlingContextToIdle() NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyForegroundToNfc() NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyHidInputRequirement(bool) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyHidNpadPolicyOwner() NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyIrSensorInputRequirement(bool) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyIrSensorPolicyOwner() NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyHidBusPolicyOwner() NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyVibrationOwner(bool) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifySixAxisSensorInputRequirement(bool) NN_NOEXCEPT NN_OVERRIDE;
    virtual void NotifyBleConnectionOwner() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnOperationModeChanged() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnPerformanceModeChanged() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnVrModeChanged(bool isVrMode, bool fromOe) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnSdCardRemoved() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnGamePlayRecordingStateChanged() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetGamePlayRecordingVisibility(bool isVisible) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetDisplayVisibility(bool isVisible, int32_t zOrder, bool staysDisplay) NN_NOEXCEPT NN_OVERRIDE;
    virtual void RequestToDisplay(uint64_t clock) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnGameCardStateChanged() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetScreenShotAppletIdentityInfoImpl(applet::AppletIdentityInfo info) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetControllerFirmwareUpdateSectionImpl(bool isInSection) NN_NOEXCEPT;
    virtual void OnLcdBacklightChanged(bool isLcdBacklightOff) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SwitchLcdBacklightOnByForce() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnWirelessPriorityModeChanged(WirelessPriorityMode mode) NN_NOEXCEPT NN_OVERRIDE;

private:

    void SetEffectiveVolumeImpl(float volume, TimeSpan fadeTime) NN_NOEXCEPT;
    ncm::ProgramId m_ScreenShotProgramId;
    album::ImageOrientation m_AlbumImageOrientation{album::ImageOrientation_None};

    os::Mutex m_AudioVolumeMutex{false};
    bool m_IsAudioAvailable{false};
    float m_AudioEffectiveVolume{0.f};
    float m_NextVolume{-1.f};
    TimeSpan m_NextFadeTime{0};

    bool m_SleepLocked{false};
    bool m_IsSleepLockedEventValid{false};
    os::SystemEventType m_SleepLockedEvent;

    void SetAcquiredSleepLockImpl(bool locked) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
        NN_SDK_ASSERT(m_SleepLocked != locked);
        this->m_SleepLocked = locked;
        if (m_IsSleepLockedEventValid)
        {
            if (locked)
            {
                os::SignalSystemEvent(&m_SleepLockedEvent);
            }
            else
            {
                os::ClearSystemEvent(&m_SleepLockedEvent);
            }
        }
    }

    void SuspendImpl(bool alsoAudio) NN_NOEXCEPT;
    void ResumeImpl() NN_NOEXCEPT;

private:

    os::Mutex m_LayerMutex{false};
    IntegratedDisplayLayerHandle m_LayerHandle{-1};
    bool m_IsLayerVisible{false};
    int32_t m_LayerZOrder{0};

    os::SdkRecursiveMutex m_GameRecordingMutex;
    uintptr_t m_GameRecordingAutoAllocatedWorkBuffer = 0;
    bool IsGameRecordingWorkBufferAutoAllocated() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_GameRecordingMutex.IsLockedByCurrentThread());
        return m_GameRecordingAutoAllocatedWorkBuffer != 0;
    }

    GamePlayRecordingState m_GamePlayRecordingState{GamePlayRecordingState::Stopped};
    bool m_GamePlayRecordingVisibility{false};
    bool m_IsGamePlayRecordingPerforming{false};
    bool m_IsGamePlayRecordingForceDisabledByFwdbgSettings{false};
    bool m_IsGamePlayRecordingDebugMode = service::IsContinuousRecordingDebugMode();
    bool m_SavesMovieOnException = service::SavesMovieAutomaticallyOnException();
    sf::SharedPointer<grcsrv::IContinuousRecorder> m_ContinuousRecorder;
    sf::SharedPointer<grcsrv::IContinuousRecorder> GetContinuousRecorder() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        return m_ContinuousRecorder;
    }
    bool CanGamePlayRecordingImpl() NN_NOEXCEPT;

    bool m_HandlesRequestToDisplay{false};
    uint64_t m_ClockForRequestToDisplay{0};
    uint64_t m_ClockForRequestToDisplayRecognized{0};

    void UpdateDisplayLayerVisibility() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_LayerMutex.IsLockedByCurrentThread());
        NN_SDK_ASSERT(m_LayerHandle >= 0);
        am::service::SetIntegratedDisplayLayerVisibility(m_LayerHandle, m_IsLayerVisible, m_LayerZOrder, GetApplet()->IsCaptureTarget());
    }

    void ApproveToDisplayImpl(uint64_t clockForRequestToDisplay) NN_NOEXCEPT;
    void OnRequestToDisplay(uint64_t clock) NN_NOEXCEPT;

public:
    Result InitializeGamePlayRecordingImpl(sf::NativeHandle transferMemoryHandle, uint64_t transferMemorySize, uint64_t extraRecordBufferSize) NN_NOEXCEPT;
    Result SetGamePlayRecordingStateImpl(GamePlayRecordingState state) NN_NOEXCEPT;
    Result AcquireSharedCaptureBuffer(bool* pOutIsScreenShotPermitted, int* pOutBufferIndex, CaptureBufferIndex index) NN_NOEXCEPT;
    Result ReleaseSharedCaptureBuffer(CaptureBufferIndex index) NN_NOEXCEPT;
    Result TakeScreenShotOfOwnLayerImpl(int index, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT;
    Result StartContinuousRecordingFlushForDebug(sf::Out<sf::NativeHandle> pOut, TimeSpan maxTime) NN_NOEXCEPT;

private:
    static bool GetScreenShotPermissionImpl(void* p) NN_NOEXCEPT
    {
        auto pSelf = reinterpret_cast<ProcessGateway*>(p);
        return pSelf->GetApplet()->IsScreenShotEnabled();
    }

public:
    Result CreateDisplayLayerImpl() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_LayerHandle < 0, am::ResultInvalidCall());
        auto h = CreateIntegratedDisplayLayer();
        NN_ABORT_UNLESS_GREATER_EQUAL(h, 0);

        auto fgMode = GetApplet()->GetForegroundMode();
        // レイヤを使用するアプレットの種類を指定（この段階ではパラメータが確定しないので戻り値は false ）
        auto info = GetApplet()->GetAppletIdentityInfo();
        if (info.IsApplication() || info.IsSystemApplication())
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerUserApplication(h, m_AppletResourceUserId));
        }
        else if (info.appletId == applet::AppletId_OverlayApplet)
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerUserOverlay(h, m_AppletResourceUserId));
        }
        else if(fgMode == window::ForegroundMode::All)
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerUserFullscreenApplet(h, m_AppletResourceUserId));
        }
        else if(fgMode == window::ForegroundMode::Partial)
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerUserPartialApplet(h, m_AppletResourceUserId));
        }
        else // window::ForegroundMode::None
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerUserNoDisplay(h, m_AppletResourceUserId));
        }

        // 出画先を指定（この段階ではパラメータが確定しないので戻り値は false ）
        if (GetApplet()->IsDestinationLayerIndirect())
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerDestinationIndirect(h, GetApplet()->GetIndirectLayerHandle()));
        }
        else
        {
            NN_ABORT_UNLESS_EQUAL(false, SpecifyIntegratedDisplayLayerDestinationDisplay(h));
        }

        // スクリーンショット可否フラグの取得関数を設定
        am::service::SetIntegratedDisplayLayerScreenShotPremissionGetterFunction(h, GetScreenShotPermissionImpl, this);

        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        this->m_LayerHandle = h;
        // NOTE:
        //   パラメータが決まっていなくても Visibility は先に設定できる。
        //   設定しておいた Visibiliry は全パラメータが確定した段階で反映される。
        UpdateDisplayLayerVisibility();
        NN_RESULT_SUCCESS;
    }

    void DestroyDisplayLayerImpl() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        if (m_LayerHandle >= 0)
        {
            am::service::DestroyIntegratedDisplayLayer(m_LayerHandle);
            m_LayerHandle = -1;
        }
    }

    vi::LayerId SpecifyLayerBufferOwnerIsAppletAndGetDisplayLayerId() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        if(!IsIntegratedDisplayLayerBufferOwnerSpecified(m_LayerHandle))
        {
            NN_ABORT_UNLESS_EQUAL(true, am::service::SpecifyIntegratedDisplayLayerBufferOwnerApplet(m_LayerHandle));
        }
        return am::service::GetIntegratedDisplayLayerLayerId(m_LayerHandle);
    }

    vi::IndirectProducerHandleType SpecifyLayerBufferOwnerIsAppletAndGetIndirectProducerHandle() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        if(!IsIntegratedDisplayLayerBufferOwnerSpecified(m_LayerHandle))
        {
            NN_ABORT_UNLESS_EQUAL(true, am::service::SpecifyIntegratedDisplayLayerBufferOwnerApplet(m_LayerHandle));
        }
        return am::service::GetIntegratedDisplayLayerIndirectProducerHandle(m_LayerHandle);
    }

    bool IsSystemBufferSharingEnabled() NN_NOEXCEPT
    {
        return am::service::IsSystemBufferSharingEnabled();
    }

    Result SpecifyLayerBufferOwnerIsSystemSharedAndGetSharedLayerHandle(nn::vi::fbshare::SharedBufferHandle* pOutBufferHandle, nn::vi::fbshare::SharedLayerHandle* pOutLayerHandle) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        NN_RESULT_THROW_UNLESS(am::service::IsSystemBufferSharingEnabled(), ResultNotSupported());
        if(!IsIntegratedDisplayLayerBufferOwnerSpecified(m_LayerHandle))
        {
            NN_ABORT_UNLESS_EQUAL(true, am::service::SpecifyIntegratedDisplayLayerBufferOwnerSystemShared(m_LayerHandle));
        }
        *pOutBufferHandle = am::service::AcquireIntegratedDisplayLayerSystemSharedBufferHandle(m_LayerHandle);
        *pOutLayerHandle = am::service::GetIntegratedDisplayLayerSharedLayerHandle(m_LayerHandle);
        NN_RESULT_SUCCESS;
    }

    Result GetSharedBufferHandle(nn::vi::fbshare::SharedBufferHandle* pOutBufferHandle) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        NN_RESULT_THROW_UNLESS(am::service::IsSystemBufferSharingEnabled(), ResultNotSupported());
        *pOutBufferHandle = am::service::AcquireIntegratedDisplayLayerSystemSharedBufferHandle(m_LayerHandle);
        NN_RESULT_SUCCESS;
    }

    void SetDisplayLayerVisibilityImpl(bool visible, int32_t zOrder) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        this->m_IsLayerVisible = visible;
        this->m_LayerZOrder = zOrder;
        if (m_LayerHandle >= 0)
        {
            UpdateDisplayLayerVisibility();
        }
    }

    void InvalidateDisplayLayer() NN_NOEXCEPT;

    os::SystemEvent* GetLibraryAppletLaunchableEventImpl() NN_NOEXCEPT
    {
        return &m_LibraryAppletLaunchableEvent;
    }

    void SetHandlesRequestToDisplay(bool handlesRequestToDisplay) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        this->m_HandlesRequestToDisplay = handlesRequestToDisplay;
        if (!handlesRequestToDisplay)
        {
            ApproveToDisplayImpl(m_ClockForRequestToDisplay);
        }
    }

    void ApproveToDisplay() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
        ApproveToDisplayImpl(m_ClockForRequestToDisplay);
    }

    bool IsSuspended() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        return !m_IsProcessActive;
    }

    void SetControllerFirmwareUpdateSectionChangedNotification() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        m_NotifiesControllerFirmwareUpdateSectionChanged = true;
        UpdateMessageSignaled();
    }

    void SetVrModeEnabledImpl(bool isEnabled) NN_NOEXCEPT
    {
        // 各アプレットへの通知は AppletSystem::SetVrModeEnabledImpl() 側で行う
        GetApplet()->GetAppletSystem()->SetVrModeEnabledImpl(isEnabled, this->m_ProxyOption.forOe);
    }

    bool SetLcdBacklighOffEnabledImpl(bool isLcdBacklightOff) NN_NOEXCEPT
    {
        auto updater = GetWindowManager()->BeginUpdate();
        bool prev = updater.RefWindowProperty(m_pWindow).requestsToSwitchLcdBacklightOff;
        updater.RefWindowProperty(m_pWindow).requestsToSwitchLcdBacklightOff = isLcdBacklightOff;
        return prev;
    }

    void SetVrModeCurtainRequiredNotification() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        m_NotifiesVrModeCurtainRequired = true;
        UpdateMessageSignaled();
    }

private:

    HomeButtonShortPressedHandleType m_HandlesHomeButtonShortPressed{HomeButtonShortPressedHandleType_EnqueueMessage};
    bool m_HandlesHomeButtonLongPressed{true};
    bool m_HandlesPowerButtonShortPressed{true};
    bool m_RequiresCaptureButtonShortPressedMessage{false};

public:

    void SetHandlesHomeButtonShortPressed(HomeButtonShortPressedHandleType type) NN_NOEXCEPT
    {
        this->m_HandlesHomeButtonShortPressed = type;
    }

    void SetHandlesHomeButtonLongPressed(bool handlesHomeButtonLongPressed) NN_NOEXCEPT
    {
        this->m_HandlesHomeButtonLongPressed = handlesHomeButtonLongPressed;
    }

    void SetHandlesPowerButtonShortPressed(bool handlesPowerButtonShortPressed) NN_NOEXCEPT
    {
        this->m_HandlesPowerButtonShortPressed = handlesPowerButtonShortPressed;
    }

    void SetRequiresCaptureButtonShortPressedMessageImpl(bool isRequired) NN_NOEXCEPT
    {
        this->m_RequiresCaptureButtonShortPressedMessage = isRequired;
    }

    Result SetWirelessPriorityModeImpl(applet::WirelessPriorityMode mode) NN_NOEXCEPT
    {
        switch (mode)
        {
            case applet::WirelessPriorityMode_None:
            {
                auto updater = GetWindowManager()->BeginUpdate();
                updater.RefWindowProperty(m_pWindow).pWirelessPriorityModeRequest = util::nullopt;
                break;
            }
            case applet::WirelessPriorityMode_OptimizedForBluetooth:
            case applet::WirelessPriorityMode_OptimizedForWlan:
            {
                auto updater = GetWindowManager()->BeginUpdate();
                updater.RefWindowProperty(m_pWindow).pWirelessPriorityModeRequest = static_cast<WirelessPriorityMode>(mode);
                break;
            }
            default:
            {
                NN_RESULT_THROW( ResultInvalidParameter() );
            }
        }
        NN_RESULT_SUCCESS;
    }

    // アプリ録画用ワークメモリを確保して grc に渡し、録画を開始する。
    void ActivateGamePlayRecording() NN_NOEXCEPT
    {
        if (!( true && m_ProxyOption.forOe
                    && !GetContinuousRecorder()
                    && GetApplet()->IsVideoCaptureWorkBufferAutoAllocation() ))
        {
            return;
        }

        auto extraRecordBufferSize = service::GetContinuousRecordingExtraMemorySize(GetApplet()->GetAppletIdentityInfo().applicationId.value);

        NN_RESULT_IGNORING_BLOCK
        {
            std::lock_guard<decltype(m_GameRecordingMutex)> lk(m_GameRecordingMutex);
            auto success = false;

            uintptr_t address;
            const auto size = oe::WorkBufferSizeForGamePlayRecording + extraRecordBufferSize;
            NN_RESULT_DO(GetApplet()->GetAppletSystem()->AllocateMemoryFromApplicationResource(&address, size));
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    GetApplet()->GetAppletSystem()->FreeMemoryToApplicationResource(address);
                }
            };

            // TransferMemory 化して grc に渡す
            os::TransferMemory transMem(reinterpret_cast<void*>(address), size, os::MemoryPermission_None);
            NN_RESULT_DO(InitializeGamePlayRecordingImpl(sf::NativeHandle(transMem.Detach(), true), size, extraRecordBufferSize));
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    m_ContinuousRecorder.Reset();
                }
            };

            // 録画開始
            NN_RESULT_DO(SetGamePlayRecordingStateImpl(GamePlayRecordingState::Recording));

            success = true;
            this->m_GameRecordingAutoAllocatedWorkBuffer = address;
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ActivateGamePlayRecording(): address=0x%p", address);
            NN_RESULT_SUCCESS;
        };
    }

    // 録画を停止して、アプリ録画用ワークメモリをアプリ用に返却する
    void DeactivateGamePlayRecording() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_GameRecordingMutex)> lk(m_GameRecordingMutex);

        if ( IsGameRecordingWorkBufferAutoAllocated() )
        {
            // 録画を停止、grc プロセスからメモリを返却
            m_ContinuousRecorder.Reset();
            GetApplet()->GetAppletSystem()->FreeMemoryToApplicationResource(m_GameRecordingAutoAllocatedWorkBuffer);
            this->m_GameRecordingAutoAllocatedWorkBuffer = 0;
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "DeactivateGamePlayRecording()");
        }
    }

    void SetTvPowerStateMatchingModeImpl(bool isHandlesCecSystemStandby, bool isDisablesOneTouchPlayCecNotification) NN_NOEXCEPT
    {
        auto updater = GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(m_pWindow).handlesCecSystemStandby = isHandlesCecSystemStandby;
        updater.RefWindowProperty(m_pWindow).disablesOneTouchPlayCecNotification = isDisablesOneTouchPlayCecNotification;
    }

};

IntegratedApplet::AppletSelfProxyImpl::AppletSelfProxyImpl(std::shared_ptr<IntegratedApplet> p, os::ProcessId processId) NN_NOEXCEPT
    : m_P(p)
    , m_CleanupLock(p->AcquireCleanupLock())
    , m_pProcessGateway(MakeShared<ProcessGateway>(p, applet::AppletResourceUserId{processId.value})) // TODO: os::ProcessId to ARUID
{
    // Window の活性化は、SetProcessHandle() 後に ActivateWindow() の中で行う。
}

IntegratedApplet::AppletSelfProxyImpl::~AppletSelfProxyImpl() NN_NOEXCEPT
{
    if (m_IsLastApplicationCaptureBufferOwner)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAppletProcessExitedLeavingCaptureBufferMapped());
    }
    if (m_IsLastForegroundCaptureBufferOwner)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAppletProcessExitedLeavingCaptureBufferMapped());
    }
    if (m_IsCallerAppletCaptureBufferOwner)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAppletProcessExitedLeavingCaptureBufferMapped());
    }
    if (m_FatalSectionCount > 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAppletProcessExitedInFatalSection());
    }

    m_P->ClearWindowTansiter();

    // さらに自身の削除も試みる
    auto pAppletSystem = GetApplet()->GetAppletSystem();
    this->m_P.reset();
    pAppletSystem->CleanApplets();
}

Result IntegratedApplet::AppletSelfProxyImpl::EnterFatalSection() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_FatalSectionCountMutex)> lk(m_FatalSectionCountMutex);
    ++this->m_FatalSectionCount;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::LeaveFatalSection()NN_NOEXCEPT
{
    std::unique_lock<decltype(m_FatalSectionCountMutex)> lk(m_FatalSectionCountMutex);
    NN_RESULT_THROW_UNLESS(m_FatalSectionCount > 0, ResultUnbalancedFatalSectionControl());
    --this->m_FatalSectionCount;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetAppletResourceUserId(sf::Out<am::AppletResourceUserId> pOut) NN_NOEXCEPT
{
    *pOut = m_pProcessGateway->GetAppletResourceUserId();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetAppletResourceUserIdOfCallerApplet(sf::Out<am::AppletResourceUserId> pOut) NN_NOEXCEPT
{
    auto pParent = GetApplet()->GetParent();
    NN_RESULT_THROW_UNLESS(pParent, ResultNoCallerApplet());

    auto pParentProxy = pParent->GetSelfProxy();
    NN_RESULT_THROW_UNLESS(pParentProxy, ResultCallerAppletIsWinding());

    return pParentProxy->GetAppletResourceUserId(pOut);
}

void IntegratedApplet::AppletSelfProxyImpl::SetProcessHandle(sf::NativeHandle processHandle) NN_NOEXCEPT
{
    m_pProcessGateway->SetProcessHandle(std::move(processHandle));
}

void IntegratedApplet::AppletSelfProxyImpl::SetHandlesHomeButtonShortPressed(HomeButtonShortPressedHandleType type) NN_NOEXCEPT
{
    m_pProcessGateway->SetHandlesHomeButtonShortPressed(type);
}

void IntegratedApplet::AppletSelfProxyImpl::SetHandlesHomeButtonLongPressed(bool handlesHomeButtonLongPressed) NN_NOEXCEPT
{
    m_pProcessGateway->SetHandlesHomeButtonLongPressed(handlesHomeButtonLongPressed);
}

void IntegratedApplet::AppletSelfProxyImpl::SetHandlesPowerButtonShortPressed(bool handlesPowerButtonShortPressed) NN_NOEXCEPT
{
    m_pProcessGateway->SetHandlesPowerButtonShortPressed(handlesPowerButtonShortPressed);
}

Result IntegratedApplet::AppletSelfProxyImpl::SetDesirableKeyboardLayout(Bit32 layout) NN_NOEXCEPT
{
    GetApplet()->SetDesirableKeyboardLayoutImpl(layout);
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::AppletSelfProxyImpl::ActivateWindow(const ProxyOption& proxyOption) NN_NOEXCEPT
{
    // 本関数は AppletProxyManager::GetAppletProxy() の中で、
    // CreateAppletProxy(), SetProcessHandle() の発行後に呼ばれる。
    // このタイミングで Window を活性化することで、これ以降の ProcessGateway
    // の中で常に有効な ProcessHandle が利用可能であることを保証する。
    // 将来的に、AppletSelfProxyImpl コンストラクタの引数を構造体渡しにすれば、
    // SetProcessHandle() と ActivateWindow() はコンストラクタにマージ可能。
    m_pProcessGateway->Activate(proxyOption);
    m_P->SetWindowTansiter(m_pProcessGateway.get());
    if (m_P->IsExitRequested())
    {
        m_pProcessGateway->RequestExit();
    }
}

void IntegratedApplet::AppletSelfProxyImpl::RequestExit() NN_NOEXCEPT
{
    m_pProcessGateway->RequestExit();
}

bool IntegratedApplet::AppletSelfProxyImpl::IsSuspended() NN_NOEXCEPT
{
    return m_pProcessGateway->IsSuspended();
}

Result IntegratedApplet::AppletSelfProxyImpl::CreateManagedDisplayLayer(sf::Out<Bit64> pOutLayerId) NN_NOEXCEPT
{
    *pOutLayerId = m_pProcessGateway->SpecifyLayerBufferOwnerIsAppletAndGetDisplayLayerId();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::IsSystemBufferSharingEnabled() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_pProcessGateway->GetApplet()->GetAppletIdentityInfo().IsApplication(), ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_pProcessGateway->IsSystemBufferSharingEnabled(), ResultNotSupported());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetSystemSharedLayerHandle(sf::Out<vi::fbshare::SharedBufferHandle> pOutBufferHandle, sf::Out<vi::fbshare::SharedLayerHandle> pOutLayerHandle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_pProcessGateway->GetApplet()->GetAppletIdentityInfo().IsApplication(), ResultNotSupported());
    nn::vi::fbshare::SharedBufferHandle hBuffer = {};
    nn::vi::fbshare::SharedLayerHandle hLayer = {};
    NN_RESULT_DO(m_pProcessGateway->SpecifyLayerBufferOwnerIsSystemSharedAndGetSharedLayerHandle(&hBuffer, &hLayer));
    pOutBufferHandle.Set(hBuffer);
    pOutLayerHandle.Set(hLayer);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetSystemSharedBufferHandle(sf::Out<vi::fbshare::SharedBufferHandle> pOutBufferHandle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_pProcessGateway->GetApplet()->GetAppletIdentityInfo().IsApplication(), ResultNotSupported());
    nn::vi::fbshare::SharedBufferHandle hBuffer = {};
    NN_RESULT_DO(m_pProcessGateway->GetSharedBufferHandle(&hBuffer));
    pOutBufferHandle.Set(hBuffer);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCurrentFocusState(sf::Out<Bit8> focusState) NN_NOEXCEPT
{
    *focusState = static_cast<Bit8>(m_pProcessGateway->GetCurrentFocusStateImpl());
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::AppletSelfProxyImpl::PushMessage(const AppletMessage& message) NN_NOEXCEPT
{
    m_pProcessGateway->PushMessage(message);
}

Result IntegratedApplet::AppletSelfProxyImpl::GetEventHandle(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    *pOut = m_pProcessGateway->GetPopEventHandle();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReceiveMessage(sf::Out<am::AppletMessage> pOut) NN_NOEXCEPT
{
    am::AppletMessage ret;
    auto success = m_pProcessGateway->TryPopMessage(&ret);
    NN_RESULT_THROW_UNLESS(success, ResultNoMessage());
    *pOut = ret;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::RequestToAcquireSleepLock() NN_NOEXCEPT
{
    return m_pProcessGateway->RequestToAcquireSleepLock();
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseSleepLock() NN_NOEXCEPT
{
    return m_pProcessGateway->ReleaseSleepLock();
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseSleepLockTransiently() NN_NOEXCEPT
{
    return m_pProcessGateway->ReleaseSleepLockTransiently();
}

Result IntegratedApplet::AppletSelfProxyImpl::GetAcquiredSleepLockEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    return m_pProcessGateway->GetAcquiredSleepLockEvent(pOut);
}


Result IntegratedApplet::AppletSelfProxyImpl::SetHandlesRequestToDisplay(bool handlesRequestToDisplay) NN_NOEXCEPT
{
    m_pProcessGateway->SetHandlesRequestToDisplay(handlesRequestToDisplay);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ApproveToDisplay() NN_NOEXCEPT
{
    m_pProcessGateway->ApproveToDisplay();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ClearCaptureBuffer(int index, bool isScreenShotPermitted, Bit32 clearColor) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke ClearCaptureBuffer(index=%d, color=0x%08x)", index, clearColor);
    return am::service::ClearCaptureBufferImage(static_cast<am::service::CaptureBufferIndex>(index), clearColor, isScreenShotPermitted);
}

Result IntegratedApplet::AppletSelfProxyImpl::ClearAppletTransitionBuffer(Bit32 clearColor) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke ClearAppletTransitionBuffer(color=0x%08x)", clearColor);
    am::service::ClearTransitionLayerImage(clearColor);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetTvPowerStateMatchingMode(int mode) NN_NOEXCEPT
{
    switch (mode)
    {
        case oe::TvPowerStateMatchingMode_Default:
        {
            // テレビ電源連動を本体設定に戻す
            m_pProcessGateway->SetTvPowerStateMatchingModeImpl(false, false);
            NN_RESULT_SUCCESS;
        }
        case oe::TvPowerStateMatchingMode_Disabled:
        {
            // テレビ電源連動を無効化する（spsm からの通知を横取り）
            m_pProcessGateway->SetTvPowerStateMatchingModeImpl(true, true);
            NN_RESULT_SUCCESS;
        }
        default:
        {
            NN_RESULT_THROW(ResultInvalidParameter());
        }
    }
}

Result IntegratedApplet::ProcessGateway::InitializeGamePlayRecordingImpl(sf::NativeHandle transferMemoryHandle, uint64_t transferMemorySize, uint64_t extraRecordBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS( m_ProxyOption.forOe, ResultInvalidCall() );
    NN_RESULT_THROW_UNLESS( GetApplet()->IsVideoCaptureAllowed(), ResultLackOfCapability() );
    {
        std::lock_guard<decltype(m_GameRecordingMutex)> lk(m_GameRecordingMutex);
        NN_RESULT_THROW_UNLESS( !IsGameRecordingWorkBufferAutoAllocated(), ResultGamePlayRecordingWorkMemoryAlreadyTransferred() );
    }

    // grc プロセスに transferMemoryHandle と transferMemorySize を渡す
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke InitializeGamePlayRecordingImpl(size=0x%08llx)", transferMemorySize);

    decltype(m_ContinuousRecorder) pContinuousRecorder;
    {
        grc::ContinuousRecordingParameter param = {};
        param.applicationId = GetApplet()->GetAppletIdentityInfo().applicationId;
        param.aruid = GetAppletResourceUserId();
        param.videoBitRate = GetDefaultContinuousRecordingVideoBitRate();
        if (m_IsGamePlayRecordingDebugMode)
        {
            param.maxTimeInMilliSeconds = static_cast<uint32_t>(GetContinuousRecordingMaxTimeOnDebugMode().GetMilliSeconds());
            param.minTimeInMilliSeconds = 0;
        }
        else
        {
            param.maxTimeInMilliSeconds = static_cast<uint32_t>(GetDefaultContinuousRecordingMaxTime().GetMilliSeconds());
            param.minTimeInMilliSeconds = static_cast<uint32_t>(GetDefaultContinuousRecordingMinTime().GetMilliSeconds());
        }
        param.frameCountBetweenIDR = GetDefaultContinuousRecordingFrameCountBetweenIdr();
        param.fps = GetDefaultContinuousRecordingFps();
        param.recordBufferSize = 26 * 1024 * 1024 + extraRecordBufferSize;
        param.forDebug = m_IsGamePlayRecordingDebugMode;

        NN_AM_SERVICE_SCOPED_STUCK_CHECK(grc_CreateContinuousRecorder, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(grc::CreateContinuousRecorder(&pContinuousRecorder, std::move(transferMemoryHandle), transferMemorySize, param));
    }
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_ContinuousRecorder = std::move(pContinuousRecorder);
    }

    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::ProcessGateway::SetGamePlayRecordingStateImpl(GamePlayRecordingState state) NN_NOEXCEPT
{
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_GamePlayRecordingState = state;
    }
    this->OnGamePlayRecordingStateChanged();
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::ProcessGateway::SetGamePlayRecordingVisibility(bool isVisible) NN_NOEXCEPT
{
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_GamePlayRecordingVisibility = isVisible && !m_IsGamePlayRecordingForceDisabledByFwdbgSettings;
    }
    this->OnGamePlayRecordingStateChanged();
}

void IntegratedApplet::ProcessGateway::SetEffectiveVolumeImpl(float volume, TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(audio_SetVolume, 60);
    NN_ABORT_UNLESS_RESULT_SUCCESS(audio::SetAudioInsProcessMasterVolume(m_AppletResourceUserId, volume, fadeTime));
    NN_ABORT_UNLESS_RESULT_SUCCESS(audio::SetAudioOutsProcessMasterVolume(m_AppletResourceUserId, volume, fadeTime));
    NN_ABORT_UNLESS_RESULT_SUCCESS(audio::SetAudioRenderersProcessMasterVolume(m_AppletResourceUserId, volume, fadeTime));
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetEffectiveVolume(%f, %d ms)", volume, static_cast<int>(fadeTime.GetMilliSeconds()));
}

void IntegratedApplet::ProcessGateway::SetEffectiveVolume(bool isAvailable, float volume) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_AudioVolumeMutex)> lk(m_AudioVolumeMutex);
    util::optional<float> targetVolume;

    const auto defaultFadeTime = TimeSpan::FromMilliSeconds(10); // 適切な時間
    const auto fadeTime = (volume == m_NextVolume) ? m_NextFadeTime : defaultFadeTime;
    m_NextVolume = -1.f;
    m_NextFadeTime = defaultFadeTime;

    if (!m_IsAudioAvailable && !isAvailable)
    {
        // nop
    }
    else if (m_IsAudioAvailable && !isAvailable)
    {
        // when available -> not available
        targetVolume = 0.f;
    }
    else if (!m_IsAudioAvailable && isAvailable)
    {
        // when not available -> available
        targetVolume = volume;
    }
    else
    {
        // volume 変更
        if (m_AudioEffectiveVolume == volume)
        {
            // nop
            return;
        }
        targetVolume = volume;
    }
    this->m_IsAudioAvailable = isAvailable;
    this->m_AudioEffectiveVolume = volume;
    if (targetVolume)
    {
        // NEW_AM: TODO: audio 実装
        SetEffectiveVolumeImpl(*targetVolume, fadeTime);
    }
}

void IntegratedApplet::SetMainAppletWindowVolume(float volume) NN_NOEXCEPT
{
    auto updater = BeginWindowUpdate();
    updater.RefWindowProperty(m_pWindow).isMainWindowVolumeRateValid = true;
    updater.RefWindowProperty(m_pWindow).mainWindowVolumeRate = volume;
}

void IntegratedApplet::SetNextVolumeChangeFadeTime(float volume, TimeSpan fadeTime) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    auto pSelfProxy = m_pSelfProxy.lock();
    lk.unlock();
    if (pSelfProxy)
    {
        pSelfProxy->SetNextVolumeChangeFadeTime(volume, fadeTime);
    }
}

void IntegratedApplet::SetTransparentVolumeRate(float rate) NN_NOEXCEPT
{
    auto updater = BeginWindowUpdate();
    updater.RefWindowProperty(m_pWindow).transparentVolumeRate = rate;
}

bool IntegratedApplet::IsScreenShotEnabled() NN_NOEXCEPT
{
    switch (this->m_ScreenShotPermission)
    {
    case applet::ScreenShotPermission_Permit:
            return true;

    case applet::ScreenShotPermission_Forbid:
            return false;

    case applet::ScreenShotPermission_Inherit:
            if (this == GetMainApplet())
            {
                // 自身が SA ならアプリの撮影禁止状態を継承する
                return GetAppletSystem()->IsApplicationScreenShotEnabled();
            }
            return GetParent()->IsScreenShotEnabled();

    default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void IntegratedApplet::SetAppletAttribute(const AppletAttribute* pAttribute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAttribute);
    auto updater = BeginWindowUpdate();
    if (pAttribute->isVrModeDualScreenSupported)
    {
        updater.RefWindowProperty(m_pWindow).isVrModeDualScreenSupported = true;
    }
}

Result IntegratedApplet::SetScreenShotPermissionImpl(applet::ScreenShotPermission permission) NN_NOEXCEPT
{
    m_ScreenShotPermission = permission;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetAlbumImageOrientation(int orientation) NN_NOEXCEPT
{
    auto imageOrientation = static_cast<album::ImageOrientation>(orientation);
    switch (imageOrientation)
    {
        case album::ImageOrientation_None:
        case album::ImageOrientation_Rotate90:
        case album::ImageOrientation_Rotate180:
        case album::ImageOrientation_Rotate270:
                break;

        default:
                NN_RESULT_THROW( ResultInvalidParameter() );
    }
    m_pProcessGateway->SetAlbumImageOrientationImpl(imageOrientation);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetOperationModeChangedNotification(bool needNotify) NN_NOEXCEPT
{
    m_pProcessGateway->SetNotifiesOperationModeChanged(needNotify);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetPerformanceModeChangedNotification(bool needNotify) NN_NOEXCEPT
{
    m_pProcessGateway->SetNotifiesPerformanceModeChanged(needNotify);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetNotifiesFocusStateChanged(bool notifiesFocusStateChanged) NN_NOEXCEPT
{
    m_pProcessGateway->SetNotifiesFocusStateChanged(notifiesFocusStateChanged);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetFocusHandlingMode(bool notifiesFocusMessage, bool notifiesBgMessage, bool backgroundAutoSuspendable) NN_NOEXCEPT
{
    m_pProcessGateway->SetFocusHandlingModeForOld(notifiesFocusMessage, notifiesBgMessage, backgroundAutoSuspendable);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetOutOfFocusSuspendingEnabled(bool outFocusSuspending) NN_NOEXCEPT
{
    m_pProcessGateway->SetOutOfFocusSuspendingEnabledForOld(outFocusSuspending);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetRestartMessageEnabled(bool enabled) NN_NOEXCEPT
{
    m_pProcessGateway->SetNotifiesRestartFromSuspend(enabled);
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::AppletSelfProxyImpl::SetNextVolumeChangeFadeTime(float volume, TimeSpan fadeTime) NN_NOEXCEPT
{
    m_pProcessGateway->SetNextVolumeChangeFadeTime(volume, fadeTime);
}

void IntegratedApplet::AppletSelfProxyImpl::InvalidateDisplayLayer() NN_NOEXCEPT
{
    m_pProcessGateway->InvalidateDisplayLayer();
}

Result IntegratedApplet::AppletSelfProxyImpl::GetIndirectLayerProducerHandle(sf::Out<vi::IndirectProducerHandleType> pOut) NN_NOEXCEPT
{
    auto h = m_pProcessGateway->SpecifyLayerBufferOwnerIsAppletAndGetIndirectProducerHandle();
    NN_RESULT_THROW_UNLESS(h, am::ResultInvalidCall());
    *pOut =  h;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::OverrideAutoSleepTimeAndDimmingTime(int autoSleepTimeInHandheld, int autoSleepTimeInConsole, int dimmingTimeInHandheld, int dimmingTimeInConsole) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OverrideAutoSleepTimeAndDimmingTime(%d,%d,%d,%d)", autoSleepTimeInHandheld, autoSleepTimeInConsole, dimmingTimeInHandheld, dimmingTimeInConsole);
    auto* p = GetApplet()->GetIdleHandlingContext();
    p->overrideAutoSleepTimeInHandheldInSeconds = autoSleepTimeInHandheld;
    p->overrideAutoSleepTimeInConsoleInSeconds  = autoSleepTimeInConsole;
    p->overrideDimmingTimeInHandheldInSeconds   = dimmingTimeInHandheld;
    p->overrideDimmingTimeInConsoleInSeconds    = dimmingTimeInConsole;

    m_pProcessGateway->RequestToSetHandlingContextToIdle();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetMediaPlaybackState(bool isInMediaPlayback) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetMediaPlaybackState(%s)", isInMediaPlayback ? "true" : "false");
    auto* p = GetApplet()->GetIdleHandlingContext();

    std::lock_guard<os::Mutex> lk(m_MediaPlaybackStateMutex);
    if (p->isInMediaPlayback != isInMediaPlayback)
    {
        p->isInMediaPlayback = isInMediaPlayback;
        if (!isInMediaPlayback)
        {
            p->timeMediaPlaybackFinished = nn::os::GetSystemTick();
        }
        m_pProcessGateway->RequestToSetHandlingContextToIdle();
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::IsGamePlayRecordingSupported(sf::Out<bool> pOut) NN_NOEXCEPT
{
    // TORIAEZU:
    // - 3.0.0NUP では常時動画撮影機能は無効なので常に false を返す
    // - 4.2.0NUP で有効になるときに true に変える必要あり
    *pOut = true;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetGamePlayRecordingState(int stateValue) NN_NOEXCEPT
{
    auto state = static_cast<GamePlayRecordingState>(stateValue);
    NN_RESULT_THROW_UNLESS( GamePlayRecordingState::Stopped == state ||
                            GamePlayRecordingState::Recording == state,
                            ResultInvalidParameter() );
    NN_RESULT_THROW_UNLESS( !GetApplet()->IsGamePlayRecordingForbiddenByBlackList(), ResultSuccess() );

    return m_pProcessGateway->SetGamePlayRecordingStateImpl(state);
}

Result IntegratedApplet::AppletSelfProxyImpl::RequestFlushGamePlayingMovieForDebug() NN_NOEXCEPT
{
    GetApplet()->GetAppletSystem()->GetWindowManager()->NotifyCaptureButtonLongPressed();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::InitializeGamePlayRecording(sf::NativeHandle transferMemoryHandle, uint64_t transferMemorySize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS( oe::WorkBufferSizeForGamePlayRecording == transferMemorySize, ResultInvalidParameter() );
    NN_RESULT_THROW_UNLESS( !GetApplet()->IsGamePlayRecordingForbiddenByBlackList(), ResultSuccess() );
    return m_pProcessGateway->InitializeGamePlayRecordingImpl(std::move(transferMemoryHandle), transferMemorySize, 0);
}

Result IntegratedApplet::AppletSelfProxyImpl::SetControllerFirmwareUpdateSection(bool isInSection) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetControllerFirmwareUpdateSection(%s)", isInSection ? "true" : "false");
    return m_pProcessGateway->SetControllerFirmwareUpdateSectionImpl(isInSection);
}

Result IntegratedApplet::AppletSelfProxyImpl::SetIdleTimeDetectionExtension(uint32_t extentionType) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetIdleTimeDetectionExtension(type=%u)", extentionType);
    auto* p = GetApplet()->GetIdleHandlingContext();
    auto prev = p->idleTimeDetectionDelayMode;
    switch (extentionType)
    {
        case IdleTimeDetectionExtension_Disabled:
        {
            p->idleTimeDetectionDelayMode = idle::IdleTimeDetectionExtension_Disabled;
            break;
        }
        case IdleTimeDetectionExtension_Enabled:
        {
            p->idleTimeDetectionDelayMode = idle::IdleTimeDetectionExtension_Enabled;
            break;
        }
        case IdleTimeDetectionExtension_EnabledUnsafe:
        {
            p->idleTimeDetectionDelayMode = idle::IdleTimeDetectionExtension_EnabledUnsafe;
            break;
        }
        default: NN_RESULT_THROW( ResultInvalidParameter() );
    }

    if (prev != p->idleTimeDetectionDelayMode)
    {
        m_pProcessGateway->RequestToSetHandlingContextToIdle();
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetIdleTimeDetectionExtension(sf::Out<uint32_t> pOut) NN_NOEXCEPT
{
    auto* p = GetApplet()->GetIdleHandlingContext();
    switch (p->idleTimeDetectionDelayMode)
    {
        case idle::IdleTimeDetectionExtension_Disabled:
        {
            *pOut = IdleTimeDetectionExtension_Disabled;
            break;
        }
        case idle::IdleTimeDetectionExtension_Enabled:
        {
            *pOut = IdleTimeDetectionExtension_Enabled;
            break;
        }
        case idle::IdleTimeDetectionExtension_EnabledUnsafe:
        {
            *pOut = IdleTimeDetectionExtension_EnabledUnsafe;
            break;
        }
        default: NN_RESULT_THROW( ResultInvalidParameter() );
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::GetLibraryAppletLaunchableEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
{
    *pOut = sf::NativeHandle(m_pProcessGateway->GetLibraryAppletLaunchableEventImpl()->GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetInputDetectionSourceSet(Bit32 inputSources) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetInputDetectionSourceSet(input=0x%08x)", inputSources);
    auto* p = GetApplet()->GetIdleHandlingContext();
    static_assert(sizeof(hid::system::InputSourceIdSet) == sizeof(inputSources), "");
    std::memcpy(&p->enabledInputSourceIdSet, &inputSources, sizeof(inputSources));
    m_pProcessGateway->RequestToSetHandlingContextToIdle();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetAutoSleepDisabled(bool isDisabled) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetAutoSleepDisabled(%s)", isDisabled ? "true" : "false");
    auto* p = GetApplet()->GetIdleHandlingContext();
    auto prev = p->isAutoSleepDisabled;
    p->isAutoSleepDisabled = isDisabled;
    if ( prev != p->isAutoSleepDisabled )
    {
        m_pProcessGateway->RequestToSetHandlingContextToIdle();
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::IsAutoSleepDisabled(sf::Out<bool> pOut) NN_NOEXCEPT
{
    auto* p = GetApplet()->GetIdleHandlingContext();
    *pOut = p->isAutoSleepDisabled;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReportUserIsActive() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke idle::ReportUserIsActive()");
    return idle::ReportUserIsActive();
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCurrentIlluminance(sf::Out<float> pOut) NN_NOEXCEPT
{
    bool overflowAlert;
    NN_UNUSED(overflowAlert);
    return GetCurrentIlluminanceEx(pOut, &overflowAlert);
}

Result IntegratedApplet::AppletSelfProxyImpl::GetCurrentIlluminanceEx(sf::Out<float> pOut, sf::Out<bool> pOutOverflowAlert) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "Invoke lbl::GetAmbientLightSensorValue()");
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(lbl_GetAmbientLightSensorValue, 30);
    *pOut = lbl::GetAmbientLightSensorValue(pOutOverflowAlert.GetPointer());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::IsIlluminanceAvailable(sf::Out<bool> pOut) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "Invoke lbl::IsIlluminanceAvailable()");
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(lbl_IsAmbientLightSensorAvailable, 30);
    *pOut = lbl::IsAmbientLightSensorAvailable();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReportMultimediaError(nn::Bit32 resultValue, const nn::sf::InBuffer& multimediaTelemetry) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ReportMultimediaError(resultValue = 0x%08x, multimediaTelemetry)\n", resultValue);
    auto result = am::service::MakeMultimediaErrorReportIfNecessary(resultValue, GetApplet(), multimediaTelemetry.GetPointerUnsafe(), multimediaTelemetry.GetSize());
    if( result.IsFailure() )
    {
        NN_AM_SERVICE_APPLET_LOG(error, GetApplet(), "MakeMultimediaErrorReport() failed (0x%08x)\n", result.GetInnerValueForDebug());
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::TakeScreenShotOfOwnLayer(int index, bool isScreenShotPermitted) NN_NOEXCEPT
{
    bool isImmediate = false;
    return m_pProcessGateway->TakeScreenShotOfOwnLayerImpl(index, isScreenShotPermitted, isImmediate);
}

Result IntegratedApplet::AppletSelfProxyImpl::TakeScreenShotOfOwnLayerEx(int index, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT
{
    return m_pProcessGateway->TakeScreenShotOfOwnLayerImpl(index, isScreenShotPermitted, isImmediate);
}

Result IntegratedApplet::AppletSelfProxyImpl::SetScreenShotAppletIdentityInfo(AppletIdentityInfo info) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetScreenShotAppletIdentityInfo(appletId=0x%08x, applicationId=0x%016llx)", info.appletId, info.applicationId);
    applet::AppletIdentityInfo appletInfo = {};
    appletInfo.appletId      = static_cast<applet::AppletId>(info.appletId);
    appletInfo.applicationId = { info.applicationId };
    return m_pProcessGateway->SetScreenShotAppletIdentityInfoImpl(appletInfo);
}

Result IntegratedApplet::AppletSelfProxyImpl::SetRequiresCaptureButtonShortPressedMessage(bool isRequired) NN_NOEXCEPT
{
    m_pProcessGateway->SetRequiresCaptureButtonShortPressedMessageImpl(isRequired);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::IsVrModeEnabled(sf::Out<bool> pOut) NN_NOEXCEPT
{
    *pOut = GetApplet()->GetAppletSystem()->IsVrModeEnabledImpl();
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetVrModeEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetVrModeEnabled(%s)", isEnabled ? "true" : "false");
    m_pProcessGateway->SetVrModeEnabledImpl(isEnabled);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::SetLcdBacklighOffEnabled(bool isLcdBacklightOff) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetLcdBacklighOffEnabled(%s)", isLcdBacklightOff ? "OFF" : "ON");
    m_pProcessGateway->SetLcdBacklighOffEnabledImpl(isLcdBacklightOff);
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::IsInControllerFirmwareUpdateSection(sf::Out<bool> pOut) NN_NOEXCEPT
{
    *pOut = GetApplet()->GetAppletSystem()->IsInControllerFirmwareUpdateSection();
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::AppletSelfProxyImpl::NotifyControllerFirmwareUpdateSectionChangedImpl() NN_NOEXCEPT
{
    m_pProcessGateway->SetControllerFirmwareUpdateSectionChangedNotification();
}

void IntegratedApplet::AppletSelfProxyImpl::NotifyVrModeCurtainRequiredImpl() NN_NOEXCEPT
{
    m_pProcessGateway->SetVrModeCurtainRequiredNotification();
}

Result IntegratedApplet::AppletSelfProxyImpl::SetWirelessPriorityMode(int mode) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return m_pProcessGateway->SetWirelessPriorityModeImpl(static_cast<applet::WirelessPriorityMode>(mode));
#else
    NN_UNUSED(mode);
    NN_RESULT_THROW( am::ResultNotSupported() );
#endif
}

void IntegratedApplet::ProcessGateway::OnWirelessPriorityModeChanged(WirelessPriorityMode mode) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "%s(%d)", NN_CURRENT_FUNCTION_NAME, mode);
    switch (mode)
    {
        case WirelessPriorityMode_OptimizedForBluetooth:
        case WirelessPriorityMode_OptimizedForWlan:
        {
            BtmSetWirelessPriorityMode(mode);
            break;
        }
        default:
        {
            // Do nothing
            break;
        }
    }
}

Result IntegratedApplet::ProcessGateway::SetScreenShotAppletIdentityInfoImpl(applet::AppletIdentityInfo info) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (info.IsApplication())
    {
        m_ScreenShotProgramId = info.applicationId;
    }
    else
    {
        ncm::SystemProgramId id = ncm::SystemProgramId::GetInvalidId();
        NN_RESULT_DO(GetApplet()->MakeSystemProgramIdFromAppletId(&id, info.appletId));
        m_ScreenShotProgramId = { id.value };
    }
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireLastApplicationCaptureSharedBuffer(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<int> pOut) NN_NOEXCEPT
{
    auto result = m_pProcessGateway->AcquireSharedCaptureBuffer(
        pOutIsScreenShotEnabled.GetPointer(), pOut.GetPointer(), am::service::CaptureBufferIndex_LastApplication
    );
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AcquireCaptureSharedBuffer(index=%d) result=0x%08x",
        am::service::CaptureBufferIndex_LastApplication, result.GetInnerValueForDebug()
    );
    return result;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseLastApplicationCaptureSharedBuffer() NN_NOEXCEPT
{
    auto result = m_pProcessGateway->ReleaseSharedCaptureBuffer(am::service::CaptureBufferIndex_LastApplication);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ReleaseCaptureSharedBuffer(index=%d) result=0x%08x",
        am::service::CaptureBufferIndex_LastApplication,
        result.GetInnerValueForDebug()
    );
    return result;
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireLastForegroundCaptureSharedBuffer(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<int> pOut) NN_NOEXCEPT
{
    auto result = m_pProcessGateway->AcquireSharedCaptureBuffer(
        pOutIsScreenShotEnabled.GetPointer(), pOut.GetPointer(), am::service::CaptureBufferIndex_LastForeground
    );
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AcquireCaptureSharedBuffer(index=%d) result=0x%08x",
        am::service::CaptureBufferIndex_LastForeground, result.GetInnerValueForDebug()
    );
    return result;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseLastForegroundCaptureSharedBuffer() NN_NOEXCEPT
{
    auto result = m_pProcessGateway->ReleaseSharedCaptureBuffer(am::service::CaptureBufferIndex_LastForeground);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ReleaseCaptureSharedBuffer(index=%d) result=0x%08x",
        am::service::CaptureBufferIndex_LastForeground,
        result.GetInnerValueForDebug()
    );
    return result;
}

Result IntegratedApplet::AppletSelfProxyImpl::AcquireCallerAppletCaptureSharedBuffer(sf::Out<bool> pOutIsScreenShotEnabled, sf::Out<int> pOut) NN_NOEXCEPT
{
    auto result = m_pProcessGateway->AcquireSharedCaptureBuffer(
        pOutIsScreenShotEnabled.GetPointer(), pOut.GetPointer(), am::service::CaptureBufferIndex_CallerApplet
    );
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AcquireCaptureSharedBuffer(index=%d) result=0x%08x",
        am::service::CaptureBufferIndex_CallerApplet, result.GetInnerValueForDebug()
    );
    return result;
}

Result IntegratedApplet::AppletSelfProxyImpl::ReleaseCallerAppletCaptureSharedBuffer() NN_NOEXCEPT
{
    auto result = m_pProcessGateway->ReleaseSharedCaptureBuffer(am::service::CaptureBufferIndex_CallerApplet);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ReleaseCaptureSharedBuffer(index=%d) result=0x%08x",
        am::service::CaptureBufferIndex_CallerApplet,
        result.GetInnerValueForDebug()
    );
    return result;
}

Result IntegratedApplet::ProcessGateway::AcquireSharedCaptureBuffer(bool* pOutIsScreenShotPermitted, int* pOutBufferIndex, CaptureBufferIndex index) NN_NOEXCEPT
{
    return am::service::AcquireCaptureBufferIndex(pOutIsScreenShotPermitted, pOutBufferIndex, m_LayerHandle, index);
}

Result IntegratedApplet::ProcessGateway::ReleaseSharedCaptureBuffer(CaptureBufferIndex index) NN_NOEXCEPT
{
    return am::service::ReleaseCaptureBufferIndex(m_LayerHandle, index);
}

Result IntegratedApplet::ProcessGateway::TakeScreenShotOfOwnLayerImpl(int index, bool isScreenShotPermitted, bool isImmediate) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Invoke TryTakeScreenShotOfOwnLayerImpl(index=%d,screenshot=%s,immediate=%s)", index, isScreenShotPermitted ? "y" : "n", isImmediate ? "y" : "n");

    return am::service::TryCaptureCurrentLayerImage(
        static_cast<am::service::CaptureBufferIndex>(index),
        m_LayerHandle,
        isScreenShotPermitted,
        isImmediate
    );
}

void IntegratedApplet::ProcessGateway::SetForeground(WindowFocusState state, bool canUseApplicationCoreAsRequested) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetForeground(%s)", state == WindowFocusState::InFocus ? "InFocus" : (state == WindowFocusState::OutOfFocus ? "OutOfFocus" : "Background"));

    GetApplet()->SendPlayLog( state == WindowFocusState::InFocus ? pdm::AppletEventType::InFocus : (state == WindowFocusState::OutOfFocus ? pdm::AppletEventType::OutOfFocus : pdm::AppletEventType::Background) );
    if( state == WindowFocusState::InFocus )
    {
        ncm::ProgramId id;
        if( GetApplet()->MakeProgramIdFromAppletIdentityInfo(&id, GetApplet()->GetAppletIdentityInfo()).IsSuccess() )
        {
            UpdateFocusedAppletHistory(id);
        }
    }

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Status.windowFocusState = state;
    m_Status.canUseApplicationCoreAsRequested = canUseApplicationCoreAsRequested;

    // LA 起動可能フラグ（終了要求が立っている場合は常に LA 起動可能へ）
    if (state == WindowFocusState::InFocus || m_Status.exitRequested)
    {
        m_LibraryAppletLaunchableEvent.Signal();
    }
    else
    {
        m_LibraryAppletLaunchableEvent.Clear();
    }

    if (GetApplet()->GetForegroundMode() == window::ForegroundMode::All)
    {
        am::service::SetIntegratedDisplayLayerAsConductorWhileForeground(m_LayerHandle, state == WindowFocusState::InFocus);
    }

    UpdateStatus(false);
}

void IntegratedApplet::ProcessGateway::OnHomeButtonShortPressed() NN_NOEXCEPT
{
    switch (m_HandlesHomeButtonShortPressed)
    {
    case HomeButtonShortPressedHandleType_EnqueueMessage:
        {
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnHomeButtonShortPressed");
            AppletMessage message = {};
            message.message[0] = applet::Message_DetectShortPressingHomeButton;
            PushMessage(message);
            break;
        }
    case HomeButtonShortPressedHandleType_NotifyBlockingHomeButton:
        {
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "HomeButtonShortPressed is blocked");
            SendOverlayNotificationOfBlockingHomeButtonShortPressed();
            break;
        }
    case HomeButtonShortPressedHandleType_DoNothing:
        {
            break;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void IntegratedApplet::OnHomeButtonLongPressed() NN_NOEXCEPT
{
    // 試遊台の場合は HOME ボタン長押しを常に無効化
    if ( IsQuestMode() )
    {
        return;
    }
    NN_AM_SERVICE_APPLET_LOG(seq, this, "HomeButtonLongPressed is blocked");
    SendOverlayNotificationOfBlockingHomeButtonLongPressed();
}

void IntegratedApplet::ProcessGateway::OnHomeButtonLongPressed() NN_NOEXCEPT
{
    if (m_HandlesHomeButtonLongPressed)
    {
        GetApplet()->OnHomeButtonLongPressed();
    }
}

void IntegratedApplet::ProcessGateway::OnPowerButtonShortPressed() NN_NOEXCEPT
{
    if (m_HandlesPowerButtonShortPressed)
    {
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnPowerButtonShortPressed");
        AppletMessage message = {};
        message.message[0] = applet::Message_DetectShortPressingPowerButton;
        PushMessage(message);
    }
}

void IntegratedApplet::ProcessGateway::OnPowerButtonMiddlePressed() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnPowerButtonMiddlePressed");
    AppletMessage message = {};
    message.message[0] = applet::Message_DetectMiddlePressingPowerButton;
    PushMessage(message);
}

void IntegratedApplet::ProcessGateway::OnPowerButtonLongPressed() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnPowerButtonLongPressed");
    AppletMessage message = {};
    message.message[0] = applet::Message_DetectLongPressingPowerButton;
    PushMessage(message);
}

void IntegratedApplet::ProcessGateway::OnCaptureButtonShortPressed() NN_NOEXCEPT
{
    if (m_BootMode == pm::BootMode_Maintenance)
    {
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnCaptureButtonShortPressed() but ignored since boot mode is maintenance");
        return;
    }

    // LCD バックライトを強制 ON
    SwitchLcdBacklightOnByForce();

    // スクリーンショット撮影
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    auto& programId = m_ScreenShotProgramId;
    if (GetApplet()->IsScreenShotEnabled())
    {
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnCaptureButtonShortPressed() and shot: progId=0x%016llx orientation=%d", programId, m_AlbumImageOrientation);
        if (m_RequiresCaptureButtonShortPressedMessage)
        {
            lk.unlock();

            // CAPTURE ボタン短押しをメッセージで通知する
            AppletMessage message = {};
            message.message[0] = applet::Message_CaptureButtonPressedShortly;
            PushMessage(message);
            return;
        }
        else
        {
            auto request = display::ScreenShotRequest::GetTakingScreenShotRequest(
                m_LayerHandle,
                { programId.value }, // FIXME
                nn::capsrv::ConvertImageOrientationToScreenShotOrientation(m_AlbumImageOrientation),
                nn::os::GetSystemTick(),
                m_ProxyOption.forOe // アプリの時だけ UserIdList を埋め込む
            );

            return service::RequestScreenShotTask(request);
        }
    }
    else
    {
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnCaptureButtonShortPressed() but refused: progId=0x%016llx", programId);
        auto request = display::ScreenShotRequest::GetNotifyingRefusedRequest(
            m_LayerHandle,
            { programId. value }, // FIXME
            nn::os::GetSystemTick()
        );

        return service::RequestScreenShotTask(request);
    }
}

bool IntegratedApplet::ProcessGateway::CanGamePlayRecordingImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
    if (m_IsGamePlayRecordingDebugMode)
    {
        return true;
    }
    return true
        && m_GamePlayRecordingVisibility
        && m_GamePlayRecordingState == GamePlayRecordingState::Recording
        && GetApplet()->IsScreenShotEnabled()
        && service::IsGrcAvailable()
    ;
}

void IntegratedApplet::ProcessGateway::OnCaptureButtonLongPressed() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnCaptureButtonLongPressed");

    // LCD バックライトを強制 ON
    SwitchLcdBacklightOnByForce();

    // 各アプレットおよび Starter の場合は動画保存禁止を通知する
    if (!m_ProxyOption.forOe)
    {
        service::SendOverlayNotificationOfContinuousRecordingNotPermitted();
        return;
    }

    // ここから下は全てアプリの場合
    // nmeta で Disable なら NotEnabled を OA に通知
    // nmeta で Manual でブラックリスト入りの場合も NotEnabled を OA に通知
    if (!GetApplet()->IsVideoCaptureAllowed() ||
         GetApplet()->IsGamePlayRecordingForbiddenByBlackList() )
    {
        service::SendOverlayNotificationOfContinuousRecordingNotEnabled();
        return;
    }

    // 動画ファイルの書出しを行なう（アプリのみがハンドリング）
    auto pContinuousRecorder = GetContinuousRecorder();
    if (pContinuousRecorder)
    {
        if ((std::lock_guard<decltype(m_Mutex)>(m_Mutex), CanGamePlayRecordingImpl()))
        {
            NN_AM_SERVICE_SCOPED_STUCK_CHECK(grc_StartFlush, 60);
            grc::ContinuousRecordingFlushParameter param = {};
            if (m_ProxyOption.forOe)
            {
                bool isCopyrightValid = service::IsApplicationCopyrightTextureSet();
                param.isCopyrightImageComposited = isCopyrightValid;
                param.imageOrientation = static_cast<uint8_t>(m_AlbumImageOrientation);
            }

            grcsrv::ContinuousRecordingFlushParameter grcsrvParam = {};
            NN_STATIC_ASSERT(sizeof(grcsrvParam) >= sizeof(param));
            std::memcpy(&grcsrvParam, &param, sizeof(param));
            auto result = pContinuousRecorder->StartFlush(grcsrvParam);
            NN_RESULT_IGNORING_BLOCK
            {
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(grc::ResultFlushIsInProcess)
                    {
                        // nop
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        // TODO: StartFlush() 時のエラーハンドリングが必要かも
                        NN_AM_SERVICE_APPLET_LOG(error, GetApplet(), "pContinuousRecorder->StartFlush() result=0x%08x", result.GetInnerValueForDebug());
                    }
                NN_RESULT_END_TRY
                NN_RESULT_SUCCESS;
            };
            return;
        }
        else
        {
            // CanGamePlayRecordingImpl() == false の場合
            service::SendOverlayNotificationOfContinuousRecordingNotPermitted();
            return;
        }
    }
    else
    {
        // 動画撮影許可だが動画撮影準備前の場合
        service::SendOverlayNotificationOfContinuousRecordingNotPermitted();
        return;
    }
}

void IntegratedApplet::OnGamePlayRecordingStateChanged() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_StateMutex)> lk(m_StateMutex);
    auto pSelfProxy = m_pSelfProxy.lock();
    lk.unlock();
    if (pSelfProxy)
    {
        pSelfProxy->OnGamePlayRecordingStateChangedImpl();
    }
}

void IntegratedApplet::AppletSelfProxyImpl::OnGamePlayRecordingStateChangedImpl() NN_NOEXCEPT
{
    m_pProcessGateway->OnGamePlayRecordingStateChanged();
}

void IntegratedApplet::ProcessGateway::OnGamePlayRecordingStateChanged() NN_NOEXCEPT
{
    auto pContinuousRecorder = GetContinuousRecorder();
    if (!pContinuousRecorder)
    {
        return;
    }

    // 動画撮影の一時停止／再開通知の必要性を精査
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    auto nextGamePlayRecording = CanGamePlayRecordingImpl();
    if (m_IsGamePlayRecordingPerforming != nextGamePlayRecording)
    {
        m_IsGamePlayRecordingPerforming = nextGamePlayRecording;
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnGamePlayRecordingStateChanged() : %s", nextGamePlayRecording ? "true" : "false");

        // grc プロセスへの通知を行なう
        if (nextGamePlayRecording)
        {
            service::EnableIntegratedDisplayLayerToRecordingLayerStackForApplication(m_LayerHandle);
            NN_AM_SERVICE_SCOPED_STUCK_CHECK(grc_StartRecording, 60);
            NN_ABORT_UNLESS_RESULT_SUCCESS(pContinuousRecorder->StartRecording());
        }
        else
        {
            NN_AM_SERVICE_SCOPED_STUCK_CHECK(grc_StopRecording, 60);
            NN_ABORT_UNLESS_RESULT_SUCCESS(pContinuousRecorder->StopRecording());
            service::DisableIntegratedDisplayLayerToRecordingLayerStackForApplication(m_LayerHandle);
        }
    }
}

Result IntegratedApplet::ProcessGateway::StartContinuousRecordingFlushForDebug(sf::Out<sf::NativeHandle> pOut, TimeSpan maxTime) NN_NOEXCEPT
{
    auto pContinuousRecorder = GetContinuousRecorder();
    NN_RESULT_THROW_UNLESS(pContinuousRecorder, ResultInvalidCall());
    pContinuousRecorder->CancelFlush();
    grc::ContinuousRecordingFlushParameter parameter = {};
    parameter.isCopyrightImageComposited = false;
    parameter.imageOrientation = static_cast<uint8_t>(m_AlbumImageOrientation);
    parameter.overrides = true;
    parameter.overrideParameter.maxTime = maxTime;
    parameter.overrideParameter.minTime = {};
    parameter.overrideParameter.cutsPrevious = false;
    grcsrv::ContinuousRecordingFlushParameter grcsrvParameter = {};
    static_assert(sizeof(grcsrvParameter) >= sizeof(parameter), "");
    std::memcpy(&grcsrvParameter, &parameter, sizeof(parameter));
    return pContinuousRecorder->StartFlushWithEvent(pOut, grcsrvParameter);
}

Result IntegratedApplet::AppletSelfProxyImpl::StartContinuousRecordingFlushForDebug(sf::Out<sf::NativeHandle> pOut, int64_t maxTimeInNanoSeconds) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(service::IsDevelopmentFunctionEnabled(), ResultDevelopmentFunctionCalled());
    return m_pProcessGateway->StartContinuousRecordingFlushForDebug(pOut, TimeSpan::FromNanoSeconds(maxTimeInNanoSeconds));
}

void IntegratedApplet::ProcessGateway::RequestToAcquireSleepLockImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_SleepLockCount == 1);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "RequestToAcquireSleepLockImpl");
    auto updater = GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).needsApprovalForSleep = true;
}

void IntegratedApplet::ProcessGateway::ReleaseSleepLockImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_SleepLockCount == 0);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ReleaseSleepLockImpl(clock=%llu)", static_cast<unsigned long long>(m_ApprovingClockForSleep));
    auto updater = GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).needsApprovalForSleep = false;
    updater.RefWindowProperty(m_pWindow).approvedClockForSleep = util::Exchange(&m_ApprovingClockForSleep, 0);
    this->m_ReleaseSleepLockRequested = false;
    this->m_ReleaseSleepLockRequestedRecognized = false;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::ReleaseSleepLockTransientlyImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_ApprovingClockForSleep != 0);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ReleaseSleepLockTransientlyImpl(clock=%llu)", static_cast<unsigned long long>(m_ApprovingClockForSleep));
    auto updater = GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).approvedClockForSleep = util::Exchange(&m_ApprovingClockForSleep, 0);
    this->m_ReleaseSleepLockRequested = false;
    this->m_ReleaseSleepLockRequestedRecognized = false;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::RequestApprovalForSleep(uint64_t clock) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "RequestApprovalForSleep(clock=%llu)", static_cast<unsigned long long>(clock));
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_SDK_ASSERT(clock != 0);
    NN_SDK_ASSERT(m_ApprovingClockForSleep == 0);
    this->m_ApprovingClockForSleep = clock;
    this->m_ReleaseSleepLockRequested = true;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnSleepRequiredByHighTemperature() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SleepRequiredByHighTemperature");
    m_SleepRequiredByHighTemperature = true;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnSleepRequiredByLowBattery() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SleepRequiredByLowBattery");
    m_SleepRequiredByByLowBattery = true;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnAutoPowerDown() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AutoPowerDown");
    m_AutoPowerDown = true;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnCecSystemStandbyReceived() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnCecSystemStandbyReceived");
    AppletMessage message = {};
    message.message[0] = applet::Message_DetectReceivingCecSystemStandby;
    PushMessage(message);
}

void IntegratedApplet::ProcessGateway::SetCecStateAndNotifyChanged(bool disablesOneTouchPlayCecNotification) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetCecStateAndNotifyChanged(disablesOneTouchPlayCecNotification=%s)", disablesOneTouchPlayCecNotification ? "true" : "false");
    omm::SetApplicationCecSettingsAndNotifyChanged( disablesOneTouchPlayCecNotification ? omm::TvPowerStateMatchingMode_Disabled : omm::TvPowerStateMatchingMode_Default );
}

void IntegratedApplet::ProcessGateway::OnSleepLockAcquired() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SleepLockAcquired");
    this->m_ApprovingClockForSleep = 0;
    SetAcquiredSleepLockImpl(true);
}

void IntegratedApplet::ProcessGateway::OnSleepLockReleased() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnSleepLockReleased");
    SetAcquiredSleepLockImpl(false);
}

void IntegratedApplet::ProcessGateway::BeforeSleepSystem() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "BeforeSleepSystem()");
    m_Status.wakefulness = Wakefulness::Sleep;
    UpdateStatus(false);
}

void IntegratedApplet::ProcessGateway::AfterSleepSystem() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Status.wakefulness = Wakefulness::Waking;
    UpdateStatus(false);
}

void IntegratedApplet::ProcessGateway::SetCanRun(bool canRun) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Status.canRun = canRun;
    UpdateStatus(false);
}

void IntegratedApplet::ProcessGateway::UpdateStatus(bool forceSignal) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
    if (m_Status.wakefulness == Wakefulness::Waking)
    {
        // wakefulness を Awake と仮定した際に非 Background になるのであれば Awake に移行
        auto awakeStatus = m_Status;
        awakeStatus.wakefulness = Wakefulness::Awake;
        switch (awakeStatus.GetFocusState(GetShimLibraryMode()))
        {
            case FocusState_InFocus:
            case FocusState_OutOfFocus:
            {
                m_Status.wakefulness = Wakefulness::Awake;
                break;
            }
            default: ;
        }
    }
    auto statusUpdated = false;
    auto newIsProcessActive = m_Status.IsProcessActive();
    if (newIsProcessActive != m_IsProcessActive)
    {
        if (newIsProcessActive)
        {
            ResumeImpl();
        }
        else
        {
            SuspendImpl(m_Status.isFatal ? false : true);
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "RestartFromSuspend");
            if (m_NotifiesRestartFromSuspend)
            {
                this->m_RestartFromSuspend = true;
            }
        }
        this->m_IsProcessActive = newIsProcessActive;
        statusUpdated = true;
    }
    if (m_Status.isFatal)
    {
        return;
    }
    auto newFocusState = m_Status.GetFocusState(GetShimLibraryMode());
    if (newFocusState != m_FocusState)
    {
        this->m_FocusStateChanged = true;
        this->m_FocusState = newFocusState;
        statusUpdated = true;
    }
    if (forceSignal || statusUpdated)
    {
        UpdateMessageSignaled();
    }
}

void IntegratedApplet::ProcessGateway::SuspendImpl(bool alsoAudio) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SuspendImpl(%d)", (int)alsoAudio);
    if (alsoAudio)
    {
        SuspendAudioInProcess(m_AppletResourceUserId);
    }
    auto result = SuspendWindowImplUnsafe( this->GetProcessHandle() );
    if (!result.IsSuccess())
    {
        NN_AM_SERVICE_APPLET_LOG(error, GetApplet(), "SuspendWindowImplUnsafe() result=0x%08x", result.GetInnerValueForDebug());
    }
}

void IntegratedApplet::ProcessGateway::ResumeImpl() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "ResumeImpl()");
    auto result = ResumeWindowImplUnsafe( this->GetProcessHandle() );
    if (!result.IsSuccess())
    {
        NN_AM_SERVICE_APPLET_LOG(error, GetApplet(), "ResumeWindowImplUnsafe() result=0x%08x", result.GetInnerValueForDebug());
    }
    ResumeAudioInProcess(m_AppletResourceUserId);
}

void IntegratedApplet::ProcessGateway::OnFatal() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Status.isFatal = true;
    UpdateStatus(false);
}


void IntegratedApplet::ProcessGateway::OnProcessException(const os::ProcessId& processId) NN_NOEXCEPT
{
    if (!service::IsDevelopmentFunctionEnabled())
    {
        return;
    }
    if (processId == GetApplet()->GetCurrentProcessId() && std::exchange(m_SavesMovieOnException, false))
    {
        sf::NativeHandle h;
        this->StartContinuousRecordingFlushForDebug(&h, 0);
    }
}

void IntegratedApplet::ProcessGateway::SetHandlingContextToIdle() NN_NOEXCEPT
{
    auto context = *(GetApplet()->GetIdleHandlingContext());
    context.operationMode = GetApplet()->GetAppletSystem()->GetOperationMode();
    context.aruid = GetAppletResourceUserId();
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetHandlingContextToIdle(%lld, %d,%d,%d,%d, %d,%s, 0x%08x, %s)",
        context.timeMediaPlaybackFinished.GetInt64Value(),
        context.overrideAutoSleepTimeInHandheldInSeconds,
        context.overrideAutoSleepTimeInConsoleInSeconds,
        context.overrideDimmingTimeInHandheldInSeconds,
        context.overrideDimmingTimeInConsoleInSeconds,
        context.idleTimeDetectionDelayMode,
        context.isInMediaPlayback ? "true" : "false",
        *reinterpret_cast<Bit32*>(&context.enabledInputSourceIdSet),
        context.operationMode == omm::OperationMode::Handheld ? "Handheld" : "Console");
    NN_ABORT_UNLESS_RESULT_SUCCESS( idle::SetHandlingContext(context) );
}

void IntegratedApplet::ProcessGateway::RequestToSetHandlingContextToIdle() NN_NOEXCEPT
{
    static uint64_t idleHandlingContextUpdateCount = 0;
    auto updater = GetWindowManager()->BeginUpdate();
    ++idleHandlingContextUpdateCount;
    updater.RefWindowProperty(m_pWindow).updateCount = idleHandlingContextUpdateCount;
}

void IntegratedApplet::ProcessGateway::NotifyForegroundToNfc() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyForegroundToNfc(ARUID=0x%016llx)", GetAppletResourceUserId());
    nfc::am::NotifyForegroundApplet(GetAppletResourceUserId());
}

void IntegratedApplet::ProcessGateway::NotifyHidInputRequirement(bool needInput) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyHidInputRequirement(ARUID=%lld, %s)", GetAppletResourceUserId(), needInput ? "true" : "false");
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(hid_EnableAppletToGetInput, 60);
    hid::system::EnableAppletToGetInput(GetAppletResourceUserId(), needInput);
}

void IntegratedApplet::ProcessGateway::NotifyHidNpadPolicyOwner() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyHidNpadPolicyOwner(ARUID=%lld)", GetAppletResourceUserId());
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(hid_SetAruid, 60);
    hid::system::SetAppletResourceUserId(GetAppletResourceUserId());
}

void IntegratedApplet::ProcessGateway::NotifyIrSensorInputRequirement(bool needInput) NN_NOEXCEPT
{
#if 1
    // ローンチ時点では下記の呼出しは不要とのこと。
    // 将来的に partial-FG などが必要とする可能性がある。
    NN_UNUSED(needInput);
#else
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyIrSensorInputRequirement(ARUID=%lld, %s)", GetAppletResourceUserId(), needInput ? "true" : "false");
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(irsensor_EnableAppletToGetInput, 60);
    irsensor::system::EnableAppletToGetInput(GetAppletResourceUserId(), needInput);
#endif
}

void IntegratedApplet::ProcessGateway::NotifyIrSensorPolicyOwner() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyIrSensorPolicyOwner(ARUID=%lld)", GetAppletResourceUserId());
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(irsensor_SetAruid, 60);
    irsensor::system::SetAppletResourceUserId(GetAppletResourceUserId());
}

void IntegratedApplet::ProcessGateway::NotifyHidBusPolicyOwner() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyHidBusPolicyOwner(ARUID=%lld)", GetAppletResourceUserId());
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(hidbus_SetAruid, 60);
    hidbus::system::SetAppletResourceUserId(GetAppletResourceUserId());
}

void IntegratedApplet::ProcessGateway::NotifyVibrationOwner(bool isOwner) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyVibrationOwner(ARUID=%lld, %s)", GetAppletResourceUserId(), isOwner ? "true" : "false");
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(hid_SetAruidValidForVibration, 60);
    hid::system::SetAruidValidForVibration(GetAppletResourceUserId(), isOwner);
}

void IntegratedApplet::ProcessGateway::NotifySixAxisSensorInputRequirement(bool needInput) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifySixAxisSensorInputRequirement(ARUID=%lld, %s)", GetAppletResourceUserId(), needInput ? "true" : "false");
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(hid_EnableAppletToGetSixAxisSensor, 60);
    hid::system::EnableAppletToGetSixAxisSensor(GetAppletResourceUserId(), needInput);
}

void IntegratedApplet::ProcessGateway::NotifyBleConnectionOwner() NN_NOEXCEPT
{
    auto aruid = GetAppletResourceUserId();
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "NotifyBleConnectionOwner(ARUID=%lld)", aruid);
    BtmSetAppletResourceUserId(aruid);
}

void IntegratedApplet::ProcessGateway::OnOperationModeChanged() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnOperationModeChanged");
    SwitchLcdBacklightOnByForce();
    if (auto pContinuousRecorder = GetContinuousRecorder())
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(grc_ResetFlushTime, 60);
        auto additionalTime = service::GetDefaultContinuousRecordingAdditionalTimeOnDockInOut();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pContinuousRecorder->ResetFlushTime(additionalTime.GetNanoSeconds()));
    }
    if (!m_NotifiesOperationModeChanged)
    {
        return;
    }
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_OperationModeChanged = true;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnPerformanceModeChanged() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnPerformanceModeChanged");
    if (!m_NotifiesPerformanceModeChanged)
    {
        return;
    }
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_PerformanceModeChanged = true;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnVrModeChanged(bool isVrMode, bool fromOe) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnVrModeChanged: VR-mode=%s", isVrMode ? "true" : "false");
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_ProxyOption.forOe)
    {
        // アプリ以外の各アプレットへは常に変更を通知
        m_VrModeChanged = true;
    }
    else if (!fromOe)
    {
        // アプリに対しては、アプレットからの VR モード解除のみ通知
        if (!isVrMode)
        {
            m_VrModeDisabledBySystem = true;
        }
    }
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::OnLcdBacklightChanged(bool isLcdBacklightOff) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnLcdBacklightChanged(backlight: %s)", isLcdBacklightOff ? "off" : "on");

    const TimeSpan fadeTime = TimeSpan::FromMilliSeconds(100);

    if (isLcdBacklightOff)
    {
        // LCD バックライト消灯
        NN_AM_SERVICE_STUCK_CHECKED(lbl_SwitchBacklightOff, 30, lbl::SwitchBacklightOff(fadeTime));
    }
    else
    {
        // LCD バックライト点灯
        NN_AM_SERVICE_STUCK_CHECKED(lbl_SwitchBacklightOn, 30, lbl::SwitchBacklightOn(fadeTime));
    }
}

void IntegratedApplet::ProcessGateway::SwitchLcdBacklightOnByForce() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SwitchLcdBacklightOnByForce");

    // LCD バックライトを強制的に ON に戻す
    if ( this->SetLcdBacklighOffEnabledImpl(false) )
    {
        // 前の値が true ならメッセージも通知
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        m_LcdBackLightOnNotification = true;
        UpdateMessageSignaled();
    }
}

void IntegratedApplet::ProcessGateway::OnSdCardRemoved() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnSdCardRemoved");
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_SdCardRemoved = true;
    UpdateMessageSignaled();
}

Result IntegratedApplet::ProcessGateway::SetControllerFirmwareUpdateSectionImpl(bool isInSection) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    auto pAppletSystem = GetApplet()->GetAppletSystem();

    // true->true や false->false の重ね掛けは認めない。
    // また、true->false に戻せるのは false->true を行ったオーナーに限定する。
    NN_RESULT_THROW_UNLESS( m_ControllerFirmwareUpdateSectionOwner != isInSection, ResultUnbalancedControllerFirmwareUpdateSectionControl() );
    NN_RESULT_DO( pAppletSystem->SetControllerFirmwareUpdateSection(isInSection) );
    this->m_ControllerFirmwareUpdateSectionOwner = isInSection;

    pAppletSystem->NotifyControllerFirmwareUpdateSectionChanged();
    NN_RESULT_SUCCESS;
}

void IntegratedApplet::ProcessGateway::SetDisplayVisibility(bool isVisible, int32_t zOrder, bool staysDisplay) NN_NOEXCEPT
{
    if (!isVisible && staysDisplay)
    {
        auto p = GetApplet();
        p->OnLostForeground();
    }
    else if (isVisible)
    {
        GetApplet()->OnGetForeground();
    }

    am::service::SetIntegratedDisplayLayerTransitionLayerRequested(m_LayerHandle, staysDisplay);
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "SetDisplayVisibility(%s, z = %d)", isVisible ? "true" : "false", static_cast<int>(zOrder));
    this->SetDisplayLayerVisibilityImpl(isVisible, zOrder);
}

void IntegratedApplet::ProcessGateway::ApproveToDisplayImpl(uint64_t clock) NN_NOEXCEPT
{
    auto updater = GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).Display(clock);
}

void IntegratedApplet::ProcessGateway::RequestToDisplay(uint64_t clock) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_LayerMutex)> lk(m_LayerMutex);
    if (m_HandlesRequestToDisplay)
    {
        lk.unlock();
        OnRequestToDisplay(clock);
    }
    else
    {
        ApproveToDisplayImpl(clock);
    }
}

void IntegratedApplet::ProcessGateway::OnGameCardStateChanged() NN_NOEXCEPT
{
    GetApplet()->OnGameCardStateChanged();
}

void IntegratedApplet::ProcessGateway::OnRequestToDisplay(uint64_t clock) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "OnRequestToDisplay(%llu)", clock);
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    this->m_ClockForRequestToDisplay = clock;
    UpdateMessageSignaled();
}

void IntegratedApplet::ProcessGateway::InvalidateDisplayLayer() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_LayerMutex)> lk(m_LayerMutex);
    if (m_LayerHandle >= 0)
    {
        {
            auto updater = GetWindowManager()->BeginUpdate();
            updater.RefWindowProperty(m_pWindow).displayIsValid = false;
        }
        // TODO: 調整
        //   除外されたレイヤの除外を確定と、遷移レイヤを確定するためのスリープ
        //   その他の倍の時間をとっている
        os::SleepThread(TimeSpan::FromMilliSeconds(1000 / 60 * 10));
    }
}

Result IntegratedApplet::MoveWindowToTop() NN_NOEXCEPT
{
    auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
    updater.MoveToTop(m_pWindowGroup.get());
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::MoveWindowToScreenLock() NN_NOEXCEPT
{
    auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).globalOrder = window::WindowGlobalOrder::ScreenLock;
    NN_RESULT_SUCCESS;
}

Result IntegratedApplet::MoveWindowToNormal() NN_NOEXCEPT
{
    auto updater = GetAppletSystem()->GetWindowManager()->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).globalOrder = window::WindowGlobalOrder::Normal;
    NN_RESULT_SUCCESS;
}

}}}
